package notifications

import (
	"fmt"
	"sync"
	"sync/atomic"

	"code.justin.tv/cb/oracle/internal/clients/db"
	"code.justin.tv/cb/oracle/internal/clients/discovery"
	"code.justin.tv/cb/oracle/internal/clients/sns"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/cb/oracle/internal/clients"
	"code.justin.tv/cb/oracle/internal/notifications/jobs"
	"code.justin.tv/cb/oracle/internal/workerpool"
	"code.justin.tv/cb/oracle/mocks"

	"testing"

	"net/http"
	"net/http/httptest"
)

func mockStats() *mocks.MockStatsClient {
	mockStats := &mocks.MockStatsClient{}
	mockStats.On("Inc", mock.Anything, mock.Anything, mock.Anything).Return(nil)
	return mockStats
}

func TestAddEventsToDynamoJobsQueue(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		payload := `{"name": "test game name"}`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	mockDiscoveryClient := discovery.NewHTTPClient(server.URL)

	dynamoJobsQueue := make(chan workerpool.Job)
	snsJobsQueue := make(chan workerpool.Job)
	events := testEventsFetch()

	n := &NotificationHandler{
		Clients: &clients.Clients{
			Discovery: mockDiscoveryClient,
		},
		Stats: mockStats(),
	}
	cacheKey := "key"

	ty := sns.NotificationType{Email: true}

	go n.addEventsToDynamoJobsQueue(events, ty, dynamoJobsQueue, snsJobsQueue, cacheKey)

	for _, e := range events {
		j := <-dynamoJobsQueue
		p, ok := j.(*jobs.GetSubscribersFromDynamo)
		a.True(ok)
		a.Equal(p.Event, e)
		a.Equal(p.CacheKey, cacheKey)
		a.Equal(p.NotificationType.Email, true)
	}
}

func TestProcessDynamoJobsQueue(t *testing.T) {
	a := assert.New(t)
	dynamoJobsQueue := make(chan workerpool.Job, 3)
	snsJobsQueue := make(chan workerpool.Job, 3)

	n := &NotificationHandler{
		Stats: mockStats(),
	}

	for i := 0; i < 3; i++ {
		dynamoJobsQueue <- &mockedDynamoJob{
			val:       i,
			processed: snsJobsQueue,
		}
	}
	close(dynamoJobsQueue)

	err := n.processDynamoJobsQueue(dynamoJobsQueue, snsJobsQueue)
	a.NoError(err)

	i := 0
	for j := range snsJobsQueue {
		s, ok := j.(*mockedSNSJob)
		a.True(ok)
		a.Equal(s.squared, s.val*s.val)
		i++
	}
	a.Equal(i, 3)
}

func TestProcessSNSJobsQueue(t *testing.T) {
	a := assert.New(t)
	snsJobsQueue := make(chan workerpool.Job, 3)

	n := &NotificationHandler{
		Stats: mockStats(),
	}
	var c int32 = 0
	wg := sync.WaitGroup{}
	wg.Add(3)
	for i := 0; i < 3; i++ {
		snsJobsQueue <- &mockedSNSJob{
			val:     i,
			counter: &c,
			wg:      &wg,
		}
	}
	close(snsJobsQueue)

	err := n.processSNSJobsQueue(snsJobsQueue, "")
	a.NoError(err)

	wg.Wait()
	a.Equal(3, int(c))
}

func testEventsFetch() []*db.Event {
	events := []*db.Event{}
	for i := 0; i < 3; i++ {
		d := fmt.Sprintf("description for %d", i)
		events = append(events, &db.Event{
			ID:          i,
			Description: &d,
			GameID:      123,
		})
	}

	return events
}

type mockedDynamoJob struct {
	val       int
	processed chan<- workerpool.Job
}

type mockedSNSJob struct {
	val     int
	squared int
	wg      *sync.WaitGroup
	counter *int32
}

func (m *mockedDynamoJob) Process() error {
	m.processed <- &mockedSNSJob{
		val:     m.val,
		squared: m.val * m.val,
	}
	return nil
}

func (m *mockedSNSJob) Process() error {
	defer m.wg.Done()
	atomic.AddInt32(m.counter, 1)
	return nil
}
