package workerpool

import (
	"fmt"
	"sync"

	"github.com/cactus/go-statsd-client/statsd"

	log "github.com/Sirupsen/logrus"
)

type Job interface {
	Process() error
}

// workerPool contains its own job queue channel which it processes jobs from.
// Job must implement a `Process()` method.
type WorkerPool struct {
	JobQueue   <-chan Job
	NumWorkers int
	Stats      statsd.Statter
	StatName   string
}

// Run spins up the number of workers specified in this workerpool to process jobs from the specified channel.
func (w *WorkerPool) Run() (*sync.WaitGroup, error) {
	wg := sync.WaitGroup{}
	wg.Add(w.NumWorkers)

	for i := 0; i < w.NumWorkers; i++ {
		go func() {
			defer wg.Done()

			for job := range w.JobQueue {
				err := job.Process()
				if err != nil {
					log.WithError(err).Error("workerpool: job error")
					err := w.Stats.Inc(fmt.Sprintf("%v.fail", w.StatName), 1, 1)
					if err != nil {
						log.WithError(err).Error(fmt.Sprintf("Failed to increment stat for %s", w.StatName))
					}
				} else {
					err := w.Stats.Inc(fmt.Sprintf("%v.success", w.StatName), 1, 0.1)
					if err != nil {
						log.WithError(err).Error(fmt.Sprintf("Failed to increment stat for %s", w.StatName))
					}
				}
			}
		}()
	}

	return &wg, nil
}
