package workerpool_test

import (
	"time"

	"code.justin.tv/cb/oracle/internal/workerpool"
	"code.justin.tv/cb/oracle/mocks"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"

	"testing"
)

const sleepTime = 10 * time.Millisecond

func TestRunWorker(t *testing.T) {
	a := assert.New(t)
	jobQueue := make(chan workerpool.Job, 3)
	processed := make(chan squared)

	numJobs := 3

	for i := 0; i < numJobs; i++ {
		jobQueue <- &squareJob{
			value:          i,
			processedQueue: processed,
		}
	}
	close(jobQueue)

	mockStats := &mocks.MockStatsClient{}
	mockStats.On("Inc", mock.Anything, mock.Anything, mock.Anything).Return(nil)

	wp := &workerpool.WorkerPool{
		JobQueue:   jobQueue,
		NumWorkers: 2,
		Stats:      mockStats,
	}

	tStart := time.Now()
	wg, err := wp.Run()
	a.NotNil(wg)
	a.NoError(err)

	go func() {
		wg.Wait()
		tEnd := time.Now()
		// time to finish jobs should be roughly numJubs * sleepTime / numWorkers
		a.WithinDuration(tStart.Add(30*time.Millisecond), tEnd, sleepTime)
	}()

	for i := 0; i < numJobs; i++ {
		v := <-processed
		a.Equal(v.squared, v.value*v.value)
	}

}

type squareJob struct {
	value          int
	processedQueue chan squared
}

type squared struct {
	value   int
	squared int
}

func (j *squareJob) Process() error {
	time.Sleep(sleepTime)

	j.processedQueue <- squared{
		value:   j.value,
		squared: j.value * j.value,
	}

	return nil
}
