package main

import (
	"net/http"
	"strconv"
	"time"

	"code.justin.tv/cb/oracle/config"
	"code.justin.tv/cb/oracle/internal/api"
	"code.justin.tv/cb/oracle/internal/auth"
	"code.justin.tv/cb/oracle/internal/clients"
	"code.justin.tv/cb/oracle/internal/clients/stats"
	"code.justin.tv/cb/oracle/internal/httpserver"
	"code.justin.tv/common/gometrics"
	log "github.com/Sirupsen/logrus"
	_ "github.com/lib/pq"
	"goji.io/pat"
)

func init() {
	config.SetupRollbarLogging()
	config.Load()

	err := auth.InitializeDecoder()
	if err != nil {
		log.WithError(err).Panic("Failed set up Cartman decoder!")
	}
}

func main() {
	log.Info("Web application starting in ", config.Environment)

	// Instantiate client interfaces:
	clientsObj, err := clients.NewClients()
	if err != nil {
		log.WithError(err).Panic("Failed to instantiate clients on application start-up")
	}

	// statsd error logging
	stats.SetupLogging(clientsObj.Stats)

	// Gometrics
	gometrics.Monitor(clientsObj.Stats, time.Second*5)

	server := httpserver.NewServer(clientsObj.Stats)
	server.StatHandle(pat.Get("/health"), healthCheck, "health_check")

	// Start app
	app := api.NewApiServer(clientsObj)

	// handle is namespacing pattern /* to app,
	// everything in app mux starts with namespace
	// e.g. (root.handle)/v1/* -> (app.HandleFunc)/events -> /v1/events
	server.Handle(pat.New("/*"), app)

	// Application endpoints are handled below:
	app.StatHandleFunc(pat.Get("/v1/manager_events"), app.V1ListAvailableManagerEvents, "v1.list_available_manager_events")
	// Event Extensions routes
	app.StatHandleFunc(pat.Get("/v1/manager_events/:event_id/extensions"), app.V1ListEventExtensions, "v1.read_event_extensions")
	app.StatHandleFunc(pat.Post("/v1/manager_events/:event_id/extensions"), app.V1CreateEventExtension, "v1.create_event_extension")
	app.StatHandleFunc(pat.Put("/v1/manager_events/:event_id/extensions/:key"), app.V1UpdateEventExtension, "v1.update_event_extensions")
	app.StatHandleFunc(pat.Delete("/v1/manager_events/:event_id/extensions/:key"), app.V1DeleteEventExtension, "v1.delete_event_extensions")

	// Public Events API
	app.StatHandleFunc(pat.Get("/v1/events"), app.V1ListAvailableEvents, "v1.list_available_events")
	app.StatHandleFunc(pat.Post("/v1/events"), app.V1CreateEvent, "v1.create_event")
	app.StatHandleFunc(pat.Get("/v1/events/:event_id"), app.V1ReadEvent, "v1.read_event")
	app.StatHandleFunc(pat.Put("/v1/events/:event_id"), app.V1UpdateEvent, "v1.update_event")
	app.StatHandleFunc(pat.Delete("/v1/events/:event_id"), app.V1CreatorDeleteEvent, "v1.creator_delete_event")
	app.StatHandleFunc(pat.Get("/v1/suggestions/game/:game_id"), app.V1SuggestEventsForGame, "v1.suggestions_game_event")

	// Admin Events API
	app.StatHandleFunc(pat.Delete("/v1/admin/events/:event_id"), app.V1AdminDeleteEvent, "v1.admin_delete_event")
	app.StatHandleFunc(pat.Post("/v1/admin/channels/:channel_id/event_cover_images"), app.V1AdminUploadChannelEventCoverImage, "v1.admin_upload_cover_image")
	app.StatHandleFunc(pat.Post("/v1/admin/events"), app.V1AdminCreateEvent, "v1.admin_create_event")
	app.StatHandleFunc(pat.Get("/v1/admin/featured_channels"), app.V1ListFeaturedChannels, "v1.list_featured_channels")
	app.StatHandleFunc(pat.Post("/v1/admin/channels/:channel_id/feature"), app.V1FeatureChannel, "v1.create_featured_channel")
	app.StatHandleFunc(pat.Delete("/v1/admin/channels/:channel_id/feature"), app.V1UnfeatureChannel, "v1.delete_featured_channel")

	app.StatHandleFunc(pat.Post("/v1/channels/:channel_id/event_cover_images"), app.V1UploadChannelEventCoverImage, "v1.upload_cover_image")

	app.StatHandleFunc(pat.Get("/v1/users/:user_id/notifications/events/:event_id/settings"), app.V1ReadUserEventNotificationSettings, "v1.read_subscriptions")
	app.StatHandleFunc(pat.Get("/v1/users/:user_id/notifications/events/settings"), app.V1ReadUserEventsNotificationSettings, "v1.read_subscriptions_for_events")
	app.StatHandleFunc(pat.Put("/v1/users/:user_id/notifications/events/:event_id/settings"), app.V1UpdateUserEventNotificationSettings, "v1.create_subscription")

	log.Info("Listening on port 8000")
	log.Fatal(http.ListenAndServe(":8000", server))
}

// healthCheck is for internal monitoring of the server's health.
func healthCheck(w http.ResponseWriter, r *http.Request) {
	text := "Health check OK"

	w.Header().Add("Cache-Control", "no-cache, no-store, must-revalidate")
	w.Header().Add("Content-Length", strconv.Itoa(len(text)))
	w.Header().Add("Content-Type", "text/plain; charset=utf-8")

	w.WriteHeader(http.StatusOK)

	_, err := w.Write([]byte(text))
	if err != nil {
		log.WithError(err).Error("[HEALTH CHECK] Failed to respond to automatic health check")
	}
}
