package xray

import (
	"context"
	"crypto/rand"
	"fmt"
	"os"
	"sync"
	"time"

	"code.justin.tv/foundation/xray/internal/plugins"
)

type segment struct {
	sync.Mutex
	parent           *segment
	openSegments     int
	sampled          bool
	requestWasTraced bool // Used by xray.RequestWasTraced

	// Required
	TraceID   string  `json:"trace_id,omitempty"`
	ID        string  `json:"id"`
	Name      string  `json:"name"`
	StartTime float64 `json:"start_time"`
	EndTime   float64 `json:"end_time,omitempty"`

	// Optional
	InProgress bool   `json:"in_progress,omitempty"`
	ParentID   string `json:"parent_id,omitempty"`
	Fault      bool   `json:"fault,omitempty"`
	Error      bool   `json:"error,omitempty"`
	Throttle   bool   `json:"throttle,omitempty"`
	Cause      struct {
		WorkingDirectory string      `json:"working_directory,omitempty"`
		Paths            []string    `json:"paths,omitempty"`
		Exceptions       []exception `json:"exceptions,omitempty"`
	} `json:"cause,omitempty"`
	ResourceARN string `json:"resource_arn,omitempty"`
	Origin      string `json:"origin,omitempty"`

	Type         string   `json:"type,omitempty"`
	Namespace    string   `json:"namespace,omitempty"`
	User         string   `json:"user,omitempty"`
	PrecursorIDs []string `json:"precursor_ids,omitempty"`

	// HTTP
	HTTP struct {
		Request struct {
			Method        string `json:"method,omitempty"`
			URL           string `json:"url,omitempty"` // http(s)://host/path
			ClientIP      string `json:"client_ip,omitempty"`
			UserAgent     string `json:"user_agent,omitempty"`
			XForwardedFor bool   `json:"x_forwarded_for,omitempty"`
			Traced        bool   `json:"traced,omitempty"`
		} `json:"request,omitempty"`
		Response struct {
			Status        int `json:"status,omitempty"`
			ContentLength int `json:"content_length,omitempty"`
		} `json:"response,omitempty"`
	} `json:"http,omitempty"`

	AWS struct {
		AccountID string `json:"account_id,omitempty"`
		EC2       struct {
			InstanceID       string `json:"instance_id,omitempty"`
			AvailabilityZone string `json:"availability_zone,omitempty"`
		} `json:"ec2,omitempty"`
		ElasticBeanstalk struct {
			Environment  string `json:"environment,omitempty"`
			VersionLabel string `json:"version_label,omitempty"`
			DeploymentID int    `json:"deployment_id,omitempty"`
		} `json:"elastic_beanstalk,omitempty"`
		ECS struct {
			Container string `json:"container,omitempty"`
		} `json:"ecs,omitempty"`
		Tracing struct {
			SDK string `json:"sdk,omitempty"`
		} `json:"tracing,omitempty"`
		Operation string `json:"operation,omitempty"`
		Region    string `json:"region,omitempty"`
		RequestID string `json:"request_id,omitempty"`
		Retries   int    `json:"retries,omitempty"`
		QueueURL  string `json:"queue_url,omitempty"`
		TableName string `json:"table_name,omitempty"`
	} `json:"aws,omitempty"`

	Service struct {
		Version string `json:"version,omitempty"`
	} `json:"service,omitempty"`

	// SQL
	SQL struct {
		ConnectionString string `json:"connection_string,omitempty"`
		URL              string `json:"url,omitempty"` // host:port/database
		DatabaseType     string `json:"database_type,omitempty"`
		DatabaseVersion  string `json:"database_version,omitempty"`
		DriverVersion    string `json:"driver_version,omitempty"`
		User             string `json:"user,omitempty"`
		Preparation      string `json:"preparation,omitempty"` // "statement" / "call"
		SanitizedQuery   string `json:"sanitized_query,omitempty"`
	} `json:"sql,omitempty"`

	// Metadata
	Annotations map[string]interface{}            `json:"annotations,omitempty"`
	Metadata    map[string]map[string]interface{} `json:"metadata,omitempty"`

	// Children
	Subsegments []*segment `json:"subsegments,omitempty"`
}

func (seg *segment) addAnnotation(key string, value interface{}) error {
	switch value.(type) {
	case bool, int, uint, float32, float64, string:
	default:
		return fmt.Errorf("Failed to add annotation key: %q value: %q to subsegment %q. Value must be of type string, number or boolean.", key, value, seg.Name)
	}

	seg.Lock()
	defer seg.Unlock()

	if seg.Annotations == nil {
		seg.Annotations = map[string]interface{}{}
	}
	seg.Annotations[key] = value
	return nil
}

func (seg *segment) addMetadata(key string, value interface{}) error {
	seg.Lock()
	defer seg.Unlock()

	if seg.Metadata == nil {
		seg.Metadata = map[string]map[string]interface{}{}
	}
	if seg.Metadata["default"] == nil {
		seg.Metadata["default"] = map[string]interface{}{}
	}
	seg.Metadata["default"][key] = value
	return nil
}

func (seg *segment) root() *segment {
	if seg.parent == nil {
		return seg
	}
	return seg.parent.root()
}

func newTraceID() string {
	var r [12]byte
	rand.Read(r[:])
	return fmt.Sprintf("1-%08x-%02x", time.Now().Unix(), r)
}

func newSegmentID() string {
	var r [8]byte
	rand.Read(r[:])
	return fmt.Sprintf("%02x", r)
}

func newSegment(ctx context.Context, name string) (context.Context, *segment) {
	configName := config.Name()
	if configName != "" {
		name = configName
	}
	if name == "" {
		name = config.DefaultName()
	}
	if len(name) > 200 {
		name = name[:200]
	}

	parent := getSegment(ctx)
	seg := &segment{parent: parent}
	seg.Lock()
	defer seg.Unlock()

	if parent == nil {
		seg.TraceID = newTraceID()
		seg.addPlugin(plugins.InstancePluginMetadata)
		seg.Service.Version = config.ServiceVersion()
	} else {
		parent.Lock()
		parent.Subsegments = append(parent.Subsegments, seg)
		parent.openSegments++
		parent.Unlock()
	}

	seg.ID = newSegmentID()
	seg.Name = name
	seg.StartTime = float64(time.Now().UnixNano()) / float64(time.Second)
	seg.InProgress = true

	return context.WithValue(ctx, contextkey, seg), seg
}

func (seg *segment) close(err error) {
	seg.Lock()

	seg.EndTime = float64(time.Now().UnixNano()) / float64(time.Second)
	seg.InProgress = false

	if err != nil {
		seg.Fault = true
		seg.Cause.WorkingDirectory, _ = os.Getwd()
		seg.Cause.Exceptions = append(seg.Cause.Exceptions, exceptionFromError(err))
	}

	seg.Unlock()
	seg.flush(false)
}

func (seg *segment) flush(decrement bool) {
	seg.Lock()
	if decrement {
		seg.openSegments--
	}
	shouldFlush := seg.openSegments == 0 && seg.EndTime > 0
	seg.Unlock()

	if shouldFlush {
		if seg.parent == nil {
			emit(seg)
		} else {
			seg.parent.flush(true)
		}
	}
}

func (seg *segment) addPlugin(metadata *plugins.PluginMetadata) {
	//Only called within a seg locked code block
	if metadata == nil {
		return
	}

	if metadata.IdentityDocument != nil {
		seg.AWS.AccountID = metadata.IdentityDocument.AccountID
		seg.AWS.EC2.InstanceID = metadata.IdentityDocument.InstanceID
		seg.AWS.EC2.AvailabilityZone = metadata.IdentityDocument.AvailabilityZone
	}

	if metadata.ECSContainerName != "" {
		seg.AWS.ECS.Container = metadata.ECSContainerName
	}

	if metadata.BeanstalkMetadata != nil {
		seg.AWS.ElasticBeanstalk.Environment = metadata.BeanstalkMetadata.Environment
		seg.AWS.ElasticBeanstalk.VersionLabel = metadata.BeanstalkMetadata.VersionLabel
		seg.AWS.ElasticBeanstalk.DeploymentID = metadata.BeanstalkMetadata.DeploymentID
	}
}
