package view

import "time"

// EventStatusAvailable is the expected status of an Event object
// when it is available for public access.
const EventStatusAvailable = "available"

// V1EventView main view in Event.
type V1EventView struct {
	ID           int        `json:"id"`
	ChannelID    int        `json:"channel_id"`
	StartTimeUTC time.Time  `json:"start_time_utc"`
	EndTimeUTC   time.Time  `json:"end_time_utc"`
	TimeZoneID   string     `json:"time_zone_id"`
	Title        string     `json:"title"`
	Description  *string    `json:"description"`
	GameID       int        `json:"game_id"`
	CreatedAtUTC time.Time  `json:"created_at_utc"`
	UpdatedAtUTC *time.Time `json:"updated_at_utc"`

	CoverImageID          *string `json:"cover_image_id"`
	CoverImageSourceURL   *string `json:"cover_image_source_url"`
	CoverImageURLTemplate *string `json:"cover_image_url_template"`

	Extensions []*V1EventExtensionView `json:"extensions"`
}

// DeleteV1EventOutput contains the fields from the response of making
// a DELETE request to the Oracle service's v1 API.
type DeleteV1EventOutput struct {
	Status  int    `json:"status"`
	Message string `json:"message"`
}

// GetV1EventOutput contains the fields from the response of making a
// GET request via GetV1Event.
type GetV1EventOutput struct {
	Status  int                  `json:"status"`
	Message string               `json:"message"`
	Meta    GetV1EventOutputMeta `json:"meta"`
	Data    V1EventView          `json:"data"`
}

// GetV1EventOutputMeta contains the meta-data of GetV1EventOutput.
type GetV1EventOutputMeta struct {
	Status string `json:"status"`
}

// GetV1AvailableEventListInput contains the required fields for querying for a list
// 	of event objects based on event end time
type GetV1AvailableEventListInput struct {
	ChannelID        int
	EndTimeUTCAfter  *time.Time
	EndTimeUTCBefore *time.Time
	OrderBy          *string
}

// GetV1AvailableEventListOutput contains the fields from the response of making a
// GET request via GetV1AvailableEventList.
type GetV1AvailableEventListOutput struct {
	Status  int                               `json:"status"`
	Message string                            `json:"message"`
	Meta    GetV1AvailableEventListOutputMeta `json:"meta"`
	Data    []*V1EventView                    `json:"data"`
}

// GetV1AvailableEventListOutputMeta contains the meta-data of GetV1AvailableEventListOutput.
type GetV1AvailableEventListOutputMeta struct {
	ChannelID        int       `json:"channel_id"`
	EndTimeUTCAfter  time.Time `json:"end_time_utc_after"`
	EndTimeUTCBefore time.Time `json:"end_time_utc_before"`
	OrderBy          string    `json:"order_by"`
	Status           string    `json:"status"`
	Limit            int       `json:"limit"`
}

// PostV1EventInput contains the required fields for making a POST request
// to the Oracle service's v1 API.
type PostV1EventInput struct {
	ChannelID    int       `json:"channel_id"`
	StartTimeUTC time.Time `json:"start_time_utc"`
	EndTimeUTC   time.Time `json:"end_time_utc"`
	TimeZoneID   string    `json:"time_zone_id"`
	Title        string    `json:"title"`
	Description  *string   `json:"description"`
	GameID       int       `json:"game_id"`
	CoverImageID *string   `json:"cover_image_id"`
}

// PostV1EventOutput contains the fields from the response of making a
// POST request via PostV1Event.
type PostV1EventOutput struct {
	Status  int                   `json:"status"`
	Message string                `json:"message"`
	Meta    PostV1EventOutputMeta `json:"meta"`
	Data    V1EventView           `json:"data"`
}

// PostV1EventOutputMeta contains the meta-data of PostV1EventOutput.
type PostV1EventOutputMeta struct {
	Status string `json:"status"`
}

// PutV1EventInput contains the required fields for updating an existing event from Oraclev1
type PutV1EventInput struct {
	StartTimeUTC time.Time `json:"start_time_utc"`
	EndTimeUTC   time.Time `json:"end_time_utc"`
	TimeZoneID   string    `json:"time_zone_id"`
	Title        string    `json:"title"`
	Description  *string   `json:"description"`
	GameID       int       `json:"game_id"`
	CoverImageID *string   `json:"cover_image_id"`
}

// PutV1EventOutput contains the response after updating an existing event from Oraclev1
type PutV1EventOutput struct {
	Status  int                  `json:"status"`
	Message string               `json:"message"`
	Meta    PutV1EventOutputMeta `json:"meta"`
	Data    V1EventView          `json:"data"`
}

// PutV1EventOutputMeta contains the meta-data of PutV1EventOutput.
type PutV1EventOutputMeta struct {
	Status string `json:"status"`
}

// GetV1EventSuggestionsOutput contains the full response for getting suggestions
type GetV1EventSuggestionsOutput struct {
	Status  int                             `json:"status"`
	Message string                          `json:"message"`
	Meta    GetV1EventSuggestionsOutputMeta `json:"meta"`
	Data    GetV1EventSuggestionsOutputData `json:"data"`
}

// GetV1EventSuggestionsOutputMeta contains the meta-data for suggestiosn
type GetV1EventSuggestionsOutputMeta struct {
	Status string `json:"status"`
}

// GetV1EventSuggestionsOutputData contains the multiple collections of events for suggestiosn
type GetV1EventSuggestionsOutputData struct {
	Future []*V1EventView `json:"future"`
	Past   []*V1EventView `json:"past"`
	Live   []*V1EventView `json:"live"`
}
