package view

import (
	"errors"
	"strings"
	"time"
)

const (
	// MinTitleLength is the minimum number of characters for
	// an event title.
	MinTitleLength = 1

	// MaxTitleLength is the maximum number of characters for
	// an event title.
	MaxTitleLength = 140

	// MaxDescriptionLength is the maximum number of characters
	// for an event description.
	MaxDescriptionLength = 5000
)

// The constants below are error strings for validating request parameters.
const (
	InvalidEventID           = "Event ID must be a positive integer."
	InvalidChannelID         = "Channel ID must be a positive integer."
	InvalidTimeRange         = "Event start time must be before event end time."
	InvalidTimeZoneID        = "Event time zone ID must be valid from the IANA Time Zone database and cannot be 'Local'."
	InvalidTimeZoneIDLength  = "Event time zone ID must not be blank."
	InvalidTitleLength       = "Event title must be a string containing at least 1 character and no more than 140 characters."
	InvalidDescriptionLength = "Event description cannot contain more than 5,000 characters."
	InvalidGameID            = "Game ID must be a positive integer."
	MissingEventCoverImage   = "Event cover image is required."
	MissingTimeParameters    = "A valid event start time and event end time are required."
)

// Validate verifies that each of the fields in PostV1EventInput is within
// the application constraints.
func (input *PostV1EventInput) Validate() error {
	msgs := []string{}

	if input.ChannelID <= 0 {
		msgs = append(msgs, InvalidChannelID)
	}

	if input.StartTimeUTC.IsZero() || input.EndTimeUTC.IsZero() {
		msgs = append(msgs, MissingTimeParameters)
	} else if !input.EndTimeUTC.After(input.StartTimeUTC) {
		msgs = append(msgs, InvalidTimeRange)
	}

	if len(input.TimeZoneID) == 0 {
		msgs = append(msgs, InvalidTimeZoneIDLength)
	}

	_, err := time.LoadLocation(input.TimeZoneID)
	if input.TimeZoneID == "Local" || err != nil {
		msgs = append(msgs, InvalidTimeZoneID)
	}

	if len(input.Title) < MinTitleLength || len(input.Title) > MaxTitleLength {
		msgs = append(msgs, InvalidTitleLength)
	}

	if input.Description != nil && len(*input.Description) > MaxDescriptionLength {
		msgs = append(msgs, InvalidDescriptionLength)
	}

	if input.GameID <= 0 {
		msgs = append(msgs, InvalidGameID)
	}

	if len(msgs) > 0 {
		return errors.New(strings.Join(msgs, " "))
	}

	return nil
}

// Validate verifies that each of the fields in PutV1EventInput is within
// the application constraints.
func (input *PutV1EventInput) Validate() error {
	msgs := []string{}

	if input.StartTimeUTC.IsZero() || input.EndTimeUTC.IsZero() {
		msgs = append(msgs, MissingTimeParameters)
	} else if !input.EndTimeUTC.After(input.StartTimeUTC) {
		msgs = append(msgs, InvalidTimeRange)
	}

	if len(input.TimeZoneID) == 0 {
		msgs = append(msgs, InvalidTimeZoneIDLength)
	}

	_, err := time.LoadLocation(input.TimeZoneID)
	if input.TimeZoneID == "Local" || err != nil {
		msgs = append(msgs, InvalidTimeZoneID)
	}

	if len(input.Title) < MinTitleLength || len(input.Title) > MaxTitleLength {
		msgs = append(msgs, InvalidTitleLength)
	}

	if input.Description != nil && len(*input.Description) > MaxDescriptionLength {
		msgs = append(msgs, InvalidDescriptionLength)
	}

	if input.GameID <= 0 {
		msgs = append(msgs, InvalidGameID)
	}

	if len(msgs) > 0 {
		return errors.New(strings.Join(msgs, " "))
	}

	return nil
}

// Validate verifies that each of the fields in GetV1AvailableEventList
// is within the application constraints.
func (input *GetV1AvailableEventListInput) Validate() error {
	if input.ChannelID <= 0 {
		return errors.New(InvalidChannelID)
	}

	return nil
}
