package view

import (
	"strings"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/suite"
)

type PostV1EventInputSuite struct {
	suite.Suite
}

func TestPostV1EventInputSuite(t *testing.T) {
	suite.Run(t, &PostV1EventInputSuite{})
}

func (suite *PostV1EventInputSuite) TestPostV1EventInput_InvalidZeroChannelID() {
	params := &PostV1EventInput{}
	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidChannelID)
	}
}

func (suite *PostV1EventInputSuite) TestPostV1EventInput_InvalidNegativeChannelID() {
	params := &PostV1EventInput{
		ChannelID: -9000,
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidChannelID)
	}
}

func (suite *PostV1EventInputSuite) TestPostV1EventInput_InvalidTimeRange() {
	params := &PostV1EventInput{
		ChannelID:    123,
		StartTimeUTC: time.Now().AddDate(2, 0, 0),
		EndTimeUTC:   time.Now(),
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidTimeRange)
	}
}

func (suite *PostV1EventInputSuite) TestPostV1EventInput_MissingStartTime() {
	params := &PostV1EventInput{
		ChannelID:  123,
		EndTimeUTC: time.Now(),
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), MissingTimeParameters)
	}
}

func (suite *PostV1EventInputSuite) TestPostV1EventInput_MissingEndTime() {
	params := &PostV1EventInput{
		ChannelID:    123,
		StartTimeUTC: time.Now(),
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), MissingTimeParameters)
	}
}

func (suite *PostV1EventInputSuite) TestPostV1EventInput_InvalidTimeZoneIDLength() {
	params := &PostV1EventInput{
		ChannelID:    123,
		StartTimeUTC: time.Now(),
		EndTimeUTC:   time.Now().AddDate(2, 0, 0),
		TimeZoneID:   "",
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidTimeZoneIDLength)
	}
}

func (suite *PostV1EventInputSuite) TestPostV1EventInput_InvalidTimeZoneID() {
	params := &PostV1EventInput{
		ChannelID:    123,
		StartTimeUTC: time.Now(),
		EndTimeUTC:   time.Now().AddDate(2, 0, 0),
		TimeZoneID:   "not a real time zone ID",
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidTimeZoneID)
	}
}

func (suite *PostV1EventInputSuite) TestPostV1EventInput_InvalidLocalTimeZoneID() {
	params := &PostV1EventInput{
		ChannelID:    123,
		StartTimeUTC: time.Now(),
		EndTimeUTC:   time.Now().AddDate(2, 0, 0),
		TimeZoneID:   "Local",
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidTimeZoneID)
	}
}

func (suite *PostV1EventInputSuite) TestPostV1EventInput_InvalidZeroTitle() {
	params := &PostV1EventInput{
		ChannelID:    123,
		StartTimeUTC: time.Now(),
		EndTimeUTC:   time.Now().AddDate(2, 0, 0),
		TimeZoneID:   "UTC",
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidTitleLength)
	}
}

func (suite *PostV1EventInputSuite) TestPostV1EventInput_InvalidLongTitle() {
	overlimitCharacters := make([]string, MaxTitleLength+1)
	for i := range overlimitCharacters {
		overlimitCharacters[i] = "+"
	}

	params := &PostV1EventInput{
		ChannelID:    123,
		StartTimeUTC: time.Now(),
		EndTimeUTC:   time.Now().AddDate(2, 0, 0),
		TimeZoneID:   "UTC",
		Title:        strings.Join(overlimitCharacters, ""),
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidTitleLength)
	}
}

func (suite *PostV1EventInputSuite) TestPostV1EventInput_InvalidLongDescription() {
	overlimitCharacters := make([]string, MaxDescriptionLength+1)
	for i := range overlimitCharacters {
		overlimitCharacters[i] = "+"
	}

	description := strings.Join(overlimitCharacters, "")

	params := &PostV1EventInput{
		ChannelID:    123,
		StartTimeUTC: time.Now(),
		EndTimeUTC:   time.Now().AddDate(2, 0, 0),
		TimeZoneID:   "UTC",
		Title:        "Valid Title",
		Description:  &description,
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidDescriptionLength)
	}
}

func (suite *PostV1EventInputSuite) TestPostV1EventInput_InvalidZeroGameID() {
	params := &PostV1EventInput{
		ChannelID:    123,
		StartTimeUTC: time.Now(),
		EndTimeUTC:   time.Now().AddDate(2, 0, 0),
		TimeZoneID:   "UTC",
		Title:        "Valid Title",
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidGameID)
	}
}

func (suite *PostV1EventInputSuite) TestPostV1EventInput_NoError() {
	params := &PostV1EventInput{
		ChannelID:    123,
		StartTimeUTC: time.Now(),
		EndTimeUTC:   time.Now().AddDate(2, 0, 0),
		TimeZoneID:   "UTC",
		Title:        "Valid Title",
		GameID:       9999,
	}

	assert.NoError(suite.T(), params.Validate())
}

type PutV1EventInputSuite struct {
	suite.Suite
}

func TestPutV1EventInputSuite(t *testing.T) {
	suite.Run(t, &PutV1EventInputSuite{})
}

func (suite *PutV1EventInputSuite) TestPutV1EventInput_MissingStartTime() {
	params := &PutV1EventInput{
		EndTimeUTC: time.Now(),
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), MissingTimeParameters)
	}
}

func (suite *PutV1EventInputSuite) TestPutV1EventInput_MissingEndTime() {
	params := &PutV1EventInput{
		StartTimeUTC: time.Now(),
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), MissingTimeParameters)
	}
}

func (suite *PutV1EventInputSuite) TestPutV1EventInput_InvalidTimeRange() {
	params := &PutV1EventInput{
		StartTimeUTC: time.Now().AddDate(2, 0, 0),
		EndTimeUTC:   time.Now(),
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidTimeRange)
	}
}

func (suite *PutV1EventInputSuite) TestPutV1EventInput_InvalidTimeZoneIDLength() {
	params := &PutV1EventInput{
		StartTimeUTC: time.Now(),
		EndTimeUTC:   time.Now().AddDate(2, 0, 0),
		TimeZoneID:   "",
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidTimeZoneIDLength)
	}
}

func (suite *PutV1EventInputSuite) TestPutV1EventInput_InvalidTimeZoneID() {
	params := &PutV1EventInput{
		StartTimeUTC: time.Now(),
		EndTimeUTC:   time.Now().AddDate(2, 0, 0),
		TimeZoneID:   "not a real time zone ID",
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidTimeZoneID)
	}
}

func (suite *PutV1EventInputSuite) TestPutV1EventInput_InvalidLocalTimeZoneID() {
	params := &PutV1EventInput{
		StartTimeUTC: time.Now(),
		EndTimeUTC:   time.Now().AddDate(2, 0, 0),
		TimeZoneID:   "Local",
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidTimeZoneID)
	}
}

func (suite *PutV1EventInputSuite) TestPutV1EventInput_InvalidZeroTitle() {
	params := &PutV1EventInput{
		StartTimeUTC: time.Now(),
		EndTimeUTC:   time.Now().AddDate(2, 0, 0),
		TimeZoneID:   "UTC",
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidTitleLength)
	}
}

func (suite *PutV1EventInputSuite) TestPutV1EventInput_InvalidLongTitle() {
	overlimitCharacters := make([]string, MaxTitleLength+1)
	for i := range overlimitCharacters {
		overlimitCharacters[i] = "+"
	}

	params := &PutV1EventInput{
		StartTimeUTC: time.Now(),
		EndTimeUTC:   time.Now().AddDate(2, 0, 0),
		TimeZoneID:   "UTC",
		Title:        strings.Join(overlimitCharacters, ""),
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidTitleLength)
	}
}

func (suite *PutV1EventInputSuite) TestPutV1EventInput_InvalidLongDescription() {
	overlimitCharacters := make([]string, MaxDescriptionLength+1)
	for i := range overlimitCharacters {
		overlimitCharacters[i] = "+"
	}

	description := strings.Join(overlimitCharacters, "")

	params := &PutV1EventInput{
		StartTimeUTC: time.Now(),
		EndTimeUTC:   time.Now().AddDate(2, 0, 0),
		TimeZoneID:   "UTC",
		Title:        "Valid Title",
		Description:  &description,
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidDescriptionLength)
	}
}

func (suite *PutV1EventInputSuite) TestPutV1EventInput_InvalidZeroGameID() {
	params := &PutV1EventInput{
		StartTimeUTC: time.Now(),
		EndTimeUTC:   time.Now().AddDate(2, 0, 0),
		TimeZoneID:   "UTC",
		Title:        "Valid Title",
	}

	err := params.Validate()

	if assert.Error(suite.T(), err) {
		assert.Contains(suite.T(), err.Error(), InvalidGameID)
	}
}

func (suite *PutV1EventInputSuite) TestPutV1EventInput_NoError() {
	params := &PutV1EventInput{
		StartTimeUTC: time.Now(),
		EndTimeUTC:   time.Now().AddDate(2, 0, 0),
		TimeZoneID:   "UTC",
		Title:        "Valid Title",
		GameID:       9999,
	}

	assert.NoError(suite.T(), params.Validate())
}
