with channel_hour_fragment as
(
    select
        channel_id,
        date_trunc('hour', time) as timepart,
        avg(case when env = 'production' then before end) as prod_before,
        avg(case when env = 'production' then merged end) as prod_merged,
        avg(case when env = 'staging' then merged end) as staging_merged
    from tahoe_all.pn_channel_concurrents
    where
        date >= date_trunc('day', getdate() - interval '60 days')
        and date < date_trunc('day', getdate() - interval '50 days')
    group by
        channel_id,
        date_trunc('hour', time)
),
esp_fragment as
(
    select
        channel_id,
        date_trunc('hour', time) as timepart,
        count(distinct video_session_id) as video_sessions,
        count(distinct client_ip) as unique_ips
    from tahoe.edge_segment_playlist
    where
        date >= date_trunc('day', getdate() - interval '60 days')
        and date < date_trunc('day', getdate() - interval '50 days')
        and coalesce(video_node, '') <> ''
        and content_host like video_node || '%'
    group by
        channel_id,
        date_trunc('hour', time)
),
esr_fragment as
(
    select
        channel_id,
        date_trunc('hour', time) as timepart,
        count(distinct video_session_id) as video_sessions,
        count(distinct client_ip) as unique_ips
    from tahoe.edge_segment_request
    where
        date >= date_trunc('day', getdate() - interval '60 days')
        and date < date_trunc('day', getdate() - interval '50 days')
        and coalesce(video_node, '') <> ''
        and coalesce(segment_headers, '') not like '%"Range":["bytes=0-1"]%'
    group by
        channel_id,
        date_trunc('hour', time)
),
suspected_bots as
(
    select 
        esp.channel_id, 
        esp.timepart, 
        (esp.video_sessions - esr.video_sessions)::float / (esp.video_sessions) as suspected_bot_ratio_sessions,
        (esp.unique_ips - esr.unique_ips)::float / (esp.unique_ips) as suspected_bot_ratio_ips
  from esp_fragment as esp
  left join esr_fragment as esr 
    on esp.timepart = esr.timepart 
    and esp.channel_id = esr.channel_id
),
x as
(
    select
        a.channel_id,
        a.timepart,
        a.prod_before,
        a.prod_merged,
        a.staging_merged,
        ((1 - b.suspected_bot_ratio_sessions) * a.prod_before)::int as before_guess_1,
        ((1 - b.suspected_bot_ratio_ips) * a.prod_before)::int as before_guess_2,
        greatest(a.prod_before, a.prod_merged, a.staging_merged) as highest,
        b.suspected_bot_ratio_sessions,
        b.suspected_bot_ratio_ips
    from channel_hour_fragment as a
    left join suspected_bots b 
        on b.timepart = a.timepart 
        and b.channel_id = a.channel_id
),
y as 
(
    select
        channel_id,
        timepart,
        case when prod_before       >= 0.85 * highest then highest else prod_before    end as unfiltered,
        case when prod_merged       >= 0.85 * highest then highest else prod_merged    end as site_ccu,
        case when staging_merged    >= 0.85 * highest then highest else staging_merged end as beta_ccu,
        case when before_guess_1    >= 0.85 * highest then highest else before_guess_1 end as best_guess_1,
        case when before_guess_2    >= 0.85 * highest then highest else before_guess_2 end as best_guess_2,
        suspected_bot_ratio_sessions,
        suspected_bot_ratio_ips
    from x
)
select 
    channel_id,
    '60-50' as fragment,
    timepart, 
    unfiltered, 
    site_ccu, 
    beta_ccu,
    least(best_guess_1, best_guess_2) as best_guess,
    suspected_bot_ratio_sessions,
    suspected_bot_ratio_ips
from y