﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Microsoft.AspNetCore.Mvc;
using Newtonsoft.Json;
using Resonance.Core;
using Resonance.Core.Attributes;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Services.CustomConfigService;

namespace Resonance.Api.Microservices.Amp.Controllers
{
    //[ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
    public class CustomConfigController : Controller
    {
        private ICustomConfigService _service;
        public Dictionary<string, string[]> _allowedAccess;

        static CustomConfigController()
        {
        }

        public CustomConfigController(ICustomConfigService service)
        {
            
            _service = service;            
            _allowedAccess = AppConfig.Data.CustomConfiguration.AllowedAccess;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="name">Corresponds with the filename name.json.gz</param>
        /// <returns></returns>
        [HttpGet("worker/amp/config/{system}/{name}")]
        public JsonResult GetConfig(string system, string name, bool load = false)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<object>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                if (CheckAllowedAccess(system, name))
                {                    
                    string result = _service.GetConfig(system, name, load);
                    response.ResponseData = JsonConvert.DeserializeObject(result);
                    response.Success = true;
                }
                else
                {
                    response.Success = false;
                    response.ResponseData = $"Invalid Request Values ({system}, {name})";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetConfig.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        private bool CheckAllowedAccess(string system, string name)
        {
            return _allowedAccess.ContainsKey(system) && _allowedAccess[system].Contains(name);
        }  

        [HttpGet("worker/amp/config/{system}/{name}/last-updated")]
        public JsonResult GetLastUpdated(string system, string name)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<string>() 
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                if (CheckAllowedAccess(system, name))
                {
                    DateTime? lastUpdated = _service.GetLastUpdated(system, name);
                    
                    if (lastUpdated.HasValue)
                    {
                        response.ResponseData = lastUpdated.ToString();
                        response.Success = true;
                    }
                    else
                    {
                        response.Success = false;
                    }
                }
                else
                {
                    response.Success = false;
                    response.ResponseData = $"Invalid Request Values ({system}, {name})";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetLastUpdated.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }
    }
}