﻿using System;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Configuration;
using Resonance.Core;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.FilterModels;
using Resonance.Microservices.Methods;
using Resonance.Microservices.Attributes;
using Resonance.Core.Models.DatabaseModels.RequestModels;
using Resonance.Core.Models.AuthModels;
using Microsoft.AspNetCore.Http;
using Resonance.Core.Models.ApiModels.RequestModels;
using Resonance.Core.Attributes;
using Resonance.Core.Exceptions;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Services.ActivityLoggerService;
using System.Diagnostics;
using Resonance.Core.Helpers.AuthHelpers;
using Newtonsoft.Json;
using Resonance.Core.ConstantData.Amp;
using Resonance.Core.Helpers.ErrorHelpers;

namespace Resonance.Api.Microservices.Amp.Controllers
{
    
    public class RequestController : Controller
    {
        private readonly IConfiguration _config;
        private readonly IActivityLoggerService _activityLoggerService;
        private static AmpMethods ampMethod;

        static RequestController()
        {
            ampMethod = new AmpMethods();
            ampMethod.Initialize();
        }

        public RequestController(IHttpContextAccessor httpContextAccessor, IConfiguration config, IActivityLoggerService activityLoggerService)
        {
            _config = config;
            _activityLoggerService = activityLoggerService;
        }

        [ActivityLogger("Rated Request")]
        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpPost("worker/amp/rate-request")]
        public JsonResult RateRequest([FromBody]RateRequest rateRequest)
        {
            var statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<BooleanResponse>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                var result = ampMethod.RateRequest(rateRequest);
                response.Success = result;
                response.ResponseData = new BooleanResponse() { Success = result };
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Unhandled error RateRequest";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for RateRequest.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ActivityLogger("Ended Request")]
        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.CanApproveRequests)]
        [HttpPost("worker/amp/end-request/{requestID}")]
        public JsonResult EndRequest(string requestID)
        {
            var statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<bool>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                ampMethod.EndRequest(requestID);
                response.Success = true;
                response.ResponseData = true;
                
            }
            catch(SalesforceException ex)
            {
                response.Success = false;
                response.ResponseData = false;
                statuscode = 400;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex.Message}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Salesforce: {ex.Message}";
            }
            catch (Exception ex)
            {
                response.Success = false;
                response.ResponseData = false;
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Unhandled error EndRequest";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for EndRequest.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ActivityLogger("Deleted Request")]
        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.CanCreateRequests)]
        [HttpPost("worker/amp/delete-request/{requestID}")]
        public JsonResult DeleteRequest(string requestID)
        {
            int statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<bool>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();

                RequestDetails request = ampMethod.GetRequest(requestID, false, false);
                AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];

                if (!request.Request.CanDelete(HttpContext, tokenData.User))
                {
                    throw new WebRequestException(System.Net.HttpStatusCode.Forbidden, "Unable to delete");
                }

                ampMethod.DeleteRequest(requestID, tokenData.User);

                response.Success = true;
                response.ResponseData = true;
            }
            catch (SalesforceException ex)
            {
                response.Success = false;
                response.ResponseData = false;
                statuscode = 400;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex.Message}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Salesforce: {ex.Message}";
            }
            catch (Exception ex)
            {
                response.Success = false;
                response.ResponseData = false;
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Unhandled error DeleteRequest";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for DeleteRequest.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ActivityLogger("Edited Request")]
        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpPost("worker/amp/edit-request/{requestID}")]
        public JsonResult EditRequest([FromBody]RequestDetails editRequest, string requestID)
        {
            var statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<Request>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                Log.Info($"EditRequest {requestID} Object: {JsonConvert.SerializeObject(editRequest)}");
                stopwatch.Start();
                AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                var result = ampMethod.EditRequest(editRequest, requestID, tokenData.User);
                if (result != null)
                {
                    response.Success = true;
                    response.ResponseData = result;
                }
                else
                {
                    statuscode = 400;
                    response.ErrorMessage = $@"Result is empty";
                }
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Unhandled error EditRequest";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for EditRequest.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ActivityLogger("Revise Request")]
        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpPost("worker/amp/revise-request/{requestID}")]
        public JsonResult ReviseRequest([FromBody]RequestDetails editRequest, string requestID)
        {
            var statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<Request>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                if(!PermissionHelper.HasPermission(HttpContext, ConstantsPermissions.Amp.CanApproveRequests))
                {
                    throw new WebRequestException(System.Net.HttpStatusCode.Forbidden, "Cant approve requests");
                }
                var hasRevision = ampMethod.RequestHasRevision(requestID);
                if (hasRevision.Item1 == 500)
                {
                    statuscode = 500;
                    response.ErrorMessage = "Exception in RequestHasRevision";
                } else if (hasRevision.Item2)
                {
                    statuscode = 400;
                    response.ErrorMessage = $"{requestID} already has been revised";
                }

                if (statuscode == 200)
                {
                    var result = ampMethod.ReviseRequest(editRequest, requestID, tokenData.User);
                    if (result != null)
                    {
                        response.Success = true;
                        response.ResponseData = result;
                    }
                    else
                    {
                        statuscode = 400;
                        response.ErrorMessage = $@"Result is empty";
                    }
                }
            }
            catch (WebRequestException ex)
            {
                statuscode = (int)ex.StatusCode;

                string userName = "";
                try
                {
                    AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                    userName = tokenData.User;
                }
                catch (Exception)
                {

                }

                Log.Error(ex.ResponseBody + " for " + userName + " request " + requestID);
                response.ErrorMessage = ex.ResponseBody + " for " + userName + " request " + requestID;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Unhandled error ReviseRequest";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for ReviseRequest.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ActivityLogger("Process Request Revisions")]
        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpPost("worker/amp/process-request-revisions/{requestID}")]
        public JsonResult ProcessRequestRevisions(string requestID)
        {
            var statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<Request>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                RequestDetails request = ampMethod.GetRequest(requestID, false, false);

                if(!request.Request.CanSubmit(HttpContext, tokenData.User))
                {
                    throw new WebRequestException(System.Net.HttpStatusCode.Forbidden, "Only creator can process revisions");
                }
                var isRevision = ampMethod.RequestIsRevision(requestID);
                if (isRevision.Item1 == 500)
                {
                    response.ErrorMessage = "Exception in RequestIsRevision";
                    statuscode = 500;
                }
                else if (isRevision.Item2)
                {
                    response.ErrorMessage = $"Request {requestID} is not a base request";
                    statuscode = 400;
                }

                if (statuscode == 200)
                {
                    var result = ampMethod.ProcessRequestRevisions(true, requestID, tokenData);
                    if (result != null)
                    {
                        response.Success = true;
                        response.ResponseData = result;
                    }
                    else
                    {
                        statuscode = 400;
                        response.ErrorMessage = $@"Result is empty";
                    }
                }

            }
            catch (SalesforceException ex)
            {
                statuscode = 400;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex.Message}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Salesforce: {ex.Message}";
            }
            catch (WebRequestException ex)
            {
                statuscode = (int)ex.StatusCode;
                string userName = "";
                try
                {
                    AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                    userName = tokenData.User;
                }
                catch (Exception)
                {

                }

                Log.Error(ex.ResponseBody + " for " + userName + " request " + requestID);
                response.ErrorMessage = ex.ResponseBody + " for " + userName + " request " + requestID;

            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Unhandled error ProcessRequestRevisions";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for ProcessRequestRevisions.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ActivityLogger("Draft Request")]
        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpPost("worker/amp/draft-request/{requestID}")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public JsonResult DraftRequest([FromBody]RequestDetails editRequest, string requestID)
        {
            var statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<Request>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                Log.Info($"DraftRequest {requestID} Object: {JsonConvert.SerializeObject(editRequest)}");
                AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                var result = ampMethod.EditRequest(editRequest, requestID, tokenData.User, RequestStatus.Draft);
                if (result != null)
                {
                    response.Success = true;
                    response.ResponseData = result;
                }
                else
                {
                    statuscode = 400;
                    response.ErrorMessage = $@"Result is empty";
                }
            }
            catch (WebRequestException ex)
            {
                statuscode = (int)ex.StatusCode;
                string userName = "";
                try
                {
                    AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                    userName = tokenData.User;
                }
                catch (Exception)
                {

                }

                Log.Error(ex.ResponseBody + " for " + userName + " request " + requestID);
                response.ErrorMessage = ex.ResponseBody + " for " + userName + " request " + requestID;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Unhandled error DraftRequest";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for DraftRequest.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ActivityLogger("Submit Request")]
        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpPost("worker/amp/submit-request/{requestID}")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public JsonResult SubmitRequest(string requestID)
        {
            var statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<Request>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                RequestDetails request = ampMethod.GetRequest(requestID, false, false);
                AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                if (!request.Request.CanSubmit(HttpContext, tokenData.User))
                {
                    throw new WebRequestException(System.Net.HttpStatusCode.Forbidden, "Unable to submit");
                }
                var revisionID = ampMethod.RequestHasRevision(requestID);
                if (revisionID.Item1 == 500)
                {
                    statuscode = 500;
                    response.ErrorMessage = "Exception in RequestHasRevision";
                }
                else if (revisionID.Item2)
                {
                    statuscode = 400;
                    response.ErrorMessage = $"{requestID} cannot be submitted because it has a revision";
                }

                if (statuscode == 200)
                {
                    if (ampMethod.SubmitRequest(requestID, tokenData))
                    {
                        response.Success = true;
                        response.ResponseData = new Core.Models.DatabaseModels.RequestModels.Request() { ID = requestID };
                    }
                    else
                    {
                        statuscode = 400;
                        response.ErrorMessage = $@"Submit Request Failed";
                    }
                }
            }
            catch (WebRequestException ex)
            {
                statuscode = (int)ex.StatusCode;
                string userName = "";
                try
                {
                    AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                    userName = tokenData.User;
                }
                catch (Exception)
                {

                }

                Log.Error(ex.ResponseBody + " for " + userName + " request " + requestID);
                response.ErrorMessage = ex.ResponseBody + " for " + userName + " request " + requestID;
            }
            catch (SalesforceException ex)
            {
                statuscode = 400;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex.Message}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Salesforce: {ex.Message}";
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Unhandled error SubmitRequest";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for SubmitRequest.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ActivityLogger("Finalized Request")]
        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpPost("worker/amp/finalize-request/{requestID}")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public JsonResult FinalizeRequest([FromBody]FinalizeRequest finalizeRequest, string requestID)
        {
            var statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<Request>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                Log.Info($"FinalizeRequest {requestID} Object: {JsonConvert.SerializeObject(finalizeRequest)}");
                stopwatch.Start();
                if (ampMethod.ChannelsContainDuplicatePriorityValues(finalizeRequest.Channels))
                {
                    throw new Exception("Request channels cannot contain duplicate priority values");
                }
                RequestDetails request = ampMethod.GetRequest(requestID, false, false);
                AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                if (!request.Request.CanFinalize(HttpContext, tokenData.User))
                {
                    Log.Info($"Unable to Finalize:{requestID}");
                    throw new WebRequestException(System.Net.HttpStatusCode.Forbidden, $"Unable to Finalize:{requestID}");
                }
                if(request.Request.Status != RequestStatus.PendingFinalization)
                {
                    Log.Info($"Unable to Finalize, Not as {RequestStatus.PendingFinalization}:{requestID}");
                    throw new WebRequestException(System.Net.HttpStatusCode.Forbidden, $"Unable to Finalize, Not as {RequestStatus.PendingFinalization}:{requestID}");
                }
                if (ampMethod.FinalizeRequest(HttpContext, requestID, tokenData, finalizeRequest))
                {
                    response.Success = true;
                    response.ResponseData = new Core.Models.DatabaseModels.RequestModels.Request() { ID = requestID };
                }
            }
            catch (WebRequestException ex)
            {
                statuscode = (int)ex.StatusCode;
                string userName = "";
                try
                {
                    AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                    userName = tokenData.User;
                }
                catch (Exception)
                {

                }

                Log.Error(ex.ResponseBody + " for " + userName + " request " + requestID);
                response.ErrorMessage = ex.ResponseBody + " for " + userName + " request " + requestID;
            }
            catch(SalesforceException ex)
            {          
                statuscode = 400;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex.Message}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Salesforce: {ex.Message}";
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Unhandled error FinalizeRequest";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for FinalizeRequest.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpPost("worker/amp/get-request/{requestID}")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public JsonResult GetRequest(string requestID)
        {
            var statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<RequestDetails>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                var result = ampMethod.GetRequest(requestID);
                if (result != null)
                {
                    AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                    if (!result.Request.CanSee(HttpContext, tokenData.User))
                    {
                        throw new WebRequestException(System.Net.HttpStatusCode.Forbidden, "Unable to see request");
                    }
                    response.Success = true;
                    response.ResponseData = result;
                }
            }
            catch(WebRequestException ex)
            {
                statuscode = (int)ex.StatusCode;
                string userName = "";
                try
                {
                    AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                    userName = tokenData.User;
                }
                catch(Exception)
                {

                }

                Log.Error(ex.ResponseBody + " for " + userName + " request " + requestID);
                response.ErrorMessage = ex.ResponseBody + " for " + userName + " request " + requestID;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception while processing GetRequest";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetRequest.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpPost("worker/amp/get-request-and-revisions/{requestID}")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public JsonResult GetRequestAndRevisions(string requestID)
        {
            var statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<RequestWithRevisionDetails>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                var result = ampMethod.GetRequestWithRevisions(requestID);
                if (result != null)
                {
                    response.Success = true;
                    response.ResponseData = result;
                }
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetRequestAndRevisions";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetRequestAndRevisions.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpPost("worker/amp/request-user-auto-complete")]
        public JsonResult RequestUserAutoComplete([FromBody] RequestUserSearch userSearch)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiListResponse<RequestUserSearchUser>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                var result = ampMethod.SearchUsers(userSearch.LoginNameSearch);
                if (result != null)
                {
                    response.Success = true;
                    response.ResponseData = result;
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception while processing RequestUserAutoComplete";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for RequestUserAutoComplete.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ActivityLogger("Create Request")]
        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpPost("worker/amp/create-request")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public JsonResult CreateRequest([FromBody]ListingFilter filter)
        {
            var statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<Request>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                var result = ampMethod.CreateRequest(ref filter, tokenData.User);
                if (result != null)
                {
                    response.Success = true;
                    response.ResponseData = result;
                }
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Unhandled error CreateRequest";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for CreateRequest.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }


        [ActivityLogger("Create Empty Request")]
        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpPost("worker/amp/create-empty-request")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public JsonResult CreateEmptyRequest()
        {
            var statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<Request>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                var result = ampMethod.CreateEmptyRequest(tokenData.User);
                if (result != null)
                {
                    response.Success = true;
                    response.ResponseData = result;
                }
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Unhandled error CreateEmptyRequest";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for CreateEmptyRequest.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpPost("worker/amp/list-my-requests")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public JsonResult ListMyRequests()
        {
            var statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<RequestList>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                var result = ampMethod.GetRequests(tokenData.User, HttpContext);
                if (result != null)
                {
                    response.Success = true;
                    response.ResponseData = result;
                }
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Unhandled error ListMyRequests";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for ListMyRequests.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ActivityLogger("Cancel Request")]
        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.CanCreateRequests)]
        [HttpPost("worker/amp/cancel-request/{requestID}")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public JsonResult CancelRequest(string requestID)
        {
            var statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<bool>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                RequestDetails request = ampMethod.GetRequest(requestID, false, false);
                AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];

                if (!request.Request.CanCancel(HttpContext, tokenData.User))
                {
                    throw new WebRequestException(System.Net.HttpStatusCode.Forbidden, "Unable to cancel");
                }

                ampMethod.CancelRequest(requestID, tokenData.User);
                response.Success = true;
                response.ResponseData = true;
            }
            catch(WebRequestException ex)
            {
                statuscode = (int)ex.StatusCode;
                string userName = "";

                try
                {
                    AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                    userName = tokenData.User;
                }
                catch (Exception)
                {

                }

                Log.Error(ex.ResponseBody + " for " + userName + " request " + requestID);
                response.ErrorMessage = ex.ResponseBody + " for " + userName + " request " + requestID;
            }
            catch (SalesforceException ex)
            {
                statuscode = 400;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex.Message}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Salesforce: {ex.Message}";
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Unhandled error CancelRequest";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for CancelRequest.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ActivityLogger("Reject Request")]
        [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.CanCreateRequests)]
        [HttpPost("worker/amp/reject-request/{requestID}")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public JsonResult RejectRequest(string requestID)
        {
            var statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<bool>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                RequestDetails request = ampMethod.GetRequest(requestID, false, false);
                AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];

                if (!request.Request.CanReject(HttpContext, tokenData.User))
                {
                    throw new WebRequestException(System.Net.HttpStatusCode.Forbidden, "Unable to reject");
                }

                ampMethod.RejectRequest(requestID, tokenData.User);
                response.Success = true;
                response.ResponseData = true;
            }
            catch(WebRequestException ex)
            {
                statuscode = (int)ex.StatusCode;

                string userName = "";
                try
                {
                    AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                    userName = tokenData.User;
                }
                catch (Exception)
                {

                }

                Log.Error(ex.ResponseBody + " for " + userName + " request " + requestID);
                response.ErrorMessage = ex.ResponseBody + " for " + userName + " request " + requestID;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Unhandled error RejectRequest";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for RejectRequest.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }
    }
}
 