﻿using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.Infrastructure;
using Microsoft.Extensions.Configuration;
using Resonance.Core;
using Resonance.Core.Attributes;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.ServiceModels.TwitchModels;
using Resonance.Core.Services.ActivityLoggerService;
using Resonance.Core.Services.ColumnFilterService;
using Resonance.Core.Services.LdapService;
using Resonance.Microservices.Methods;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Threading.Tasks;
using static Resonance.Core.Models.FilterModels.CustomTwitchDirectFilters;

namespace Resonance.Api.Microservices.Amp.Controllers
{
    [Route("twitch-direct")]
    [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
    public class TwitchDirectController : Controller
    {
        private readonly IConfiguration _config;
        private IColumnFilterService _columnFilterService;
        private readonly IActivityLoggerService _activityLoggerService;
        private readonly IHttpContextAccessor _httpContextAccessor;
        private readonly IActionContextAccessor _actionContextAccessor;
        private readonly ILdapService _ldapService;
        private static TwitchDirectMethods directMethods = new TwitchDirectMethods();

        public TwitchDirectController(IHttpContextAccessor httpContextAccessor, IConfiguration config, IColumnFilterService columnFilterService, IActivityLoggerService activityLoggerService, IActionContextAccessor actionContextAccessor, ILdapService ldapService)
        {
            _actionContextAccessor = actionContextAccessor;
            _httpContextAccessor = httpContextAccessor;
            _config = config;
            _columnFilterService = columnFilterService;
            _activityLoggerService = activityLoggerService;
            _ldapService = ldapService;
            directMethods.Initialize();
        }

        [HttpPost("get-arbitrary-summary")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public JsonResult GetTwitchArbitrarySummary([FromBody]ArbitrarySummaryFilter filter)
        {
            var response = new ApiListResponse<TwitchDirectArbitrarySummaryModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            var stopwatch = new Stopwatch();

            try
            {
                stopwatch.Start();

                if(filter == null)
                {
                    throw new ArgumentNullException("filter");
                }

                if(filter.ChannelTimeRangeRequest == null)
                {
                    throw new ArgumentNullException("filter.ChannelTimeRangeRequest");
                }

                if(filter.ChannelTimeRangeRequest.Count == 0)
                {
                    throw new InvalidOperationException("ChannelTimeRangeRequest List is empty");
                }

                if(filter.ChannelTimeRangeRequest.Values == null)
                {
                    throw new ArgumentNullException("filter.ChannelTimeRangeRequest.Values");
                }

                if(filter.ChannelTimeRangeRequest.Values.Count == 0)
                {
                    throw new InvalidOperationException("ChannelTimeRangeRequest.Values List is empty");
                }

                response.ResponseData = directMethods.GetArbitrarySummaryData(filter);

                response.Success = true;
            }
            catch(Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetTwitchArbitrarySummary.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }
    }
}
