﻿using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.Infrastructure;
using Microsoft.Extensions.Configuration;
using Resonance.Core;
using Resonance.Core.Attributes;
using Resonance.Core.Helpers.AuthHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.ApiModels.TwitchModels;
using Resonance.Core.Models.AuthModels;
using Resonance.Core.Services.ActivityLoggerService;
using Resonance.Core.Services.ColumnFilterService;
using Resonance.Core.Services.LdapService;
using Resonance.Microservices.Attributes;
using Resonance.Microservices.Methods;
using System;
using System.Collections.Concurrent;
using System.Diagnostics;
using System.Threading.Tasks;

namespace Resonance.Api.Microservices.Amp.Controllers
{
    [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
    public class TwitchUserDetailsController : Controller
    {
        private readonly IConfiguration _config;
        private static AmpMethods ampMethod;
        private IColumnFilterService _columnFilterService;
        private readonly IActivityLoggerService _activityLoggerService;
        private readonly IHttpContextAccessor _httpContextAccessor;
        private readonly IActionContextAccessor _actionContextAccessor;
        private readonly ILdapService _ldapService;

        static TwitchUserDetailsController()
        {
            ampMethod = new AmpMethods();
            ampMethod.Initialize();
        }

        public TwitchUserDetailsController(IHttpContextAccessor httpContextAccessor, IConfiguration config, IColumnFilterService columnFilterService, IActivityLoggerService activityLoggerService, IActionContextAccessor actionContextAccessor, ILdapService ldapService)
        {
            _actionContextAccessor = actionContextAccessor;
            _httpContextAccessor = httpContextAccessor;
            _config = config;
            _columnFilterService = columnFilterService;
            _activityLoggerService = activityLoggerService;
            _ldapService = ldapService;
        }

        [ActivityLogger("UserPage")]
        [HttpGet("worker/amp/get-twitch-user-page/{twitchLogin}")]
        public async Task<JsonResult> GetUserPage(string twitchLogin)
        {
            var response = new ApiResponse<ChannelDetailsModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };

            var stopwatch = new Stopwatch();
            try
            {
                stopwatch.Start();
                AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                this._activityLoggerService.LogActivity("UserPage:" + twitchLogin, "TwitchUserDetailsController", "GetUserPage", tokenData.User);
                response.ResponseData = await ampMethod.GetChannelDetails(twitchLogin, HttpContext, response.Metrics);
                if (response.ResponseData != null)
                {
                    response.Success = true;
                }
                else
                {
                    response.ErrorMessage = "Channel not found";
                    HttpContext.Response.StatusCode = 404;
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "GetUserPage: Exception while getting channel details data";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetUserPage.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }
    }
}
