﻿using System;
using System.Linq;
using System.Collections.Generic;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Configuration;
using Resonance.Core;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.ApiModels.TwitchModels;
using Resonance.Core.Models.DatabaseModels.TwitchUserListingModels;
using Resonance.Core.Models.FilterModels;
using Resonance.Core.Models.DatabaseModels.RedshiftModels;
using Resonance.Microservices.Methods;
using Resonance.Microservices.Attributes;
using Resonance.Core.Attributes;
using Resonance.Core.Services.ColumnFilterService;
using Resonance.Core.Services.ActivityLoggerService;
using Resonance.Core.Models.AuthModels;
using Microsoft.AspNetCore.Mvc.Infrastructure;
using Resonance.Core.Models.ServiceModels.ActivityLoggerService;
using Resonance.Core.Services.LdapService;
using Resonance.Core.Services.LocalizationService;
using System.Diagnostics;
using Resonance.Core.Helpers.AuthHelpers;

namespace Resonance.Api.Microservices.Amp.Controllers
{
    [ResonanceAuth("Amp", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
    public class TwitchUserListingController : Controller
    {
        private readonly IConfiguration _config;
        private static AmpMethods ampMethod;
        private IColumnFilterService _columnFilterService;
        private readonly IActivityLoggerService _activityLoggerService;
        private readonly IHttpContextAccessor _httpContextAccessor;
        private readonly IActionContextAccessor _actionContextAccessor;
        private readonly ILdapService _ldapService;
        private readonly ILocalizationService _localizationService;

        static TwitchUserListingController()
        {
            ampMethod = new AmpMethods();
            ampMethod.Initialize();
        }

        public TwitchUserListingController(IHttpContextAccessor httpContextAccessor, IConfiguration config, IColumnFilterService columnFilterService, IActivityLoggerService activityLoggerService, IActionContextAccessor actionContextAccessor, ILdapService ldapService, ILocalizationService localizationService)
        {
            _actionContextAccessor = actionContextAccessor;
            _httpContextAccessor = httpContextAccessor;
            _config = config;
            _columnFilterService = columnFilterService;
            _activityLoggerService = activityLoggerService;
            _ldapService = ldapService;
            _localizationService = localizationService;
        }

        [HttpPost("worker/amp/get-twitch-user-listing-overview")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public JsonResult GetUserListingOverview([FromBody]ListingFilter filter)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<UserListingOverviewModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };
            stopwatch.Start();

            if (filter == null)
            {
                response.RequestEndTime = DateTime.UtcNow;
                response.ErrorMessage = "Filter is null.";
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
                return new JsonResult(response);
            }
            AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
            filter.FilterOutRestrictedColumns(Request, _columnFilterService, _activityLoggerService, tokenData.User, _actionContextAccessor);

            try
            {
                var result = ampMethod.GetUserListingOverview(ref filter);
                if (result != null)
                {
                    response.Success = true;
                    response.ResponseData = result;
                }
                else
                {
                    response.ErrorMessage = "GetUserListingOverview result null";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception while processing GetUserListingOverview";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetUserListingOverview.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("worker/amp/auto-complete-user-list")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public JsonResult AutocompleteUserList([FromBody]UserAutocompleteRequest request)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<UserAutocompleteResponse>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                response.ResponseData = new UserAutocompleteResponse() {  Entries = _ldapService.SearchUsers(request.Search, HttpContext) };
                response.Success = true;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception while processing username search";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for AutocompleteUserList.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("worker/amp/get-twitch-user-listing")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public JsonResult GetUserListing([FromBody]ListingFilter filter)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiListResponse<dynamic>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };
            stopwatch.Start();

            if (filter == null)
            {
                response.ErrorMessage = "Filter is null";
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
                return new JsonResult(response);
            }

            if (filter.Limit <= 0 || filter.Limit > 5000 || filter.Page < 0 || filter.AggregateType == Constants.AggregationType.Unknown)
            {
                response.ErrorMessage = "Filter limit <= 0, > 100, Page < 0, or Unknown aggregation type";
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
                return new JsonResult(response);
            }
            AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
            filter.FilterOutRestrictedColumns(Request, _columnFilterService, _activityLoggerService, tokenData.User, _actionContextAccessor);                             

            try
            {
                var result = ampMethod.GetUserListingByFilter(HttpContext, ref filter);
                _activityLoggerService.LogActivity(new ActivityLogData() { Action = "GetUserListing", Controller = "TwitchUserListingController", Timestamp = DateTime.UtcNow, User = tokenData.User, Message = "Searched Data: " + filter.AggregateType.ToString() });
                foreach(var searchFilter in filter.QueryFilters)
                {
                    _activityLoggerService.LogActivity(new ActivityLogData() { Action = "GetUserListing", Controller = "TwitchUserListingController", Timestamp = DateTime.UtcNow, User = tokenData.User, Message = "Used Filter: " +  searchFilter.Key });
                }
                if (result != null)
                {
                    response.Success = true;
                    response.ResponseData = result;
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception while processing GetUserListing";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetUserListing.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpGet("worker/amp/get-twitch-user-listing-columns")]
        public JsonResult GetUserListingColumns()
        {
            var stopwatch = new Stopwatch();
            var response = new ApiListResponse<string>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                response.ResponseData = TwitchUserListingColumnDefinition.Default();
                response.Success = true;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception while processing GetUserListingColumns";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetUserListingColumns.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ActivityLogger("CSV Export")]
        [HttpPost("worker/amp/get-csv-export")]
        public JsonResult GetCsvExport([FromBody]TwitchDataExportRequestModel exportRequest)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<string>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                var permissions = Core.Helpers.ApiHelpers.PermissionHelper.GetPermissions(HttpContext);                
                exportRequest.Columns = _columnFilterService.FilterOutRestrictedColumns(permissions, exportRequest.Columns, tokenData.User, _activityLoggerService, _actionContextAccessor).ToList();
                response.ResponseData = ampMethod.GetCSVExport(HttpContext, exportRequest);
                foreach (string column in exportRequest.Columns)
                {
                    _activityLoggerService.LogActivity(new ActivityLogData() { Action = "GetCsvExport", Controller = "TwitchUserListingController", Timestamp = DateTime.UtcNow, User = tokenData.User, Message = "Exported Column: " + column });
                }
                response.Success = true;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception while processing GetCsvExport";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetCsvExport.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("worker/amp/get-twitch-user-listing-exportable-columns")]
        public JsonResult GetUserExportableListingColumns()
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<TwitchDataExportColumnsModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
                var baseColumnList = TwitchUserListingColumnDefinition.AllExportableColumns.ToArray();
                baseColumnList = baseColumnList.FilterOutRestrictedColumns(Request, _columnFilterService, _activityLoggerService, tokenData.User, _actionContextAccessor).ToArray();
                response.ResponseData = new TwitchDataExportColumnsModel() { KeyGroupings = baseColumnList.GroupBy(x => TwitchUserListingColumnDefinition.GetCategory(x)).ToDictionary(x => x.Key, y => y.Select(z => new TwitchDataExportColumns() { Key = z, Label = _localizationService.GetString(z) }).ToList()) };
                response.Success = true;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception while processing GetUserExportableListingColumns";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetUserExportableListingColumns.");
                }                
                stopwatch.Stop();                
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpGet("worker/amp/get-twitch-user-listing-aggregation-types")]
        public JsonResult GetUserListingAggregationTypes()
        {
            var stopwatch = new Stopwatch();
            var response = new ApiListResponse<int>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                response.ResponseData = new int[5]
                {
                    (int)Constants.AggregationType.Week,
                    (int)Constants.AggregationType.ThirtyDay,
                    (int)Constants.AggregationType.SixtyDay,
                    (int)Constants.AggregationType.NintyDay,
                    (int)Constants.AggregationType.BankerYear
                };
                response.Success = true;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception while processing GetUserListingAggregationTypes";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetUserListingAggregationTypes.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }        
    }
}