﻿using System;
using System.Diagnostics;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.Infrastructure;
using Microsoft.Extensions.Configuration;
using Resonance.Core;
using Resonance.Core.Attributes;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.ApiModels.TwitchModels;
using Resonance.Core.Services.ActivityLoggerService;
using Resonance.Core.Services.ColumnFilterService;
using Resonance.Core.Services.LdapService;
using Resonance.Microservices.Methods;

namespace Resonance.Api.Microservices.Amp.Controllers
{
    [Route("user-admin")]
    [ResonanceAuth("Amp", requiredPermissionsAny: ConstantsPermissions.Amp.ServiceUpdate)]
    public class UserAdminController : Controller
    {
        private readonly IConfiguration _config;
        private IColumnFilterService _columnFilterService;
        private readonly IActivityLoggerService _activityLoggerService;
        private readonly IHttpContextAccessor _httpContextAccessor;
        private readonly IActionContextAccessor _actionContextAccessor;
        private readonly ILdapService _ldapService;
        private static AmpAdminMethods ampAdminMethod;

        public UserAdminController(IHttpContextAccessor httpContextAccessor, IConfiguration config, IColumnFilterService columnFilterService, IActivityLoggerService activityLoggerService, IActionContextAccessor actionContextAccessor, ILdapService ldapService)
        {
            _actionContextAccessor = actionContextAccessor;
            _httpContextAccessor = httpContextAccessor;
            _config = config;
            _columnFilterService = columnFilterService;
            _activityLoggerService = activityLoggerService;
            _ldapService = ldapService;

            ampAdminMethod = new AmpAdminMethods();
            ampAdminMethod.Initialize();
        }

        [HttpGet("manual-scorecard-entry")]
        public JsonResult ManualScorecardEntry(long twitchUserID)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<UserScorecardManualDataModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                response.ResponseData = ampAdminMethod.GetUserManualScorecard(twitchUserID);
                response.Success = true;
            }
            catch(Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception while getting user scorecard manual data";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for ManualScorecardEntry.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpPost("manual-scorecard-update")]
        public JsonResult ManualScorecardUpdate([FromBody]UserScorecardManualDataModel data)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<bool>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };

            try
            {
                stopwatch.Start();
                response.ResponseData = ampAdminMethod.UpdateUserScorecard(data);
                response.Success = true;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                response.ErrorMessage = "Exception while setting user scorecard manual data";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for ManualScorecardUpdate.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }
    }
}