﻿using System;
using System.Diagnostics;
using System.IO;
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.CookiePolicy;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.HttpOverrides;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.Logging;
using Microsoft.Extensions.Options;
using NLog.Extensions.Logging;
using NLog.Web;
using Resonance.Core;
using Resonance.Core.Exceptions;
using Resonance.Core.Helpers.ConfigurationHelpers;
using Resonance.Core.Helpers.FileHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Middleware;
using Resonance.Core.Services;
using Resonance.Core.Services.ActivityLoggerService;
using Resonance.Core.Services.CustomConfigService;
using Resonance.Core.Services.ColumnFilterService;
using static RequestResponseLoggingMiddleware;
using Microsoft.AspNetCore.Mvc.Infrastructure;
using Resonance.Core.Services.LdapService;
using Resonance.Core.Services.LocalizationService;
using Microsoft.AspNetCore.DataProtection;

namespace Resonance.Api.Microservices.Amp
{
    public class Startup
    {
        public Startup(IConfiguration configuration)
        {
            Configuration = configuration;
        }

        public IConfiguration Configuration { get; }

        // This method gets called by the runtime. Use this method to add services to the container.
        public void ConfigureServices(IServiceCollection services)
        {
            services.ConfigureCorsFilesForEnvironments();
            /*
            services.Configure<CookiePolicyOptions>(options =>
            {
                // This lambda determines whether user consent for non-essential cookies is needed for a given request.
                options.CheckConsentNeeded = context => true;
                options.MinimumSameSitePolicy = SameSiteMode.None;
            });
            */
            if (!Directory.Exists("appdata"))
            {
                Directory.CreateDirectory("appdata");
            }
            services.AddSingleton<IHttpContextAccessor, HttpContextAccessor>();            
            services.AddSingleton<IActivityLoggerService, ActivityLoggerService>(x => new ActivityLoggerService("Amp", Constants.AppConfig.Application.Environment));            
            services.AddSingleton<ICustomConfigService, CustomConfigService>(x => new CustomConfigService(Constants.AppConfig.Application.Environment));
            services.AddSingleton<IColumnFilterService, ColumnFilterService>();
            services.AddSingleton<IActionContextAccessor, ActionContextAccessor>();
            services.AddSingleton<ILdapService, LdapService>();
            services.AddSingleton<ILocalizationService, LocalizationService>((prov) => new LocalizationService("en-us", Constants.AppConfig.Application.Environment));
            services.AddMemoryCache();

            services
                .AddAntiforgery(o => o.HeaderName = "XSRF-Token")
                .AddMvc(options =>
            {
                /*
                options.CacheProfiles.Add("default", new CacheProfile
                {
                    Duration = 30,
                    Location = ResponseCacheLocation.Any
                });
                */
            })
            .SetCompatibilityVersion(CompatibilityVersion.Version_2_2);

        }

        // This method gets called by the runtime. Use this method to configure the HTTP request pipeline.
        public void Configure(IApplicationBuilder app, IHostingEnvironment env, ILoggerFactory loggerFactory)
        {
            loggerFactory.AddNLog();
            env.ConfigureNLog(FileHelper.GetRelativePath($"nlog.{Constants.AppConfig.Application.Environment}.config"));
            //app.UseHsts();
            //app.UseHttpsRedirection();
            
            app.UseForwardedHeaders(new ForwardedHeadersOptions
            {
                ForwardedHeaders = ForwardedHeaders.XForwardedFor | ForwardedHeaders.XForwardedProto
            });

            app.UseStatusCodePagesWithReExecute("/error/{0}");
            app.UseExceptionHandler("/error/500");
            app.UseMiddleware<ErrorHandlerMiddleware>(Options.Create(new ErrorHandlerMiddlewareOptions()));
            app.UseMiddleware<ConfigurationMiddleware>(Options.Create(new ConfigurationMiddlewareOptions()));
            app.UseMiddleware<TraceMiddleware>(Options.Create(new TraceMiddlewareOptions()));
            //app.UseMiddleware<ValidateAntiForgeryTokenMiddleware>();
            //app.UseStaticFiles();  
            //app.UseMiddleware<DeChunkerMiddleware>();

            /*Action<RequestProfilerModel> requestResponseHandler = requestProfilerModel =>
            {
                Log.Info(requestProfilerModel.Request);
                Log.Info(requestProfilerModel.Response);
            };
            app.UseMiddleware<RequestResponseLoggingMiddleware>(requestResponseHandler);
            */
            var cookiePolicyOptions = new CookiePolicyOptions
            {
                MinimumSameSitePolicy = SameSiteMode.Lax,
                HttpOnly = HttpOnlyPolicy.Always,
                Secure = CookieSecurePolicy.Always
            };
            app.UseCookiePolicy();            

            app.UseCors(env.EnvironmentName);

            app.UseMvc(routes =>
            {
                routes.MapRoute(
                    name: "default",
                    template: "{controller=Home}/{action=Index}/{id?}");
            });
        }

        
    }
}
