﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Resonance.Core;
using Resonance.Core.Attributes;
using Resonance.Core.Helpers.AssetHelpers;
using Resonance.Core.Helpers.AuthHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.FileHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.AuthModels;
using Resonance.Core.Models.ServiceModels.ActivityLoggerService;
using Resonance.Core.Models.ServiceModels.AtlasModels;
using Resonance.Core.Services.ActivityLoggerService;

namespace Resonance.Api.Microservices.Atlas.Controllers
{
    [Route("assets")]
    [RequestSizeLimit(100000000)]
    [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
    public class AssetController : Controller
    {
        private readonly IActivityLoggerService _activityLoggerService;

        public AssetController(IActivityLoggerService activityLoggerService)
        {
            _activityLoggerService = activityLoggerService;
        }

        /* Decomissioned 2020-05-26 by Nathan, delete after 2020-06-26. Edit product now manages this flow
        [HttpPost("upload-product")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.CanManageProducts)]
        ////[ValidateAntiForgeryToken()] // Enable once Anchor is ready - enabling this will break the asset uploader tester
        public JsonResult UploadProductImageAsset(IFormCollection form)
        {        
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };
            try
            {
                stopwatch.Start();
                int productID = 0;
                bool productIDCorrectlyFormatted = false;
                if (form.Keys.Contains("product-id"))
                {
                    productIDCorrectlyFormatted = int.TryParse(form["product-id"], out productID);
                }

                if (productIDCorrectlyFormatted)
                {
                    try
                    {
                        response.Success = FileUploadHelper.AtlasUploadFile(productID, ref response, form, HttpContext);

                        if (!response.Success)
                        {
                            Response.StatusCode = 400;
                        }
                    }
                    catch(Exception ex)
                    {
                        Log.Error(ex);
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error($@"{ex}", location: "assets/upload-product-image", context: HttpContext);
                response.ErrorMessage = "Failed to upload product image";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }
        */

        [HttpGet("get-product-assets/{productID}")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        public JsonResult GetProductImageAssets(int productID)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiListResponse<ProductAssetResponseData>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };
            try
            {
                stopwatch.Start();
                var assets = AssetHelper.GetAssetsByProductID(productID, HttpContext);
                response.Success = assets != null;
                Response.StatusCode =
                    assets != null
                    ? assets.Count() > 0
                        ? 200
                        : 404
                    : 400;
                response.ResponseData = assets ?? new List<ProductAssetResponseData>();
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                Response.StatusCode = 500;
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpGet("get-product-assets/{productID}/{assetLocation}")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        public JsonResult GetProductImageAssetsByAssetLocation(int productID, string assetLocation)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<ProductAssetResponseData>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load"
                }
            };
            try
            {
                var asset = AssetHelper.GetAssetsByProductIDAndLocation(productID, assetLocation, HttpContext);
                response.Success = asset != null;
                Response.StatusCode =
                    asset != null
                    ? 200
                    : 404;
                response.ResponseData = asset;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                Response.StatusCode = 500;
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpPost("send-asset-to-twitch-upload-service")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.CanManageOwnPremiumContentCreators)]
        public JsonResult SendAssetToTwitchUploadService(IFormCollection form)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<ProductAssetResponseData>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            try
            {
                /*
                var asset = AssetHelper.GetAssetsByProductIDAndLocation(productID, assetLocation);
                response.Success = asset != null;
                Response.StatusCode =
                    asset != null
                    ? 200
                    : 404;
                response.ResponseData = asset;
                */
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                Response.StatusCode = 500;
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpGet("get-product-image-restrictions")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        public JsonResult GetProductImageRestrictions()
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<Dictionary<string, ProductImageRestriction>>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            try
            {
                stopwatch.Start();
                response.ResponseData = AssetHelper.GetImageSlotRestrictions();
                response.Success = true;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                Response.StatusCode = 500;
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        private void LogUploadActivity(string action, string filename)
        {
            string user = "Unknown User";
            string message = "";
            AuthTokenData tokenData = (AuthTokenData)HttpContext.Items[UserAuthDataContext.AuthTokenDataKey];
            if (tokenData != null && tokenData.User != null)
            {
                user = tokenData.User;
            }
            
            message = string.Format("{0} Uploaded file {1} via {2}", user, filename, action);
            
            _activityLoggerService.LogActivity(new ActivityLogData() { Action = action, Controller = "AssetController", Timestamp = DateTime.UtcNow, User = user, Message = message });
        }
    }    
}