﻿using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Configuration;
using Resonance.Core;
using Resonance.Core.Attributes;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.ServiceModels.AtlasModels;
using Resonance.Core.Services.ActivityLoggerService;
using Resonance.Microservices.Methods;
using System;
using Resonance.Core.Models.AuthModels;
using Resonance.Core.Models.FilterModels;
using Resonance.Core.Services.ColumnFilterService;
using Microsoft.AspNetCore.Mvc.Infrastructure;
using Resonance.Core.Services.LocalizationService;
using Resonance.Core.Services.LdapService;
using Resonance.Core.Models.DatabaseModels.AtlasModels;
using System.Collections.Generic;
using Resonance.Core.Models.ApiModels.AtlasModels;
using Resonance.Microservices.ListingHelpers;
using Resonance.Core.Models.DatabaseModels.RequestModels;
using Resonance.Core.Models.ServiceModels.ActivityLoggerService;
using System.Diagnostics;
using System.Collections.Concurrent;
using Resonance.Core.Models.ApiModels.RequestModels;
using Resonance.Core.Helpers.ApiHelpers;
using System.Text.RegularExpressions;
using Resonance.Core.Helpers.FileHelpers;
using Resonance.Core.Exceptions;
using Resonance.Core.Helpers.ErrorHelpers;
using Resonance.Core.Helpers.AuthHelpers;

namespace Resonance.Api.Microservices.Atlas.Controllers
{
    [Route("event")]
    //[AutoValidateAntiforgeryToken()]
    public class EventController : Controller
    {
        private const string imagePattern = "^data:(.*)?;base64,(.*)";
        private static Regex imageRegex = new Regex(imagePattern, RegexOptions.Singleline | RegexOptions.Compiled | RegexOptions.IgnoreCase);
        private readonly IConfiguration _config;
        private readonly IActivityLoggerService _activityLoggerService;
        private IColumnFilterService _columnFilterService;
        private readonly IHttpContextAccessor _httpContextAccessor;
        private readonly IActionContextAccessor _actionContextAccessor;
        private readonly ILdapService _ldapService;
        private readonly ILocalizationService _localizationService;
        private static string controllerName = "EventController";
        private static AtlasMethods atlasMethod = new AtlasMethods();

        static EventController()
        {
            atlasMethod.Initialize();
        }

        public EventController(IHttpContextAccessor httpContextAccessor, IConfiguration config, IColumnFilterService columnFilterService, IActivityLoggerService activityLoggerService, IActionContextAccessor actionContextAccessor, ILdapService ldapService, ILocalizationService localizationService)
        {
            _config = config;
            _activityLoggerService = activityLoggerService;
            _actionContextAccessor = actionContextAccessor;
            _columnFilterService = columnFilterService;
            _ldapService = ldapService;
            _httpContextAccessor = httpContextAccessor;
            _localizationService = localizationService;
        }

        [HttpPost("get-product-list")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        //[ValidateAntiForgeryToken()]
        public JsonResult GetProductList([FromBody]ListingFilter filter)
        {
            string actionName = "GetProductList";

            var statuscode = 200;
            var stopwatch = new Stopwatch();
            ApiResponse<AtlasListing> response = ListingHelper.DefaultResponse();

            try
            {
                stopwatch.Start();
                statuscode = ListingHelper.WireUpAtlasListing(Constants.AtlasInternalEventType.Product, AtlasListingDefaults.ProductDefault(), ref response, ref filter, HttpContext, actionName, controllerName, _columnFilterService, _activityLoggerService, _actionContextAccessor, response.Metrics.ChildrenMetrics);
                response.Success = statuscode == 200;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetProductList";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for {controllerName}.{actionName}.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
         }

        /// <summary>
        /// Todo
        /// </summary>
        [HttpPost("edit-product")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.CanManageOwnPremiumContentCreators)]
        //[ValidateAntiForgeryToken()]
        public JsonResult EditProduct([FromBody]ProductEditModal data)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();
            var statuscode = 200;
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };

            try
            {
                if(data == null)
                {
                    statuscode = 400;
                    response.ErrorMessage = "Data is empty";
                }
                else
                {
                    UserAuthData tokenData = (UserAuthData)HttpContext.Items[UserAuthDataContext.UserAuthDataKey];                                        
                    var result = atlasMethod.EditProduct(tokenData, data, response.Metrics.ChildrenMetrics, HttpContext);
                    Response.StatusCode = result.Item1;
                    response.Success = Response.StatusCode == 200 ? true : false;
                    if (response.Success)
                    {
                        LogEditActivity(nameof(EditProduct), "Product", data.ProductID.ToString());
                    }
                    if (response.Success && data.FrontPageBanner  != null && data.FrontPageBanner.Length > 0 && result.Item2 > 0)
                    {
                        try
                        {
                            if (imageRegex.IsMatch(data.FrontPageBanner))
                            {
                                var match = imageRegex.Match(data.FrontPageBanner);
                                var contentType = match.Groups[1].Captures[0].Value;
                                var imageData = match.Groups[2].Captures[0].Value;
                                var fileResult = FileUploadHelper.AtlasUploadFile(result.Item2, "front-page-promotional-banner-slot", contentType, contentType, ref response, imageData, HttpContext);
                            }
                        }
                        catch(Exception ex)
                        {
                            Log.Error(ex, context: HttpContext);
                        }
                    }
                }
            }
            catch(WebRequestException ex)
            {
                // Set status code to 400 only if the status code hasn't already been changed
                if(statuscode == 200 && Response.StatusCode == 200)
                {
                    statuscode = 400;
                }
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: {ex.Message}";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            catch(Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error EditProduct";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("get-season-list")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        //[ValidateAntiForgeryToken()]
        public JsonResult GetSeasonList([FromBody]ListingFilter filter)
        {
            string actionName = "GetSeasonList";
            var stopwatch = new Stopwatch();
            var statuscode = 200;
            ApiResponse<AtlasListing> response = ListingHelper.DefaultResponse();

            try
            {
                stopwatch.Start();
                statuscode = ListingHelper.WireUpAtlasListing(Constants.AtlasInternalEventType.Season, AtlasListingDefaults.SeasonDefault(), ref response, ref filter, HttpContext, actionName, controllerName, _columnFilterService, _activityLoggerService, _actionContextAccessor, response.Metrics.ChildrenMetrics);
                response.Success = statuscode == 200;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetSeasonList";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for {controllerName}.{actionName}.", context: HttpContext);
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        [HttpPost("game-auto-complete")]
        public JsonResult GameAutoComplete([FromBody] RequestUserSearch userSearch)
        {
            var statuscode = 200;
            var stopwatch = new Stopwatch();
            var response = new ApiListResponse<string>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };

            try
            {
                stopwatch.Start();
                var result = atlasMethod.SearchGames(HttpContext, userSearch.LoginNameSearch, response.Metrics.ChildrenMetrics);
                statuscode = result.Item1;
                if (result.Item2 != null && result.Item1 == 200)
                {
                    response.Success = true;
                    response.ResponseData = result.Item2;
                }
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
                response.ErrorMessage = $@"{errorcode}: Unhandled error GameAutoComplete";
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GameAutoComplete.", context: HttpContext);
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        [HttpPost("user-auto-complete")]
        public JsonResult UserAutoComplete([FromBody] RequestUserSearch userSearch)
        {
            var stopwatch = new Stopwatch();
            var statuscode = 200;
            var response = new ApiListResponse<RequestUserSearchUser>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };

            try
            {
                stopwatch.Start();
                var result = atlasMethod.SearchUsers(HttpContext, userSearch.LoginNameSearch, response.Metrics.ChildrenMetrics);
                statuscode = result.Item1;
                if (result != null && result.Item2 != null)
                {
                    response.Success = true;
                    response.ResponseData = result.Item2;
                }
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetProduRequestUserAutoCompletectList";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for RequestUserAutoComplete.", context: HttpContext);
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("get-managed-creators")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        public JsonResult GetManagedCreators()
        {
            var stopwatch = new Stopwatch();
            var statuscode = 200;
            var response = new ApiResponse<GetManagedCreatorsResult>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };

            try
            {
                stopwatch.Stop();
                UserAuthData tokenData = (UserAuthData)HttpContext.Items[UserAuthDataContext.UserAuthDataKey];
                string amToCheck = tokenData.UserID;
                var result = atlasMethod.GetOwnerCreators(HttpContext, amToCheck, response.Metrics.ChildrenMetrics);
                statuscode = result.Item1;
                response.ResponseData = result.Item2;
                response.Success = statuscode == 200;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetManagedCreators";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("get-managed-channels/{pccID}")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        public JsonResult GetManagedChannels(int pccID)
        {
            var stopwatch = new Stopwatch();
            var statuscode = 200;
            var response = new ApiResponse<GroupedOwnedChannelList>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };

            try
            {
                stopwatch.Stop();
                var result = atlasMethod.GetOwnedChannels(HttpContext, pccID, response.Metrics.ChildrenMetrics);
                statuscode = result.Item1;
                response.ResponseData = result.Item2;
                response.Success = statuscode == 200;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetManagedChannels";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("edit-season")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.CanManageOwnPremiumContentCreators)]
        //[ValidateAntiForgeryToken()]
        public JsonResult EditSeason([FromBody]AtlasSeason data)
        {
            var stopwatch = new Stopwatch();
            var statuscode = 200;
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };

            try
            {
                stopwatch.Start();
                if (data == null)
                {
                    Response.StatusCode = 400;
                }
                else
                {
                    UserAuthData tokenData = (UserAuthData)HttpContext.Items[UserAuthDataContext.UserAuthDataKey];
                    Response.StatusCode = atlasMethod.EditSeason(tokenData, data, response.Metrics.ChildrenMetrics, HttpContext);
                    response.Success = Response.StatusCode == 200 ? true : false;
                }
            }
            catch(WebRequestException ex)
            {
                if(Response.StatusCode == 200 && statuscode == 200)
                {
                    statuscode = 400;
                }
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: {ex.Message}";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error EditSeason";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("get-event-list")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        //[ValidateAntiForgeryToken()]
        public JsonResult GetEventList([FromBody]ListingFilter filter)
        {
            string actionName = "GetEventList";
            var stopwatch = new Stopwatch();
            ApiResponse<AtlasListing> response = ListingHelper.DefaultResponse();
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                statuscode = ListingHelper.WireUpAtlasListing(Constants.AtlasInternalEventType.Event, AtlasListingDefaults.EventDefault(), ref response, ref filter, HttpContext, actionName, controllerName, _columnFilterService, _activityLoggerService, _actionContextAccessor, response.Metrics.ChildrenMetrics);
                response.Success = statuscode == 200;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetEventList";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for {controllerName}.{actionName}.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            
            return new JsonResult(response);
        }

        [HttpPost("edit-event")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.CanManageOwnPremiumContentCreators)]
        //[ValidateAntiForgeryToken()]
        public JsonResult EditEvent([FromBody]AtlasEvent data)
        {
            string requestor = "Unknown";
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                if (data == null)
                {
                    Response.StatusCode = 400;
                }
                else
                {
                    UserAuthData tokenData = (UserAuthData)HttpContext.Items[UserAuthDataContext.UserAuthDataKey];
                    requestor = tokenData.UserID;
                    Response.StatusCode = atlasMethod.EditEvent(tokenData, data, response.Metrics.ChildrenMetrics, HttpContext);
                    response.Success = Response.StatusCode == 200 ? true : false;
                    if (response.Success)
                    {
                        LogEditActivity(nameof(EditEvent), "Event", data.EventID.ToString());
                    }
                }
            }
            catch(WebRequestException ex)
            {
                if(Response.StatusCode == 200 && statuscode == 200)
                {
                    statuscode = 400;
                }
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: {ex.Message}";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error EditEvent";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
                var datanull = data == null;
                if(Constants.AppConfig.Application.Environment == "Staging")
                {
                    Log.Info($@"Edit Streams With Channels Complete, Requestor: {requestor}, Data null? {datanull}, Channel Count: {data?.ChannelCount}, PCC ID: {data?.PremiumContentCreatorID}, Product ID: {data?.ProductID}, Event ID: {data?.EventID}");
                }
            }

            return new JsonResult(response);
        }

        [HttpPost("edit-event-with-streams")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.CanManageOwnPremiumContentCreators)]
        //[ValidateAntiForgeryToken()]
        public JsonResult EditEventWithStreams([FromBody]AtlasEventWithStreamList data)
        {
            
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;
            try
            {
                if (data == null)
                {
                    statuscode = 400;
                    response.ErrorMessage = "Request Data is null";
                }
                else
                {
                    UserAuthData tokenData = (UserAuthData)HttpContext.Items[UserAuthDataContext.UserAuthDataKey];                    
                    Response.StatusCode = atlasMethod.EditEventWithStreams(tokenData, data, response.Metrics.ChildrenMetrics, HttpContext);
                    response.Success = Response.StatusCode == 200 ? true : false;
                    if (response.Success)
                    {
                        LogEditActivity(nameof(EditEventWithStreams), "Event With Streams", data.Event.EventID.ToString());
                    }
                }
            }
            catch(WebRequestException ex)
            {
                if(statuscode == 200 && Response.StatusCode == 200)
                {
                    statuscode = 400;
                }
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: {ex.Message}";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error EditEventWithStreams";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        [HttpPost("get-season-event-list")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        public JsonResult GetSeasonEventList([FromBody]ListingFilter filter)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<SeasonEventList>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                var result = atlasMethod.GetSeasonEventList(ref filter, response.Metrics.ChildrenMetrics, HttpContext);
                statuscode = result.Item1;
                response.ResponseData = result.Item2;
                response.Success = true;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetSeasonEventList";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetSeasonEventList.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpPost("get-channel-event-list")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        public JsonResult GetChannelEventList([FromBody]ListingFilter filter)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<ChannelGroupedEventList>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                var result = atlasMethod.GetChannelEventList(ref filter, response.Metrics.ChildrenMetrics, HttpContext);
                statuscode = result.Item1;
                response.ResponseData = result.Item2;
                response.Success = statuscode == 200;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetChannelEventList";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetChannelEventList.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpPost("get-topic-event-list")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        public JsonResult GetTopicEventList([FromBody]ListingFilter filter)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<GroupedEventList>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                var result = atlasMethod.GetTopicEventList(HttpContext, ref filter, response.Metrics.ChildrenMetrics);
                statuscode = result.Item1;
                response.ResponseData = result.Item2;
                response.Success = true;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetTopicEventList";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetTopicEventList.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpPost("get-format-event-list")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        public JsonResult GetFormatEventList([FromBody]ListingFilter filter)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<GroupedEventList>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                var result = atlasMethod.GetFormatEventList(HttpContext, ref filter, response.Metrics.ChildrenMetrics);
                statuscode = result.Item1;
                response.ResponseData = result.Item2;
                response.Success = statuscode == 200;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetFormatEventList";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetFormatEventList.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpPost("get-category-event-list")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        public JsonResult GetCategoryEventList([FromBody]ListingFilter filter)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<CategoryGroupedEventList>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                var result = atlasMethod.GetCategoryEventList(ref filter, response.Metrics.ChildrenMetrics, HttpContext);
                statuscode = result.Item1;
                response.ResponseData = result.Item2;
                response.Success = statuscode == 200;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetCategoryEventList";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetChannelEventList.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpPost("get-stream-list")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        //[ValidateAntiForgeryToken()]
        public JsonResult GetStreamList([FromBody]ListingFilter filter)
        {
            string actionName = "GetStreamList";
            var stopwatch = new Stopwatch();
            ApiResponse<AtlasListing> response = ListingHelper.DefaultResponse();
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                statuscode = ListingHelper.WireUpAtlasListing(Constants.AtlasInternalEventType.Stream, AtlasListingDefaults.StreamDefault(), ref response, ref filter, HttpContext, actionName, controllerName, _columnFilterService, _activityLoggerService, _actionContextAccessor, response.Metrics.ChildrenMetrics);
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetStreamList";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for {controllerName}.{actionName}.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("edit-stream")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.Administrator)]
        //[ValidateAntiForgeryToken()]
        public JsonResult EditStream([FromBody]AtlasStream data)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;

            try
            {
                stopwatch.Start();
                if (data == null)
                {
                    Response.StatusCode = 400;
                }
                else
                {
                    UserAuthData tokenData = (UserAuthData)HttpContext.Items[UserAuthDataContext.UserAuthDataKey];
                    statuscode = atlasMethod.EditStream(HttpContext, tokenData, data, response.Metrics.ChildrenMetrics);
                    response.Success = Response.StatusCode == 200 ? true : false;
                    if (response.Success)
                    {
                        LogEditActivity(nameof(EditStream), "Stream", data.StreamID.ToString());
                    }
                }
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error EditStream";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpGet("get-event-types")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        public JsonResult GetEventTypes()
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<Dictionary<string, int>>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;

            try
            {
                stopwatch.Start();
                response.ResponseData = new Dictionary<string, int>()
                {
                    { Constants.AtlasInternalEventType.Product.ToString(), (int)Constants.AtlasInternalEventType.Product },
                    { Constants.AtlasInternalEventType.Season.ToString(), (int)Constants.AtlasInternalEventType.Season },
                    { Constants.AtlasInternalEventType.Event.ToString(), (int)Constants.AtlasInternalEventType.Event },
                    { Constants.AtlasInternalEventType.Stream.ToString(), (int)Constants.AtlasInternalEventType.Stream },
                    { Constants.AtlasEventChannelType.CoStreamAuthorized.ToString(), (int)Constants.AtlasEventChannelType.CoStreamAuthorized},
                    { Constants.AtlasEventChannelType.UGCParticipant.ToString(), (int)Constants.AtlasEventChannelType.UGCParticipant},
                    { Constants.AtlasEventChannelType.CoStreamOrganic.ToString(), (int)Constants.AtlasEventChannelType.CoStreamOrganic},
                };
                response.Success = true;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetEventTypes";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetEventTypes.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpGet("get-topics-formats")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        public JsonResult GetTopicsFormats()
        {
            var stopwatch = new Stopwatch();
            var response = new ApiListResponse<TopicListEntry>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                var result = atlasMethod.GetTopicsFormats(response.Metrics.ChildrenMetrics, HttpContext);
                statuscode = result.Item1;
                response.ResponseData = result.Item2;
                response.Success = true;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetTopicsFormats";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetTopicsFormats.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("get-premium-content-creator-list")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        //[ValidateAntiForgeryToken()]
        public JsonResult GetPremiumContentCreator([FromBody]ListingFilter filter)
        {
            string actionName = "GetPremiumContentCreator";
            var stopwatch = new Stopwatch();
            ApiResponse<AtlasListing> response = ListingHelper.DefaultResponse();
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                statuscode = ListingHelper.WireUpAtlasListing(Constants.AtlasInternalEventType.PremiumContentCreator, AtlasListingDefaults.PremiumContentCreatorDefault(), ref response, ref filter, HttpContext, actionName, controllerName, _columnFilterService, _activityLoggerService, _actionContextAccessor, response.Metrics.ChildrenMetrics);
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetPremiumContentCreator";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for {controllerName}.{actionName}.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("edit-premium-content-creator")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.Administrator)]
        //[ValidateAntiForgeryToken()]
        public JsonResult EditPremiumContentCreator([FromBody]PremiumContentCreator data)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<int?>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                if (data == null)
                {
                    statuscode = 400;
                }
                else
                {
                    UserAuthData tokenData = (UserAuthData)HttpContext.Items[UserAuthDataContext.UserAuthDataKey];
                    var result = atlasMethod.EditPremiumContentCreator(tokenData, data, response.Metrics.ChildrenMetrics, HttpContext);
                    Response.StatusCode = result.Item1;
                    response.ResponseData = result.Item2;
                    response.Success = Response.StatusCode == 200 ? true : false;
                    if (response.Success)
                    {
                        LogEditActivity(nameof(EditPremiumContentCreator), "Premium Content Creator", data.PremiumContentCreatorID.ToString());
                    }
                }
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error EditPremiumContentCreator";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpPost("edit-premium-content-creator-with-channels")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.CanManageOwnPremiumContentCreators)]
        //[ValidateAntiForgeryToken()]
        public JsonResult EditPremiumContentCreatorWithChannels([FromBody]CreatorWithStreamList data)
        {
            
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<int?>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                if (data == null)
                {
                    statuscode = 400;
                }
                else
                {
                    UserAuthData tokenData = (UserAuthData)HttpContext.Items[UserAuthDataContext.UserAuthDataKey];
                    var result = atlasMethod.EditPremiumContentCreatorWithChannels(tokenData, data, response.Metrics.ChildrenMetrics, HttpContext);
                    Response.StatusCode = result.Item1;
                    response.ResponseData = result.Item2;
                    response.Success = Response.StatusCode == 200 ? true : false;
                    if (response.Success)
                    {
                        LogEditActivity(nameof(EditPremiumContentCreator), "Premium Content Creator", data.Creator.PremiumContentCreatorID.ToString());
                    }
                }
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error EditPremiumContentCreatorWithChannels";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpPost("get-account-manager-list")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        //[ValidateAntiForgeryToken()]
        public JsonResult GetAccountManager([FromBody]ListingFilter filter)
        {
            string actionName = "GetAccountManager";
            var stopwatch = new Stopwatch();
            ApiResponse<AtlasListing> response = ListingHelper.DefaultResponse();
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                statuscode = ListingHelper.WireUpAtlasListing(Constants.AtlasInternalEventType.AccountManager, AtlasListingDefaults.AccountManagerDefault(), ref response, ref filter, HttpContext, actionName, controllerName, _columnFilterService, _activityLoggerService, _actionContextAccessor, response.Metrics.ChildrenMetrics);
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetAccountManager";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for {controllerName}.{actionName}.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("edit-account-manager")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.Administrator)]
        //[ValidateAntiForgeryToken()]
        public JsonResult EditAccountManager([FromBody]AtlasAccountManager data)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;

            try
            {
                stopwatch.Start();
                if (data == null)
                {
                    statuscode = 400;
                    response.ErrorMessage = "Data is null";
                }
                else
                {
                    UserAuthData tokenData = (UserAuthData)HttpContext.Items[UserAuthDataContext.UserAuthDataKey];
                    statuscode = atlasMethod.EditAccountManager(HttpContext, tokenData, data, response.Metrics.ChildrenMetrics);
                    response.Success = statuscode == 200 ? true : false;
                    if (response.Success)
                    {
                        LogEditActivity(nameof(EditAccountManager), "Account Manager", data.AccountManagerID.ToString());
                    }
                }
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error EditAccountManager";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpPost("get-pcc-am-map-list")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        //[ValidateAntiForgeryToken()]
        public JsonResult GetPccAmMap([FromBody]ListingFilter filter)
        {
            string actionName = "GetPccAmMap";
            var stopwatch = new Stopwatch();
            ApiResponse<AtlasListing> response = ListingHelper.DefaultResponse();
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                statuscode = ListingHelper.WireUpAtlasListing(Constants.AtlasInternalEventType.PccToAmMap, AtlasListingDefaults.PccToAmMapDefault(), ref response, ref filter, HttpContext, actionName, controllerName, _columnFilterService, _activityLoggerService, _actionContextAccessor, response.Metrics.ChildrenMetrics);
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetPccAmMap";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for {controllerName}.{actionName}.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("edit-pcc-am-map")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.CanManageOwnPremiumContentCreators)]
        //[ValidateAntiForgeryToken()]
        public JsonResult EditPccAmMap([FromBody]PremiumContentCreatorToAccountManagerMap data)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                if (data == null)
                {
                    Response.StatusCode = 400;
                }
                else
                {
                    UserAuthData tokenData = (UserAuthData)HttpContext.Items[UserAuthDataContext.UserAuthDataKey];                                        
                    Response.StatusCode = atlasMethod.EditPremiumContentCreatorAccountManager(tokenData, data, response.Metrics.ChildrenMetrics, HttpContext);
                    response.Success = Response.StatusCode == 200 ? true : false;
                    if (response.Success)
                    {
                        LogEditActivity(nameof(EditPccAmMap), "PccAmMap", data.PccToAmMapID.ToString());
                    }
                }
            }
            catch(WebRequestException ex)
            {
                if(Response.StatusCode == 200 && statuscode == 200)
                {
                    statuscode = 400;
                }
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: {ex.Message}";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetSeasonEventList";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpPost("get-pcc-channel-map-list")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        //[ValidateAntiForgeryToken()]
        public JsonResult GetPccChannelMap([FromBody]ListingFilter filter)
        {
            string actionName = "GetPccChannelMap";
            var stopwatch = new Stopwatch();
            ApiResponse<AtlasListing> response = ListingHelper.DefaultResponse();
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                statuscode = ListingHelper.WireUpAtlasListing(Constants.AtlasInternalEventType.PccToChannelMap, AtlasListingDefaults.PccToChannelMapDefault(), ref response, ref filter, HttpContext, actionName, controllerName, _columnFilterService, _activityLoggerService, _actionContextAccessor, response.Metrics.ChildrenMetrics);
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetPccChannelMap";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for {controllerName}.{actionName}.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("edit-pcc-channel-map")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.CanManageOwnPremiumContentCreators)]
        //[ValidateAntiForgeryToken()]
        public JsonResult EditPccChannelMap([FromBody]PremiumContentCreatorToChannelMap data)
        {
            
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                if (data == null)
                {
                    statuscode = 400;
                    response.ErrorMessage = "Data is null";
                }
                else
                {
                    UserAuthData tokenData = (UserAuthData)HttpContext.Items[UserAuthDataContext.UserAuthDataKey];
                    statuscode = atlasMethod.EditPremiumContentCreatorChannels(tokenData, data, response.Metrics.ChildrenMetrics, HttpContext);
                    response.Success = statuscode == 200 ? true : false;
                    if (response.Success)
                    {
                        LogEditActivity(nameof(EditPccChannelMap), "PccChannelMap", data.PccChannelMapId.ToString());
                    }
                }
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error EditPccChannelMap";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpGet("get-contracts-by-pcc")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        public JsonResult GetContractsByPCC()
        {
            string actionName = "GetContractsByPCC";
            var stopwatch = new Stopwatch();
            
            ApiResponse<IList<ContractsByPCC>> response = new ApiResponse<IList<ContractsByPCC>>
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };

            try
            {
                stopwatch.Start();
                response.ResponseData = atlasMethod.GetContractsByPremiumContentCreator(HttpContext);
                response.Success = true;
            }
            catch (Exception ex)
            {
                Response.StatusCode = 500;
                Log.Error(ex);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for {controllerName}.{actionName}.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("get-contract-list")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        public JsonResult GetContractList([FromBody]ListingFilter filter)
        {
            string actionName = "GetContractList";
            var stopwatch = new Stopwatch();
            ApiResponse<AtlasListing> response = ListingHelper.DefaultResponse();
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                statuscode = ListingHelper.WireUpAtlasListing(Constants.AtlasInternalEventType.Contract, AtlasListingDefaults.ContractDefault(), ref response, ref filter, HttpContext, actionName, controllerName, _columnFilterService, _activityLoggerService, _actionContextAccessor, response.Metrics.ChildrenMetrics, requireOwnershipVerification: true);
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetContractList";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for {controllerName}.{actionName}.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("get-contract-list-with-channels")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        public JsonResult GetContractListWithChannels([FromBody]ListingFilter filter)
        {
            string actionName = "GetContractListWithChannels";
            var stopwatch = new Stopwatch();
            ApiResponse<AtlasListing> response = ListingHelper.DefaultResponse();
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                statuscode = ListingHelper.WireUpAtlasListing(Constants.AtlasInternalEventType.Contract, AtlasListingDefaults.ContractDefault(), ref response, ref filter, HttpContext, actionName, controllerName, _columnFilterService, _activityLoggerService, _actionContextAccessor, response.Metrics.ChildrenMetrics, requireOwnershipVerification: true);
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetContractListWithChannels";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for {controllerName}.{actionName}.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("edit-contract")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.CanManageOwnPremiumContentCreators)]
        //[ValidateAntiForgeryToken()]
        public JsonResult EditContract([FromBody]AtlasContract data)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                if (data == null)
                {
                    statuscode = 400;
                }
                else
                {   
                    UserAuthData tokenData = (UserAuthData)HttpContext.Items[UserAuthDataContext.UserAuthDataKey];
                    statuscode = atlasMethod.EditContract(tokenData, data, response.Metrics.ChildrenMetrics, HttpContext);
                    response.Success = statuscode == 200 ? true : false;
                    if (response.Success)
                    {
                        LogEditActivity(nameof(EditContract), "Contract", data.ContractID.ToString());
                    }
                }
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error EditContract";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpPost("edit-contract-with-channels")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.CanManageOwnPremiumContentCreators)]
        //[ValidateAntiForgeryToken()]
        public JsonResult EditContractWithChannels([FromBody]AtlasContractWithChannelList data)
        {
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var stopwatch = new Stopwatch();
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                if (data == null)
                {
                    statuscode = 400;
                    response.ErrorMessage = "Data is null";
                }
                else
                {                    
                    UserAuthData tokenData = (UserAuthData)HttpContext.Items[UserAuthDataContext.UserAuthDataKey];
                    statuscode = atlasMethod.EditContractWithChannels(tokenData, data, response.Metrics.ChildrenMetrics, HttpContext).Item1;
                    response.Success = statuscode == 200 ? true : false;
                    if (response.Success)
                    {
                        LogEditActivity(nameof(EditContractWithChannels), "Contract With Channels", data.Contract.ContractID.ToString());
                    }
                }
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error EditContractWithChannels";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpPost("get-contract-channel-list")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        public JsonResult GetContractChannelList([FromBody]ListingFilter filter)
        {
            string actionName = "GetContractChannelList";
            var stopwatch = new Stopwatch();
            ApiResponse<AtlasListing> response = ListingHelper.DefaultResponse();
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                statuscode = ListingHelper.WireUpAtlasListing(Constants.AtlasInternalEventType.ContractChannel, AtlasListingDefaults.ContractChannelDefault(), ref response, ref filter, HttpContext, actionName, controllerName, _columnFilterService, _activityLoggerService, _actionContextAccessor, response.Metrics.ChildrenMetrics, requireOwnershipVerification: true);
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetContractChannelList";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for {controllerName}.{actionName}.");
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("edit-contract-channel")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.Administrator)]
        //[ValidateAntiForgeryToken()]
        public JsonResult EditContractChannel([FromBody] Core.Models.DatabaseModels.AtlasModels.AtlasContractChannel data)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };

            try
            {
                stopwatch.Start();
                if (data == null)
                {
                    Response.StatusCode = 400;
                }
                else
                {   
                    UserAuthData tokenData = (UserAuthData)HttpContext.Items[UserAuthDataContext.UserAuthDataKey];
                    Response.StatusCode = atlasMethod.EditContractChannels(HttpContext, tokenData, data, response.Metrics.ChildrenMetrics);
                    response.Success = Response.StatusCode == 200 ? true : false;
                    if (response.Success)
                    {
                        LogEditActivity(nameof(EditContractChannel), "Contract Channel", data.ChannelContractID.ToString());
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpGet("get-active-calendar-events/{pccID}")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        //[ValidateAntiForgeryToken()]
        public JsonResult GetActiveCalendarEvents(int pccID)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<CalendarEventListByDate>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                var events = atlasMethod.GetActiveEventsForCalendar(HttpContext, pccID);
                statuscode = events.Item1;
                response.ResponseData = events.Item2;
                response.Success = statuscode == 200;
            }
            catch(Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetActiveCalendarEvents";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpPost("edit-contract-account-manager-map")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.Administrator)]
        //[ValidateAntiForgeryToken()]
        public JsonResult EditContractAccountManagerMap([FromBody] AtlasContractAccountManagerMap data)
        {
            
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                if (data == null)
                {
                    statuscode = 400;
                }
                else
                {
                    UserAuthData tokenData = (UserAuthData)HttpContext.Items[UserAuthDataContext.UserAuthDataKey];
                    statuscode = atlasMethod.EditContractAccountManagerMap(HttpContext, tokenData, data, response.Metrics.ChildrenMetrics);
                    response.Success = statuscode == 200 ? true : false;
                    if (response.Success)
                    {
                        LogEditActivity(nameof(EditContractAccountManagerMap), "Contract Account Manager Map", data.ContractAccountManagerMapID.ToString());
                    }
                }
            }
            catch (WebRequestException ex)
            {
                if(Response.StatusCode == 200 && statuscode == 200)
                {
                    statuscode = 400;
                }
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: {ex.Message}";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error EditContractAccountManagerMap";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }
            return new JsonResult(response);
        }

        [HttpPost("get-aggregate-pcc-stats")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        //[ValidateAntiForgeryToken()]
        public JsonResult GetAggregatePccStats([FromBody]AggregatePccStatsRequest data)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<IEnumerable<PccStats>>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;
            try
            {
                stopwatch.Start();
                if(data != null && data.PccIDs != null && data.PccIDs.Length > 0)
                {
                    var result = atlasMethod.AggregateProductStats(HttpContext, data.PccIDs);
                    statuscode = result.Item1;
                    response.ResponseData = result.Item2;
                }
                else
                {
                    statuscode = 200;
                    response.ResponseData = null;
                }
                response.Success = statuscode == 200;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetAggregatePccStats";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpPost("get-channel-meta-data")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        //[ValidateAntiForgeryToken()]
        public JsonResult GetChannelMetaData([FromBody]ChannelMetaDataRequest data)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<IEnumerable<ChannelMetaData>>()            
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;

            try
            {
                stopwatch.Start();
                var result = atlasMethod.GetChannelMetaDataForEvent(HttpContext, data.EventID);
                statuscode = result.Item1;
                response.ResponseData = result.Item2;
                response.Success = statuscode == 200;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetChannelMetaData";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpGet("get-pcc-channel-data/{pccID}")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        //[ValidateAntiForgeryToken()]
        public JsonResult GetPccChannelData(int pccID)
        {
            var stopwatch = new Stopwatch();
            var response = new ApiResponse<IEnumerable<PccChannelData>>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };
            var statuscode = 200;

            try
            {
                stopwatch.Start();
                var result = atlasMethod.GetPccChannelData(HttpContext, pccID);
                statuscode = result.Item1;
                response.ResponseData = result.Item2;
                response.Success = statuscode == 200;
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetPccChannelData";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }        

        [HttpGet("get-event-csv-by-pcc/{pccID}")]
        [ResonanceAuth("Atlas", requiredPermissions: ConstantsPermissions.Atlas.GeneralAccess)]
        //[ValidateAntiForgeryToken()]
        public JsonResult GetEventCsvByPcc(int pccID)
        {
            var stopwatch = new Stopwatch();            
            var response = new ApiResponse<string>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };

            try
            {
                stopwatch.Start();
                response.ResponseData = atlasMethod.GetEventCsvByPcc(pccID);
                response.Success = true;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception while processing GetEventCsvByPcc";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        private void LogEditActivity(string action, string item, string editItemId)
        {
            string user = "Unknown User";
            string message = "";
            UserAuthData tokenData = (UserAuthData)HttpContext.Items[UserAuthDataContext.UserAuthDataKey];
            if (tokenData != null && tokenData.UserID != null)
            {
                user = tokenData.UserID;                
            }

            message = string.Format("{0} edited {1} with id {2}", user, item, editItemId);

            _activityLoggerService.LogActivity(new ActivityLogData() { Action = action, Controller = "EventController", Timestamp = DateTime.UtcNow, User = user, Message = message });
        }
    }
}
