﻿using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Configuration;
using System;
using Newtonsoft.Json;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.AuthModels;
using Resonance.Core.Helpers.ApiHelpers;
using Microsoft.Extensions.Caching.Memory;
using System.Collections.Generic;
using System.Threading.Tasks;
using System.IdentityModel.Tokens.Jwt;
using Resonance.Core.Services.UserGroupSyncerService;
using Resonance.Microservices.Methods;
using System.Diagnostics;
using Resonance.Core.Services.ActivityLoggerService;
using Resonance.Core.Models.ApiModels;
using Resonance.Core;
using Resonance.Core.Models.ConfigurationModels;
using System.Linq;
using Resonance.Core.Helpers.ConfigurationHelpers;

namespace Resonance.Api.Microservices.Auth.Controllers
{
    public class AuthController : Controller
    {
        private readonly IConfiguration _config;
        private IUserGroupSyncerService _ldapService;
        private readonly IActivityLoggerService _activityLoggerService;
        private static AuthMethods authMethods { get; set; }

        static AuthController()
        {
            authMethods = new AuthMethods();
            authMethods.Initialize();
        }

        public AuthController(IConfiguration config, IMemoryCache cache, IUserGroupSyncerService ldapService, IActivityLoggerService activityLoggerService)
        {   
            _config = config;
            _ldapService = ldapService;
            _activityLoggerService = activityLoggerService;
        }

        [HttpGet("worker/auth/get-auth-redirect/{siteID}")]
        public JsonResult GetAuthRedirect(string siteID)
        {
            var response = new ApiResponse<string>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                var enumResult = Enum.TryParse<Constants.SiteID>(siteID, true, out Constants.SiteID site);
                if (enumResult && site != Constants.SiteID.Unknown)
                {
                    var oauthService = AuthServiceHelper.GetOAuthHelper(site);
                    string redirectURL = oauthService.GetRedirectUrl(Constants.OAuthType.TwitchSSO);
                    response.ResponseData = redirectURL;
                    response.Success = true;
                }
                else
                {
                    Response.StatusCode = 400;
                    response.ErrorMessage = "Unknown site";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for GetAuthRedirect.");
                }
            }

            return new JsonResult(response);
        }        

        [HttpPost("worker/auth/verify-auth")]        
        public JsonResult VerifyAuth([FromBody]VerifyAuthRequest tokenRequest)
        {
            var response = new ApiResponse<JWTData>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                var enumResult = Enum.TryParse<Constants.SiteID>(tokenRequest.SiteID, true, out Constants.SiteID site);
                if (!enumResult || site == Constants.SiteID.Unknown)
                {
                    Response.StatusCode = 400;
                    response.ErrorMessage = "Specified site id was not found";
                }
                else
                {
                    var oauthService = AuthServiceHelper.GetOAuthHelper(site);
                    string finalToken = oauthService.VerifyAuthorizeToken(tokenRequest.BearerToken, HttpContext);
                    dynamic oAuthToken = JsonConvert.DeserializeObject<dynamic>(finalToken);
                    string accessToken = oAuthToken.access_token;
                    dynamic userInfo = JsonConvert.DeserializeObject<dynamic>(oauthService.GetUserInfo(accessToken, HttpContext));
                    JWTData jwt = null;
                    try
                    {
                        jwt = JWTHelper.MakeAuthJWT(tokenRequest.SiteID, userInfo, HttpContext);
                    }
                    catch (Exception ex)
                    {
                        Log.Error($"Error making jwt from user info {JsonConvert.SerializeObject(userInfo)}", context: HttpContext);
                        throw ex;
                    }
                    _activityLoggerService.LogActivity("Logged In", "AuthController", "VerifyAuth", jwt.Profile);

                    Task.Run(() => { AddUsernameToCache(jwt.Profile); });

                    response.ResponseData = jwt;
                    response.Success = true;
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for VerifyAuth.");
                }
            }
            return new JsonResult(response);
        }

        private void AddUsernameToCache(string username)
        {
            try
            {                
                _ldapService.AddUserToQueue(username);
            }
            catch (Exception e)
            {
                Log.Error(e, context: HttpContext);
            }
        }
        
        [HttpPost("worker/auth/verify-jwt/")]
        public JsonResult VerifyJWT([FromBody]VerifyJWTTokenRequest request)
        {
            var response = new ApiResponse<Dictionary<string, object>>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (request == null)
                {
                    throw new ArgumentNullException("request");
                }
                JwtPayload decryptedData = JWTHelper.VerifyJWT(request.JwtToken, HttpContext);
                OverrideGroups(decryptedData);
                
                var dict = new Dictionary<string, object>(decryptedData);
                
                response.ResponseData = dict;
                response.Success = true;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                HttpContext.Response.StatusCode = 400;
                response.ErrorMessage = "Token validation failed";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for VerifyJWT.");
                }
            }

            return new JsonResult(response);
        }

        private void OverrideGroups(JwtPayload payload)
        {
            object username = null;
            payload.TryGetValue("user", out username);

            if (username != null)
            {
                var groups = _ldapService.GetGroupsByUsername(username.ToString());

                if (groups != null)
                {
                    payload["groups"] = groups;
                }
            }
        }
    }
}
