﻿using System.Diagnostics;
using System.IO;
using Microsoft.AspNetCore.Builder;
using Microsoft.AspNetCore.CookiePolicy;
using Microsoft.AspNetCore.Hosting;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.HttpOverrides;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Caching.Memory;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.DependencyInjection;
using Microsoft.Extensions.Logging;
using Microsoft.Extensions.Options;
using NLog.Extensions.Logging;
using NLog.Web;
using Resonance.Core;
using Resonance.Core.Helpers.ConfigurationHelpers;
using Resonance.Core.Helpers.FileHelpers;
using Resonance.Core.Services.LdapService;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Middleware;
using Resonance.Core.Services.ActivityLoggerService;
using Resonance.Core.Services.UserGroupSyncerService;
using Microsoft.AspNetCore.DataProtection;

namespace Resonance.Api.WorkerNode
{
    public class Startup
    {
        public Startup(IConfiguration configuration)
        {
            Configuration = configuration;
        }

        public IConfiguration Configuration { get; }

        // This method gets called by the runtime. Use this method to add services to the container.
        public void ConfigureServices(IServiceCollection services)
        {
            services.ConfigureCorsFilesForEnvironments();

            services.Configure<CookiePolicyOptions>(options =>
            {
                // This lambda determines whether user consent for non-essential cookies is needed for a given request.
                options.CheckConsentNeeded = context => true;
                options.MinimumSameSitePolicy = SameSiteMode.None;
            });
            if (!Directory.Exists("appdata"))
            {
                Directory.CreateDirectory("appdata");
            }
            services.AddAntiforgery(o => o.HeaderName = "XSRF-TOKEN");
            services.AddMvc(options =>
            {
                options.CacheProfiles.Add("default", new CacheProfile
                {
                    Duration = 30,
                    Location = ResponseCacheLocation.Any
                });
            })
            .SetCompatibilityVersion(CompatibilityVersion.Version_2_2);
            services.AddSingleton<IHttpContextAccessor, HttpContextAccessor>();
            services.AddSingleton<IActivityLoggerService, ActivityLoggerService>(x => new ActivityLoggerService("Auth", Constants.AppConfig.Application.Environment));
            services.AddSingleton<IUserGroupSyncerService, LdapGroupSyncerService>();
            services.AddSingleton<ILdapService, LdapService>();
            services.AddMemoryCache();

            
        }        

        // This method gets called by the runtime. Use this method to configure the HTTP request pipeline.
        public void Configure(IApplicationBuilder app, IHostingEnvironment env, IMemoryCache cache, ILoggerFactory loggerFactory)
        {
            loggerFactory.AddNLog();
            env.ConfigureNLog(FileHelper.GetRelativePath($"nlog.{Constants.AppConfig.Application.Environment}.config"));                           

            //app.UseHsts();
            //app.UseHttpsRedirection();            

            app.UseForwardedHeaders(new ForwardedHeadersOptions
            {
                ForwardedHeaders = ForwardedHeaders.XForwardedFor | ForwardedHeaders.XForwardedProto
            });
            app.UseStatusCodePagesWithReExecute("/error/{0}");
            app.UseExceptionHandler("/error/500");
            app.UseMiddleware<ErrorHandlerMiddleware>(Options.Create(new ErrorHandlerMiddlewareOptions()));
            app.UseMiddleware<ConfigurationMiddleware>(Options.Create(new ConfigurationMiddlewareOptions()));
            app.UseMiddleware<TraceMiddleware>(Options.Create(new TraceMiddlewareOptions()));
            //app.UseMiddleware<ValidateAntiForgeryTokenMiddleware>();
            //app.UseStaticFiles();


            var cookiePolicyOptions = new CookiePolicyOptions
            {
                MinimumSameSitePolicy = SameSiteMode.Lax,
                HttpOnly = HttpOnlyPolicy.Always,
                Secure = CookieSecurePolicy.Always,
            };
            app.UseCookiePolicy();            
            
            app.UseCors(env.EnvironmentName);
            
            app.UseMvc(routes =>
            {
                routes.MapRoute(
                    name: "default",
                    template: "{controller=Home}/{action=Index}/{id?}");
            });
        }
        private void CookieDeleted(DeleteCookieContext cookie)
        {
            if (Constants.AppConfig.Application.Environment != "Production")
            {
                Log.Verbose($"Cookie Deleted: {cookie.CookieName}:{cookie.Context.User.Identity}");
            }
        }

        private void CookieCreated(AppendCookieContext cookie)
        {
            if (Constants.AppConfig.Application.Environment != "Production")
            {
                Log.Verbose($"Cookie Created: {cookie.CookieName}:{cookie.Context.User.Identity}:{cookie.CookieValue}");
            }
        }
    }
}
