﻿using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Configuration;
using Microsoft.Extensions.Logging;
using Resonance.Api.Microservices.Explorer.Models;
using Resonance.Core;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StringHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.ApiModels.ExplorerModels;
using System;
using System.Threading.Tasks;

namespace Resonance.Api.Microservices.Explorer.Controllers
{
    [Route("compendium")]
    public class CompendiumController : Controller
    {
        private readonly IConfiguration _config;
        private readonly ILogger<CompendiumController> log;

        public CompendiumController(IConfiguration config, ILogger<CompendiumController> logger)
        {
            _config = config;
            log = logger;
        }

        [HttpGet("get-filter-types")]
        public JsonResult GetFilterTypes()
        {
            var model = new GetFilterTypesViewModel();
            return new JsonResult(model);
        }

        [HttpGet("get-filter-class-meta-data")]
        public JsonResult GetFilterClass()
        {
            var model = new GetFilterClassViewModel();
            return new JsonResult(model);
        }

        [HttpGet("build-example")]
        public IActionResult BuildExample()
        {
            var model = new BuildExampleViewModel();
            return View(model);
        }

        [HttpGet("table-list")]
        public async Task<IActionResult> TableList()
        {
            var model = new TableListViewModel();

            try
            {
                using (var conn = await DBManagerMysql.GetConnectionAsync(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 600;
                        command.CommandText =
                        $@"
                            select
                                coalesce(a.`Type`, '') as TableType,
                                coalesce(a.`Schema`, '') as TableSchema,
                                coalesce(a.`Name`, '') as TableName,
                                coalesce(a.`Sql`, '') as TableSql,
                                coalesce(b.`Definition`, '') as TableDefinition
                            from development.maintenance_table_tracking as a
                            left join development.maintenance_table_tracking_definition as b
                                on a.`Type` = b.`Type`
                                and a.`Schema` = b.`Schema`
                                and a.`Name` = b.`Name`
                            ;
                        ";
                        using(var reader = await command.ExecuteReaderAsync())
                        {
                            if (reader.HasRows)
                            {
                                while(await reader.ReadAsync())
                                {
                                    var item = new TableListDisplayModel()
                                    {
                                        Type = reader.GetString(0),
                                        Schema = reader.GetString(1),
                                        Name = reader.GetString(2),
                                        Sql = reader.GetString(3),
                                        Definition = reader.GetString(4)
                                    };
                                    model.Tables.Add(item);
                                }
                            }
                        }
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }
            return View(model);
        }

        [HttpGet("table-update")]
        public async Task<IActionResult> TableUpdate(string type, string schema, string name)
        {
            if (string.IsNullOrWhiteSpace(type))
            {
                throw new ArgumentNullException("type");
            }
            if (string.IsNullOrWhiteSpace(schema))
            {
                throw new ArgumentNullException("schema");
            }
            if (string.IsNullOrWhiteSpace("name"))
            {
                throw new ArgumentNullException("name");
            }
            var model = new TableUpdateViewModel();

            try
            {
                using(var conn = await DBManagerMysql.GetConnectionAsync(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = 600;
                        command.CommandText =
                        $@"
                            select
                                a.`Type`,
                                a.`Schema`,
                                a.`Name`,
                                coalesce(b.`Definition`, '') as Definition
                            from {Constants.DatabaseSchema}maintenance_table_tracking as a
                            left join {Constants.DatabaseSchema}maintenance_table_tracking_definition as b
                                on a.`Type` = b.`Type`
                                and a.`Schema` = b.`Schema`
                                and a.`Name` = b.`Name`
                            where
                                a.`Type` = @type
                                and a.`Schema` = @schema
                                and a.`Name` = @name
                            ;
                        ";
                        command.Parameters.AddWithValue("@type", type);
                        command.Parameters.AddWithValue("@schema", schema);
                        command.Parameters.AddWithValue("@name", name);
                        using(var reader = await command.ExecuteReaderAsync())
                        {
                            if (reader.HasRows)
                            {
                                while (await reader.ReadAsync())
                                {
                                    var data = new TableDefinitionUpdateModel()
                                    {
                                        Type = reader.GetString(0),
                                        Schema = reader.GetString(1),
                                        Name = reader.GetString(2),
                                        Definition = reader.GetString(3)
                                    };
                                    model.UpdateData = data;
                                }
                            }
                        }
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }
            return View(model);
        }

        [HttpPost("table-update")]
        [ValidateAntiForgeryToken]
        public async Task<JsonResult> TableUpdate([FromBody]TableDefinitionUpdateModel data)
        {
            var response = new ApiResponse<bool>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            response.ResponseData = false;

            try
            {
                if(data != null && !string.IsNullOrWhiteSpace(data.Type) && !string.IsNullOrWhiteSpace(data.Schema) && !string.IsNullOrWhiteSpace(data.Name) && !string.IsNullOrWhiteSpace(data.Definition))
                {
                    using (var conn = await DBManagerMysql.GetConnectionAsync(true))
                    {
                        using (var command = conn.GetCommand())
                        {
                            command.CommandTimeout = 600;
                            command.CommandText =
                            $@"
                                insert into {Constants.DatabaseSchema}maintenance_table_tracking_definition (`Type`, `Schema`, `Name`, `Definition`)
                                values(@type, @schema, @name, @definition)
                                on duplicate key update
                                `Definition` = @definition
                                ;
                            ";
                            command.Parameters.AddWithValue("@type", data.Type);
                            command.Parameters.AddWithValue("@schema", data.Schema);
                            command.Parameters.AddWithValue("@name", data.Name);
                            command.Parameters.AddWithValue("@definition", data.Definition.GzipBase64Encode());
                            command.ExecuteNonQueryWithMeasurements("explorer_table_definition_update", context: HttpContext);
                        }
                    }
                    response.ResponseData = true;
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }
            return new JsonResult(response);
        }
    }
}