﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Configuration;
using Newtonsoft.Json;
using Resonance.Api.Shared.ViewModels;
using Resonance.Core;
using Resonance.Core.Attributes;
using Resonance.Core.Extensions;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.ErrorHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.ApiModels.ExplorerModels;
using Resonance.Core.Models.ConfigurationModels.Permissions;
using Resonance.Core.ViewModels;
using static Resonance.Core.Constants;

namespace Resonance.Api.Microservices.Controllers
{
    //[ResonanceAuth("Denied")]
    [Route("permission-admin")]
    public class PermissionController : Controller
    {
        private static string controllerName = "PermissionController";

        private readonly IConfiguration config;
        private static PermissionHelper permissionHelper { get; set; }

        static PermissionController()
        {
            permissionHelper = new PermissionHelper();
            permissionHelper.Initialize();
        }

        public PermissionController(IConfiguration _config)
        {
            config = _config;
        }

        [HttpGet("edit")]
        public IActionResult ConsolidatedEdit()
        {
            var model = new PermissionManagementViewModel("None");
            return View(model);
        }

        [HttpGet("migration-edit")]
        public IActionResult MigrationEdit()
        {
            var model = new PermissionMigrationViewModel();
            return View(model);
        }

        [HttpPost("get-migration-permissions")]
        [ValidateAntiForgeryToken()]
        public JsonResult GetMigrationPermissions([FromBody]MigrationPermissionRequestModel request)
        {
            var stopwatch = new Stopwatch();
            var statuscode = 200;
            var response = new ApiResponse<PermissionMigrationViewModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new System.Collections.Concurrent.ConcurrentBag<ElapsedTimeModel>()
                }
            };

            try
            {
                stopwatch.Start();
                statuscode = 200;

                var validationResult = request.Validate();
                statuscode = validationResult.Item2;
                response.ErrorMessage = validationResult.Item3;
                if (validationResult.Item1)
                {
                    response.ResponseData = new PermissionMigrationViewModel(request.SourceEnvironment, request.TargetEnvironment, request.Project);
                }
                response.Success = statuscode == 200 ? true : false;
            }
            catch (Exception ex)
            {
                statuscode = 400;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetMigrationPermissions";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return new JsonResult(response);
        }

        [HttpGet("copy-permissions")]
        public IActionResult CopyPermissions(string sourceProject, string targetProject)
        {
            try
            {
                if 
                (
                    !string.IsNullOrWhiteSpace(sourceProject) && !string.IsNullOrWhiteSpace(targetProject)
                    && sourceProject != "None" && targetProject != "None"
                    && sourceProject != targetProject
                    && sourceProject != "undefined" && targetProject != "undefined"
                )
                {
                    var sourceModel = new PermissionManagementViewModel(sourceProject);
                    permissionHelper.WritePermissionsToS3(targetProject, sourceModel.Configuration, HttpContext);
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                return RedirectToAction("ConsolidatedEdit", "Permission", fragment: "copy-project-success=False");
            }
            return RedirectToAction("ConsolidatedEdit", "Permission", fragment: "copy-project-success=True");
        }

        [HttpPost("edit")]
        public IActionResult UpdatePermissions(IFormCollection data)
        {
            var projectName = "None";
            if (data != null && data.ContainsKey("select_project") && !string.IsNullOrWhiteSpace(data["select_project"]))
            {
                projectName = data["select_project"];
            }
            // Process page inputs into configurations
            var rawPermissions = ((string)data?["raw_permissions"]).GetRawPermissions();
            var rawTokens = ((string)data?["raw_tokens"]).GetRawTokens();
            var rawRoles = ((string)data?["raw_roles"]).GetRawRoles();
            var rawLdapGroups = ((string)data?["raw_ldap_groups"]).GetRawLdapGroups();
            var rawLdapUsers = ((string)data?["raw_ldap_users"]).GetRawLdapUsers();
            var rawColumnPermissions = ((string)data?["raw_column_permissions"]).GetRawColumnPermissions();
            var mappingRoleLdapGroup = ((string)data?["role_to_ldap_group_mapping"]).GetRoleToLdapGroupMappings();
            var mappingRoleLdapUser = ((string)data?["role_to_ldap_user_mapping"]).GetRoleToLdapUserMappings();
            var mappingRoleToken = ((string)data?["role_to_token_mapping"]).GetRoleToTokenMappings();
            var mappingRolePermission = ((string)data?["role_to_permission_mapping"]).GetRoleToPermissionMappings();

            // Process data list into configuration model for saving to s3

            try
            {
                var newPermissions = new PermissionConfigurationModel()
                {
                    Permissions = rawPermissions,
                    Tokens = rawTokens,
                    Roles = rawRoles,
                    LdapGroups = rawLdapGroups,
                    LdapNames = rawLdapUsers,
                    ColumnPermissions = rawColumnPermissions,
                    RoleToLdapGroupMap = mappingRoleLdapGroup,
                    RoleToLdapNameMap = mappingRoleLdapUser,
                    RoleToTokenMap = mappingRoleToken,
                    RoleToPermissionMap = mappingRolePermission,
                    //ColumnList
                };
                if (newPermissions != null && newPermissions != default(PermissionConfigurationModel))
                {
                    permissionHelper.WritePermissionsToS3(projectName, newPermissions, HttpContext);
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                return RedirectToAction("ConsolidatedEdit", "Permission", fragment: "update-project-success=False");
            }

            return RedirectToAction("ConsolidatedEdit", "Permission", fragment: "update-project-success=True");
        }

        [HttpPost("migration-edit")]
        public IActionResult UpdateMigrationPermissions(IFormCollection data)
        {
            string actionName = "UpdateMigrationPermissions";

            var stopwatch = new Stopwatch();
            var statuscode = 200;
            var response = new ApiResponse<EmptyModel>()
            {
                Success = false,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                RequestStartTime = DateTime.UtcNow,
                Metrics = new ElapsedTimeModel()
                {
                    MetricName = "page_load",
                    ChildrenMetrics = new ConcurrentBag<ElapsedTimeModel>()
                }
            };

            try
            {
                var rawPermissions = ((string)data?["target_raw_permissions"]).GetRawPermissions();
                var rawTokens = ((string)data?["target_raw_tokens"]).GetRawTokens();
                var rawRoles = ((string)data?["target_raw_roles"]).GetRawRoles();
                var rawLdapGroups = ((string)data?["target_raw_ldap_groups"]).GetRawLdapGroups();
                var rawLdapUsers = ((string)data?["target_raw_ldap_users"]).GetRawLdapUsers();
                var rawColumnPermissions = ((string)data?["target_raw_column_permissions"]).GetRawColumnPermissions();
                var mappingRoleLdapGroup = ((string)data?["target_role_to_ldap_group_mapping"]).GetRoleToLdapGroupMappings();
                var mappingRoleLdapUser = ((string)data?["target_role_to_ldap_user_mapping"]).GetRoleToLdapUserMappings();
                var mappingRoleToken = ((string)data?["target_role_to_token_mapping"]).GetRoleToTokenMappings();
                var mappingRolePermission = ((string)data?["target_role_to_permission_mapping"]).GetRoleToPermissionMappings();

                try
                {
                    var newPermissions = new PermissionConfigurationModel()
                    {
                        Permissions = rawPermissions,
                        Tokens = rawTokens,
                        Roles = rawRoles,
                        LdapGroups = rawLdapGroups,
                        LdapNames = rawLdapUsers,
                        ColumnPermissions = rawColumnPermissions,
                        RoleToLdapGroupMap = mappingRoleLdapGroup,
                        RoleToLdapNameMap = mappingRoleLdapUser,
                        RoleToTokenMap = mappingRoleToken,
                        RoleToPermissionMap = mappingRolePermission,
                    };

                    var targetProjectName = "None";
                    var targetEnvironment = "None";
                    if (data != null && data.ContainsKey("project") && !string.IsNullOrWhiteSpace(data["project"]))
                    {
                        targetProjectName = data["project"];
                    }
                    if (data != null && data.ContainsKey("target_environment") && !string.IsNullOrWhiteSpace(data["target_environment"]))
                    {
                        targetEnvironment = data["target_environment"];
                    }
                    if (newPermissions != null && newPermissions != default(PermissionConfigurationModel))
                    {
                        var result = permissionHelper.WritePermissionsToS3(targetProjectName, newPermissions, HttpContext, targetEnvironment);
                        Log.Verbose($@"Result: {result}");
                    }
                }
                catch (Exception ex)
                {
                    Log.Error(ex, context: HttpContext);
                }
            }
            catch (Exception ex)
            {
                statuscode = 500;
                var errorcode = ErrorCodeHelper.CalculateErrorCode(ex.ToString());
                response.ErrorMessage = $@"{errorcode}: Unhandled error GetSeasonList";
                Log.Error($@"{errorcode}: {ex}", context: HttpContext);
            }
            finally
            {
                response.Success = statuscode == 200;
                Response.StatusCode = statuscode;
                response.RequestEndTime = DateTime.UtcNow;
                if (!response.Success)
                {
                    Log.Warn($@"Response was not successful for {controllerName}.{actionName}.", context: HttpContext);
                }
                stopwatch.Stop();
                response.Metrics.ElapsedMS = stopwatch.ElapsedMilliseconds;
            }

            return RedirectToAction("MigrationEdit", "Permission", fragment: $"update-project-success={response.Success}");
        }


        [HttpGet("get-permissions-for-project")]
        public JsonResult GetProjectPermissions(string projectName)
        {
            var response = new ApiResponse<PermissionManagementViewModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };
            try
            {
                var data = new PermissionManagementViewModel(projectName);
                response.ResponseData = data;
                response.Success = true;
                
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }
            return new JsonResult(response);
        }

        [HttpGet("items")]
        public IActionResult Items()
        {
            var model = new PermissionManagementViewModel(AppConfig.Application.Name);
            return View(model);
        }

        //Note: Seleting a permission needs to remove that section under Existing "Models"
        [HttpPost("items")]
        //[ValidateAntiForgeryToken()]
        public IActionResult ProcessItems(IFormCollection data)
        {
            try
            {
                var formData = new Dictionary<PermissionFormType, PermissionSingleItemModel>();

                if (data.ContainsKey("role-delete") && !string.IsNullOrWhiteSpace(data["role-delete"]))
                {
                    var formType = PermissionFormType.DeleteRole;
                    var itemid = data["role-delete"];

                    formData.Add(formType, new PermissionSingleItemModel()
                    {
                        FormType = formType,
                        ItemID = itemid
                    });
                }

                if (data.ContainsKey("permission-delete") && !string.IsNullOrWhiteSpace(data["permission-delete"]))
                {
                    var formType = PermissionFormType.DeletePermission;
                    var itemid = data["permission-delete"];

                    formData.Add(formType, new PermissionSingleItemModel()
                    {
                        FormType = formType,
                        ItemID = itemid
                    });                                    
                }

                if (data.ContainsKey("ldap-group-delete") && !string.IsNullOrWhiteSpace(data["ldap-group-delete"]))
                {
                    var formType = PermissionFormType.DeleteLdapGroup;
                    var itemid = data["ldap-group-delete"];

                    formData.Add(formType, new PermissionSingleItemModel()
                    {
                        FormType = formType,
                        ItemID = itemid
                    });
                }

                if (data.ContainsKey("ldap-name-delete") && !string.IsNullOrWhiteSpace(data["ldap-name-delete"]))
                {
                    var formType = PermissionFormType.DeleteLdapName;
                    var itemid = data["ldap-name-delete"];

                    formData.Add(formType, new PermissionSingleItemModel()
                    {
                        FormType = formType,
                        ItemID = itemid
                    });
                }

                if (data.ContainsKey("token-delete") && !string.IsNullOrWhiteSpace(data["token-delete"]))
                {
                    var formType = PermissionFormType.DeleteToken;
                    var itemid = data["token-delete"];

                    formData.Add(formType, new PermissionSingleItemModel()
                    {
                        FormType = formType,
                        ItemID = itemid
                    });
                }

                if (data.ContainsKey("restricted-columns-delete") && !string.IsNullOrWhiteSpace(data["restricted-columns-delete"]))
                {
                    var formType = PermissionFormType.DeleteModel;
                    var itemid = data["restricted-columns-delete"];

                    formData.Add(formType, new PermissionSingleItemModel()
                    {
                        FormType = formType,
                        ItemID = itemid
                    });
                }

                /* New Single Items */
                if (data.ContainsKey("role-name") && !string.IsNullOrWhiteSpace(data["role-name"]))
                {
                    var formType = PermissionFormType.NewRole;
                    var newvalue = data["role-name"];

                    formData.Add(formType, new PermissionSingleItemModel()
                    {
                        FormType = formType,
                        OldValue = null,
                        ItemID = Guid.NewGuid().ToString("D"),
                        ItemDescription = data.GetFirstOrDefault("role-description"),
                        NewValue = newvalue
                    });
                }

                if (data.ContainsKey("column-permissions-column-name") && !string.IsNullOrWhiteSpace(data["column-permissions-column-name"]))
                {
                    var formType = PermissionFormType.NewModel;
                    var newvalue = data["column-permissions-column-name"];
                    
                    var columnString = data.GetFirstOrDefault("restricted-columns-permissions");
                    string[] columns = columnString.Split(',');                    

                    formData.Add(formType, new PermissionColumnItemModel()
                    {
                        FormType = formType,
                        OldValue = null,
                        ItemID = Guid.NewGuid().ToString("D"),
                        ItemDescription = data.GetFirstOrDefault("restricted-columns-description"),
                        NewValue = newvalue,
                        Columns = columns.ToList()                                             
                    });
                }

                if (data.ContainsKey("permission-name") && !string.IsNullOrWhiteSpace(data["permission-name"]))
                {
                    var formType = PermissionFormType.NewPermission;
                    var newvalue = data["permission-name"];

                    formData.Add(formType, new PermissionSingleItemModel()
                    {
                        FormType = formType,
                        OldValue = null,
                        ItemID = Guid.NewGuid().ToString("D"),
                        ItemDescription = data.GetFirstOrDefault("permission-description"),
                        NewValue = newvalue
                    });
                }

                if (data.ContainsKey("ldap-group-name") && !string.IsNullOrWhiteSpace(data["ldap-group-name"]))
                {
                    var formType = PermissionFormType.NewLdapGroup;
                    var newvalue = data["ldap-group-name"];

                    formData.Add(formType, new PermissionSingleItemModel()
                    {
                        FormType = formType,
                        OldValue = null,
                        ItemID = Guid.NewGuid().ToString("D"),
                        ItemDescription = data.GetFirstOrDefault("ldap-group-description"),
                        NewValue = newvalue
                    });
                }

                if (data.ContainsKey("ldap-name") && !string.IsNullOrWhiteSpace(data["ldap-name"]))
                {
                    var formType = PermissionFormType.NewLdapName;
                    var newvalue = data["ldap-name"];

                    formData.Add(formType, new PermissionSingleItemModel()
                    {
                        FormType = formType,
                        OldValue = null,
                        ItemID = Guid.NewGuid().ToString("D"),
                        ItemDescription = data.GetFirstOrDefault("ldap-name-description"),
                        NewValue = newvalue
                    });
                }

                if (data.ContainsKey("token-name") && !string.IsNullOrWhiteSpace(data["token-name"]))
                {
                    var formType = PermissionFormType.NewToken;
                    var newvalue = data["token-name"];
                    var ipWhitelist = data.GetFirstOrDefault("token-whitelist", false, false);

                    formData.Add(formType, new PermissionSingleItemModel()
                    {
                        FormType = formType,
                        OldValue = null,
                        ItemID = Guid.NewGuid().ToString("D"),
                        ItemDescription = data.GetFirstOrDefault("token-description"),
                        NewValue = newvalue,
                        IPWhitelist = ipWhitelist.Split(",")
                    });
                }

                permissionHelper.WritePermissionsToS3("Legacy", permissionHelper.ProcessNewItems(formData), HttpContext);
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }

            var model = new PermissionManagementViewModel(AppConfig.Application.Name);
            return View("items", model);
        }

        [HttpGet("maps")]
        public IActionResult Maps()
        {
            var model = new PermissionManagementViewModel(AppConfig.Application.Name);
            return View(model);
        }

        [HttpPost("maps")]
        //[ValidateAntiForgeryToken()]
        public IActionResult ProcessMaps(IFormCollection data)
        {
            var model = new PermissionManagementViewModel(AppConfig.Application.Name);

            try
            {
                var formData = new Dictionary<PermissionFormType, PermissionMapModel>();

                /* Maps */
                if (data.ContainsKey("role-permission-select") && data["role-permission-select"].Count > 0)
                {
                    var roleModel = model.Configuration.Roles.FirstOrDefault(x => x.RoleID == data["role-permission-select"]);
                    if (roleModel != null)
                    {
                        model.Configuration.RoleToPermissionMap.Remove(roleModel.RoleName);
                        model.Configuration.RoleToPermissionMap.Add(roleModel.RoleName, new List<string>());

                        foreach (var permission in data["permission-select"])
                        {
                            var perm = model.Configuration.Permissions.FirstOrDefault(x => x.PermissionID == permission);
                            if (perm != null)
                            {
                                model.Configuration.RoleToPermissionMap[roleModel.RoleName].Add(perm.PermissionName);
                            }
                        }
                    }
                }

                if (data.ContainsKey("role-token-select") && data["role-token-select"].Count > 0)
                {
                    var roleModel = model.Configuration.Roles.FirstOrDefault(x => x.RoleID == data["role-token-select"]);
                    if (roleModel != null)
                    {
                        model.Configuration.RoleToTokenMap.Remove(roleModel.RoleName);
                        model.Configuration.RoleToTokenMap.Add(roleModel.RoleName, new List<string>());

                        foreach (var token in data["token-select"])
                        {
                            var tok = model.Configuration.Tokens.FirstOrDefault(x => x.TokenID == token);
                            if (tok != null)
                            {
                                model.Configuration.RoleToTokenMap[roleModel.RoleName].Add(tok.TokenName);
                            }
                        }
                    }
                }

                if (data.ContainsKey("role-ldap-group-select") && data["role-ldap-group-select"].Count > 0)
                {
                    var roleModel = model.Configuration.Roles.FirstOrDefault(x => x.RoleID == data["role-ldap-group-select"]);
                    if (roleModel != null)
                    {
                        model.Configuration.RoleToLdapGroupMap.Remove(roleModel.RoleName);
                        model.Configuration.RoleToLdapGroupMap.Add(roleModel.RoleName, new List<string>());

                        foreach (var ldapGroup in data["ldap-group-select"])
                        {
                            var ldap = model.Configuration.LdapGroups.FirstOrDefault(x => x.LdapGroupName == ldapGroup);
                            if (ldap != null)
                            {
                                model.Configuration.RoleToLdapGroupMap[roleModel.RoleName].Add(ldap.LdapGroupName);
                            }
                        }
                    }
                }

                if (data.ContainsKey("role-ldap-name-select") && data["role-ldap-name-select"].Count > 0)
                {
                    var roleModel = model.Configuration.Roles.FirstOrDefault(x => x.RoleID == data["role-ldap-name-select"]);
                    if (roleModel != null)
                    {
                        model.Configuration.RoleToLdapNameMap.Remove(roleModel.RoleName);
                        model.Configuration.RoleToLdapNameMap.Add(roleModel.RoleName, new List<string>());

                        foreach (var ldapName in data["ldap-name-select"])
                        {
                            var ldap = model.Configuration.LdapNames.FirstOrDefault(x => x.LdapNameID == ldapName);
                            if (ldap != null)
                            {
                                model.Configuration.RoleToLdapNameMap[roleModel.RoleName].Add(ldap.LdapName);
                            }
                        }
                    }
                }

                permissionHelper.WritePermissionsToS3("Legacy", permissionHelper.ProcessNewMaps(model.Configuration), HttpContext);
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }

            return View("maps", model);
        }

        [HttpGet("advanced-edit")]
        public IActionResult AdvancedEdit(IFormCollection data)
        {
            var model = new PermissionManagementViewModel(AppConfig.Application.Name);
            return View(model);
        }

        [HttpPost("advanced-edit")]
        //[ValidateAntiForgeryToken()]
        public IActionResult ProcessAdvancedEdit(IFormCollection data)
        {
            try
            {
                if (data.ContainsKey("advanced-edit-input") && !string.IsNullOrWhiteSpace(data["advanced-edit-input"]))
                {
                    var permissions = JsonConvert.DeserializeObject<PermissionConfigurationModel>(data["advanced-edit-input"]);
                    if(permissions != null && permissions != default(PermissionConfigurationModel))
                    {
                        permissionHelper.WritePermissionsToS3("Legacy", permissions, HttpContext);
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            var model = new PermissionManagementViewModel(AppConfig.Application.Name);
            return View("AdvancedEdit", model);
        }


        [HttpPost("delete-item")]
        //[ValidateAntiForgeryToken()]
        public IActionResult DeleteItem(IFormCollection data)
        {
            throw new NotImplementedException();
        }

        [HttpPost("delete-map")]
        //[ValidateAntiForgeryToken()]
        public IActionResult DeleteMap(IFormCollection data)
        {
            throw new NotImplementedException();
        }

        /* Get Methods to help with debugging */
        [HttpGet("get-all-maps")]
        public JsonResult GetAllMaps()
        {
            var response = new ApiResponse<PermissionConfigurationModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                response.ResponseData = permissionHelper.GetPermissionsFromS3(AppConfig.Application.Name);
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }
            return new JsonResult(response);
        }

        [HttpGet("get-permissions")]
        public JsonResult GetPermissions()
        {
            var response = new ApiListResponse<PermissionModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                response.ResponseData = permissionHelper.GetPermissionsFromS3(AppConfig.Application.Name).Permissions;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }
            return new JsonResult(response);
        }

        [HttpGet("get-roles")]
        public JsonResult GetRoles()
        {
            var response = new ApiListResponse<RoleModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                response.ResponseData = permissionHelper.GetPermissionsFromS3(AppConfig.Application.Name).Roles;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }
            return new JsonResult(response);
        }

        [HttpGet("get-ldap-groups")]
        public JsonResult GetLdapGroups()
        {
            var response = new ApiListResponse<LdapGroupModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                response.ResponseData = permissionHelper.GetPermissionsFromS3(AppConfig.Application.Name).LdapGroups;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }
            return new JsonResult(response);
        }

        [HttpGet("get-ldap-names")]
        public JsonResult GetLdapNames()
        {
            var response = new ApiListResponse<LdapNameModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                response.ResponseData = permissionHelper.GetPermissionsFromS3(AppConfig.Application.Name).LdapNames;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }
            return new JsonResult(response);
        }

        [HttpGet("get-tokens")]
        public JsonResult GetTokens()
        {
            var response = new ApiListResponse<TokenModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                response.ResponseData = permissionHelper.GetPermissionsFromS3(AppConfig.Application.Name).Tokens;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }
            return new JsonResult(response);
        }

        [HttpGet("get-role-permission-map")]
        public JsonResult GetRolePermissionMap()
        {
            var response = new ApiResponse<Dictionary<string, List<string>>>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                response.ResponseData = permissionHelper.GetPermissionsFromS3(AppConfig.Application.Name).RoleToPermissionMap;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }
            return new JsonResult(response);
        }

        [HttpGet("get-role-ldap-group-map")]
        public JsonResult GetRoleLdapGroupMap()
        {
            var response = new ApiResponse<Dictionary<string, List<string>>>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                response.ResponseData = permissionHelper.GetPermissionsFromS3(AppConfig.Application.Name).RoleToLdapGroupMap;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }
            return new JsonResult(response);
        }

        [HttpGet("get-role-ldap-name-map")]
        public JsonResult GetRoleLdapNameMap()
        {
            var response = new ApiResponse<Dictionary<string, List<string>>>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                response.ResponseData = permissionHelper.GetPermissionsFromS3(AppConfig.Application.Name).RoleToTokenMap;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }
            return new JsonResult(response);
        }

        [HttpGet("get-role-token-map")]
        public JsonResult GetRoleTokenMap()
        {
            var response = new ApiResponse<Dictionary<string, List<string>>>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                response.ResponseData = permissionHelper.GetPermissionsFromS3(AppConfig.Application.Name).RoleToTokenMap;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }
            return new JsonResult(response);
        }
    }
}