﻿using System;
using System.IO;
using Microsoft.AspNetCore.Hosting;
using Microsoft.Extensions.Configuration;
using NLog.Web;
using Resonance.Core;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.ConfigurationHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StatsDHelpers;

namespace Resonance.Api.Microservices.Explorer
{
    public class Program
    {
        private static string originalEnvironment = null;

        public static void Main(string[] args)
        {
            try
            {
                var builder = new ConfigurationBuilder();
                builder.AddCommandLine(args);
                var config = builder.Build();

                try
                {
                    originalEnvironment = config["environment"];
                }
                catch (Exception)
                {
                    throw new ArgumentNullException($@"'Environment' argument must be specified. This can be done through the launch file, or through the IDE argument section.");
                }

                AppConfig.Initialize(ref originalEnvironment, ref args);

                switch (AppConfig.Data.Application.Environment)
                {
                    case "Development":
                    case "Staging":
                    case "Production":
                    {
                        break;
                    }
                    default:
                    {
                        throw new NotImplementedException($"Undefined application environment '{config["Resonance:Application:Environment"]}', please define handling in Resonance.Api.WorkerNode.Program.cs and Resonance.Core.Constants.cs");
                    }
                }
                Constants.InitConstants(AppConfig.Data, "Explorer");
                Log.Initialize();
                CloudwatchHelper.Initialize("Api.Explorer");
                string ip = "telegraf.cursedata.io";
                StatsDHelper.Initialize(AppConfig.Data.Application.BaseCloudwatchLogPath, ip, 8125, "Resonance", $"Explorer", TimeSpan.FromMinutes(1));

                Log.Info($"({AppConfig.Data.Application.Environment})Init Main");
                DBManagerMysql.Initialize("resonance");
                BuildWebHost(args).Run();
            }
            catch (Exception ex)
            {
                //NLog: catch setup errors
                Log.Error($"{ex} -Stopped program because of exception");
            }
            finally
            {
                // Ensure to flush and stop internal timers/threads before application-exit (Avoid segmentation fault on Linux)
                Log.Info($"Application is shutting down");
                NLog.LogManager.Shutdown();
            }
        }

        public static IWebHost BuildWebHost(string[] args)
        {
            var host = new WebHostBuilder()
                .UseConfiguration(AppConfig.GetConfiguration())
                .UseContentRoot(Directory.GetCurrentDirectory())
                .UseKestrel()
                .UseStartup<Startup>()
                .UseNLog()
                .Build();

            return host;
        }
    }
}
