﻿using System;
using System.Collections.Generic;
using System.Linq;
using Resonance.Core;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StatsDHelpers;
using Resonance.Core.Models.ApiModels;
using Microsoft.AspNetCore.Mvc;
using Microsoft.Extensions.Configuration;
using Resonance.Api.Microservices.TwitchUserLookup;
using Resonance.Core.Models.ServiceModels.TwitchModels;
using Resonance.Microservices.Methods;
using Resonance.Core.Attributes;

namespace Resonance.Api.MicroServices.TwitchUserLookup.Controllers
{
    [ResonanceAuth("TwitchUserLookup", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
    public class TwitchUserController : Controller
    {
        private readonly IConfiguration _config;
        private static TwitchUserMethods twitchUserMethods { get; set; }

        static TwitchUserController()
        {
            twitchUserMethods = new TwitchUserMethods();
            twitchUserMethods.Initialize();
        }

        public TwitchUserController(IConfiguration config)
        {
            _config = config;
        }

        [HttpGet("worker/twitch-user/get-by-twitch-user-id/{twitchUserID:long}")]
        public JsonResult GetTwitchUserByTwitchUserID(long twitchUserID)
        {
            var response = new ApiResponse<TwitchUserModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (twitchUserID > 0)
                {
                    var linkdata = twitchUserMethods.GetTwitchUser(twitchUserID: twitchUserID);
                    response.Success = true;
                    response.ResponseData = linkdata;
                }
                else
                {
                    response.ErrorMessage = "Twitch User ID <= 0";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception in GetTwitchUserByTwitchUserID";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        [HttpGet("worker/twitch-user/get-by-twitch-login/{twitchLogin}")]
        public JsonResult GetTwitchUserByTwitchLogin(string twitchLogin)
        {
            var response = new ApiResponse<TwitchUserModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (!string.IsNullOrWhiteSpace(twitchLogin))
                {
                    var linkdata = twitchUserMethods.GetTwitchUser(twitchLogin: twitchLogin);
                    response.Success = true;
                    response.ResponseData = linkdata;
                }
                else
                {
                    response.ErrorMessage = "Twitch Login is empty";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception in GetTwitchUserByTwitchLogin";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        [HttpPost("worker/twitch-user/get-by-list-twitch-user-id")]
        public JsonResult GetTwitchUserListByTwitchUserIDs([FromBody]long[] twitchUserIDs)
        {
            var response = new ApiListResponse<TwitchUserModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (twitchUserIDs != null && twitchUserIDs.Length > 0)
                {
                    var linkdata = twitchUserMethods.GetTwitchUsers(twitchUserIDs: twitchUserIDs);
                    if (linkdata != null)
                    {
                        response.Success = true;
                        response.ResponseData = linkdata;
                    }
                    else
                    {
                        response.ErrorMessage = "Link data is null";
                    }
                }
                else
                {
                    response.ErrorMessage = "Twitch User IDs null or empty";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception in GetTwitchUserListByTwitchUserIDs";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        [HttpPost("worker/twitch-user/get-by-list-twitch-login")]
        public JsonResult GetTwitchUserListByTwitchLogins([FromBody]string[] twitchLogins)
        {
            var response = new ApiListResponse<TwitchUserModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (twitchLogins != null && twitchLogins.Length > 0)
                {
                    var linkdata = twitchUserMethods.GetTwitchUsers(twitchLogins: twitchLogins);
                    if (linkdata != null)
                    {
                        response.Success = true;
                        response.ResponseData = linkdata;
                    }
                    else
                    {
                        response.ErrorMessage = "Link data is null";
                    }
                }
                else
                {
                    response.ErrorMessage = "Twitch logins null or empty";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception in GetTwitchUserListByTwitchLogins";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        [ResonanceAuth("TwitchUserLookup", requiredPermissionsAny: ConstantsPermissions.Amp.ServiceUpdate)]
        [HttpPost("worker/twitch-user/delete-twitch-user")]
        public JsonResult DeleteTwitchUser([FromBody]TwitchUserModel user)
        {
            throw new NotImplementedException();
        }

        [ResonanceAuth("TwitchUserLookup", requiredPermissionsAny: ConstantsPermissions.Amp.ServiceUpdate)]
        [HttpPost("worker/twitch-user/insert-update-twitch-user")]
        public JsonResult InsertUpdateTwitchUser([FromBody]TwitchUserModel user)
        {
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                response.Success = twitchUserMethods.InsertUpdateTwitchUser(Constants.DatabaseSchema, ref user) ?? false;
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception in InsertUpdateTwitchUser";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        /// <summary>
        /// Inserts missing users and updates existing users with specified information
        /// </summary>
        [ResonanceAuth("TwitchUserLookup", requiredPermissionsAny: ConstantsPermissions.Amp.ServiceUpdate)]
        [HttpPost("worker/twitch-user/bulk-insert-update-twitch-user")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        [RequestSizeLimit(100_000_000)]
        public JsonResult InsertUpdateBulkTwitchUser([FromBody]TwitchUserListModel data)
        {
            var response = new ApiListResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (Request.Headers != null)
                {
                    var keys = Request.Headers
                        .Where(x => x.Key.StartsWith("scanner-"))
                        .Select(x => $"{x.Key}={x.Value}");
                    if (keys.Count() > 0)
                    {
                        var headers = string.Join(",", keys);
                        Log.Info($"Received Bulk Insert Twitch User Update Request: {headers}", context: HttpContext);
                    }
                }

                if(data != null && data.Data != null && data.Data.Length > 0)
                {
                    response.ReceivedRecords = data.Data.Length;

                    var batchStart = DateTime.UtcNow;
                    var batchID = Guid.NewGuid().ToString().Replace("-", "");
                    data.WorkerIdentifier = ConstantsWorker.WorkerIdentifier;
                    data.BatchID = batchID;
                    data.BatchStart = batchStart;

                    var processedCount = twitchUserMethods.InsertUpdateBulkTwitchUser(ConstantsWorker.WorkerIdentifier, batchID, batchStart, ConstantsWorker.WorkerS3Bucket, ConstantsWorker.WorkerS3Path, Constants.DatabaseSchema, data.Data.Distinct().ToArray());

                    response.ProcessedRecords = processedCount;
                    response.Success = true;
                }
                else
                {
                    response.ErrorMessage = "Data is null or empty";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception in InsertUpdateBulkTwitchUser";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        /// <summary>
        /// Updates all users with inactive information if they are in the system
        /// </summary>
        [ResonanceAuth("TwitchUserLookup", requiredPermissionsAny: ConstantsPermissions.Amp.ServiceUpdate)]
        [HttpPost("worker/twitch-user/bulk-update-inactive-user")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        [RequestSizeLimit(100_000_000)]
        public JsonResult UpdateInactiveStatus([FromBody]TwitchBannedOrDeletedUserListModel data)
        {
            var response = new ApiListResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                var keys = Request.Headers
                        .Where(x => x.Key.StartsWith("scanner-"))
                        .Select(x => $"{x.Key}={x.Value}");
                if (keys.Count() > 0)
                {
                    var headers = string.Join(",", keys);
                    Log.Info($"Received Bulk Twitch User Update Inactive Request: {headers}", context: HttpContext);
                }

                if (data != null && data.Data != null && data.Data.Length > 0)
                {
                    response.ReceivedRecords = data.Data.Length;

                    var batchStart = DateTime.UtcNow;
                    var batchID = Guid.NewGuid().ToString().Replace("-", "");
                    var processedCount = twitchUserMethods.UpdateInactiveUsers(Constants.DatabaseSchema, data.Data.Distinct().ToArray()) ?? 0;

                    response.ProcessedRecords = processedCount;
                    response.Success = true;
                }
                else
                {
                    response.ErrorMessage = "Data is null or empty";
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception in UpdateInactiveStatus";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }
            return new JsonResult(response);
        }

        /// <summary>
        /// Updates specified user with inactive information if they are in the system
        /// </summary>
        [ResonanceAuth("TwitchUserLookup", requiredPermissionsAny: ConstantsPermissions.Amp.ServiceUpdate)]
        [HttpPost("worker/twitch-user/update-inactive-user")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public JsonResult UpdateInactiveStatus([FromBody]TwitchBannedOrDeletedUserModel data)
        {
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (data != null && data.UserID > 0)
                {
                    var batchStart = DateTime.UtcNow;
                    var batchID = Guid.NewGuid().ToString().Replace("-", "");
                    response.Success = twitchUserMethods.UpdateInactiveUser(Constants.DatabaseSchema, data) ?? false;
                }
                else
                {
                    response.ErrorMessage = "Data is null or empty";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, context: HttpContext);
                response.ErrorMessage = "Exception in UpdateInactiveStatus";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }
            return new JsonResult(response);
        }
    }
}