﻿using System;
using System.Linq;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels;
using Microsoft.AspNetCore.Mvc;
using Resonance.Core;
using Resonance.Core.Models.ApiModels.TwitchModels;
using Resonance.Core.Models.ApiModels.YoutubeModels;
using Microsoft.Extensions.Configuration;
using static Resonance.Core.Constants;
using Resonance.Core.Models.ServiceModels.TwitchModels;
using Resonance.Microservices.Methods;
using Resonance.Core.Attributes;

namespace Resonance.Api.Microservices.Controllers.TwitchVerifiedLink
{
    public class TwitchVerifiedLinkController : Controller
    {
        private static TwitchVerifiedLinkMethods twitchVerifiedLink { get; set; }
        private static TwitchUserMethods twitchUserMethods { get; set; }
        private readonly IConfiguration _config;

        static TwitchVerifiedLinkController()
        {
            twitchUserMethods = new TwitchUserMethods();
            twitchUserMethods.Initialize();
            twitchVerifiedLink = new TwitchVerifiedLinkMethods();
            twitchVerifiedLink.Initialize();
        }

        public TwitchVerifiedLinkController(IConfiguration config)
        {
            _config = config;
        }

        /// <summary>
        /// Use a Twitch User ID to get Verified Link Data, or return null [ResponseData]
        /// </summary>
        [ResonanceAuth("TwitchVerifiedLink", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpGet("worker/twitch-verified-link/get-by-twitch-user-id/{twitchUserID:long}")]
        public JsonResult GetTwitchVerifiedLinkByTwitchUserID(long twitchUserID)
        {
            var response = new ApiResponse<TwitchVerifiedLinkModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (twitchUserID > 0)
                {
                    var linkdata = twitchVerifiedLink.GetTwitchVerifiedLinkByTwitchUserID(twitchUserID);
                    response.Success = true;
                    response.ResponseData = linkdata;
                }
                else
                {
                    response.ErrorMessage = "Twitch User ID is <= 0";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                response.ErrorMessage = "Exception in GetTwitchVerifiedLinkByTwitchUserID";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        /// <summary>
        /// Get a single Twitch Verified Link Result based on Twitch Login. Populated to [ResponseData]
        /// </summary>
        [ResonanceAuth("TwitchVerifiedLink", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpGet("worker/twitch-verified-link/get-by-twitch-login/{twitchLogin}")]
        public JsonResult GetTwitchVerifiedLinkByTwitchLogin(string twitchLogin)
        {
            var response = new ApiResponse<TwitchVerifiedLinkModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (!string.IsNullOrWhiteSpace(twitchLogin))
                {
                    var linkdata = twitchVerifiedLink.GetTwitchVerifiedLinkByTwitchLogin(twitchLogin);
                    response.Success = true;
                    response.ResponseData = linkdata;
                }
                else
                {
                    response.ErrorMessage = "Twitch login is blank";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                response.ErrorMessage = "Exception in GetTwitchVerifiedLinkByTwitchLogin";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        /// <summary>
        /// Get ALL twitch verified link data, populated in [ResponseData]
        /// </summary>
        [ResonanceAuth("TwitchVerifiedLink", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpGet("worker/twitch-verified-link/get-all")]
        public JsonResult GetTwitchVerifiedLinkGetAll()
        {
            var response = new ApiListResponse<TwitchVerifiedLinkModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                var linkdata = twitchVerifiedLink.GetTwitchVerifiedLinkGetAll();
                response.Success = true;
                response.ResponseData = linkdata;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                response.ErrorMessage = "Exception in GetTwitchVerifiedLinkGetAll";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        /// <summary>
        /// Get Twitch Verified Link data for users in the supplied list. Populated to [ResponseData]
        /// </summary>
        [ResonanceAuth("TwitchVerifiedLink", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpPost("worker/twitch-verified-link/get-by-list-twitch-user-id")]
        public JsonResult GetTwitchVerifiedLinkByTwitchUserIDList([FromBody]UserIDListModel twitchUserIDs)
        {
            var response = new ApiListResponse<TwitchVerifiedLinkModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (twitchUserIDs != null && twitchUserIDs.UserIDs != null && twitchUserIDs.UserIDs.Length > 0)
                {
                    var linkdata = twitchVerifiedLink.GetTwitchVerifiedLinkByTwitchUserIDs(twitchUserIDs.UserIDs.Distinct().ToArray());
                    if (linkdata != null)
                    {
                        response.Success = true;
                        response.ResponseData = linkdata;
                    }
                    else
                    {
                        response.ErrorMessage = "Link data is null";
                    }
                }
                else
                {
                    response.ErrorMessage = "Twitch User Ids null or empty";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                response.ErrorMessage = "Exception in GetTwitchVerifiedLinkByTwitchUserIDList";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        /// <summary>
        /// Get Twitch Verified Link Results using a Twitch Login. Populated to [ResponseData]
        /// </summary>
        [ResonanceAuth("TwitchVerifiedLink", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpPost("worker/twitch-verified-link/get-by-list-twitch-login")]
        public JsonResult GetTwitchVerifiedLinkByTwitchLogin([FromBody]UserLoginListModel twitchUserLogins)
        {
            var response = new ApiListResponse<TwitchVerifiedLinkModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (twitchUserLogins != null && twitchUserLogins.TwitchLogins != null && twitchUserLogins.TwitchLogins.Length > 0)
                {
                    var linkdata = twitchVerifiedLink.GetTwitchVerifiedLinkByTwitchLogins(twitchUserLogins.TwitchLogins.Distinct().ToArray());
                    if (linkdata != null)
                    {
                        response.Success = true;
                        response.ResponseData = linkdata;
                    }
                    else
                    {
                        response.ErrorMessage = "Link data is null";
                    }
                }
                else
                {
                    response.ErrorMessage = "Twitch User Logins null or empty";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                response.ErrorMessage = "Exception in GetTwitchVerifiedLinkByTwitchLogin";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        /// <summary>
        /// Get a list of Twitch Verified Link Results using Youtube Tags. Populated to [ResponseData]
        /// </summary>
        [ResonanceAuth("TwitchVerifiedLink", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpPost("worker/twitch-verified-link/get-by-list-youtube-tags")]
        public JsonResult GetTwitchVerifiedLinkByYoutubeTags([FromBody]UserTagListModel youtubeTags)
        {
            var response = new ApiListResponse<TwitchVerifiedLinkModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (youtubeTags != null && youtubeTags.Tags != null && youtubeTags.Tags.Length > 0)
                {
                    var linkdata = twitchVerifiedLink.GetTwitchVerifiedLinkByYoutubeTags(youtubeTags.Tags.Distinct().ToArray());
                    if (linkdata != null)
                    {
                        response.Success = true;
                        response.ResponseData = linkdata;
                    }
                    else
                    {
                        response.ErrorMessage = "Link data is null";
                    }
                }
                else
                {
                    response.ErrorMessage = "Youtube tags null or empty";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                response.ErrorMessage = "Exception in GetTwitchVerifiedLinkByYoutubeTags";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        /// <summary>
        /// Get a single Twitch Verified Link Result based on Youtube Tag. Populated to [ResponseData]
        /// </summary>
        [ResonanceAuth("TwitchVerifiedLink", requiredPermissions: ConstantsPermissions.Amp.GeneralAccess)]
        [HttpGet("worker/twitch-verified-link/get-by-youtube-tag/{youtubeTag}")]
        public JsonResult GetTwitchVerifiedLinkByYoutubeTag(string youtubeTag)
        {
            var response = new ApiResponse<TwitchVerifiedLinkModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (!string.IsNullOrWhiteSpace(youtubeTag))
                {
                    var linkdata = twitchVerifiedLink.GetTwitchVerifiedLinkByYoutubeTag(youtubeTag);
                    response.Success = true;
                    response.ResponseData = linkdata;
                }
                else
                {
                    response.ErrorMessage = "Youtube tag is empty";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                response.ErrorMessage = "Exception in GetTwitchVerifiedLinkByYoutubeTag";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        /// <summary>
        /// Bulk insert Twitch Verified Link Data, maximum request size is 100MB
        /// </summary>
        [ResonanceAuth("TwitchVerifiedLink", requiredPermissionsAny: ConstantsPermissions.Amp.ServiceUpdate)]
        [HttpPost("worker/twitch-verified-link/bulk-insert-update")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        [RequestSizeLimit(100_000_000)]
        //[DisableRequestSizeLimit]
        public JsonResult InsertUpdateBulkTwitchVerifiedLink([FromBody]TwitchVerifiedLinkListModel data)
        {
            var response = new ApiListResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (Request.Headers != null)
                {
                    var keys = Request.Headers
                        .Where(x => x.Key.StartsWith("scanner- "))
                        .Select(x => $"{x.Key}={x.Value}");
                    if (keys.Count() > 0)
                    {
                        var headers = string.Join(",", keys);
                        Log.Info($"Received Bulk Insert Update Request: {headers}");
                    }
                }

                if (data != null && data.Data != null && data.Data.Length > 0)
                {
                    response.ReceivedRecords = data.Data.Length;

                    var batchStart = DateTime.UtcNow;
                    var batchID = Guid.NewGuid().ToString().Replace("-", "");
                    data.WorkerIdentifier = ConstantsWorker.WorkerIdentifier;
                    data.BatchID = batchID;
                    data.BatchStart = batchStart;

                    var processedCount = twitchVerifiedLink.InsertUpdateBulkTwitchVerifiedLink(ConstantsWorker.WorkerIdentifier, batchID, batchStart, ConstantsWorker.WorkerS3Bucket, ConstantsWorker.WorkerS3Path, Constants.DatabaseSchema, data.Data.Distinct().ToArray());

                    response.ProcessedRecords = processedCount;
                    response.Success = true;
                }
                else
                {
                    response.ErrorMessage = "Data is null or empty";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                response.ErrorMessage = "Exception in InsertUpdateBulkTwitchVerifiedLink";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        /// <summary>
        /// Insert specified Twitch Verified Link data and return the updated information in the [ResponseData]
        /// </summary>
        [ResonanceAuth("TwitchVerifiedLink", requiredPermissionsAny: ConstantsPermissions.Amp.ServiceUpdate)]
        [HttpPost("worker/twitch-verified-link/insert-update")]
        [ResponseCache(Duration = 0, Location = ResponseCacheLocation.None, NoStore = true)]
        public JsonResult InsertUpdateTwitchVerifiedLink([FromBody]TwitchVerifiedLinkModel data)
        {
            var response = new ApiResponse<TwitchVerifiedLinkModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (Request.Headers != null)
                {
                    var keys = Request.Headers
                        .Where(x => x.Key.StartsWith("web-"))
                        .Select(x => $"{x.Key}={x.Value}");
                    if (keys.Count() > 0)
                    {
                        var headers = string.Join(",", keys);
                        Log.Info($"Received Insert Update Twitch Verified Link Request: {headers}");
                    }
                }

                if (data != null)
                {
                    var batchStart = DateTime.UtcNow;
                    var batchID = Guid.NewGuid().ToString().Replace("-", "");

                    var processedData = twitchVerifiedLink.InsertUpdateTwitchVerifiedLink(Constants.DatabaseSchema, ref data);
                    if (processedData != null)
                    {
                        response.ResponseData = processedData;
                        response.Success = true;
                    }

                    var user = new TwitchUserModel()
                    {
                        TwitchUserID = data.TwitchUserID.Value,
                        TwitchLogin = data.TwitchLogin,
                        IsActive = true,
                        InactiveReason = TwitchInactiveStatus.Active
                    };
                    twitchUserMethods.InsertUpdateTwitchUser(Constants.DatabaseSchema, ref user);
                }
                else
                {
                    response.ErrorMessage = "Data is null";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                response.ErrorMessage = "Exception in InsertUpdateTwitchVerifiedLink";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        /// <summary>
        /// Delete Twitch Verified Link data using the Twitch User ID
        /// </summary>
        [ResonanceAuth("TwitchVerifiedLink", requiredPermissionsAny: ConstantsPermissions.Amp.ServiceUpdate)]
        [HttpPost("worker/twitch-verified-link/delete-by-twitch-user-id")]
        public JsonResult DeleteByTwitchUserID([FromBody]TwitchDeletedUserListModel twitchDeletedUsers)
        {
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (Request.Headers != null)
                {
                    var keys = Request.Headers
                        .Where(x => x.Key.StartsWith("web-"))
                        .Select(x => $"{x.Key}={x.Value}");
                    if (keys.Count() > 0)
                    {
                        var headers = string.Join(",", keys);
                        Log.Info($"Received Delete Request: {headers}");
                    }
                }

                if (twitchDeletedUsers != null && twitchDeletedUsers.DeletedUsers != null && twitchDeletedUsers.DeletedUsers.Length > 0)
                {
                    var linkdata = twitchVerifiedLink.DeleteTwitchVerifiedLinkByTwitchUserIDs(twitchDeletedUsers);
                    if (linkdata != false)
                    {
                        response.Success = true;
                    }
                    else
                    {
                        response.ErrorMessage = "Link data is false";
                    }
                }
                else
                {
                    response.ErrorMessage = "Twitch Deleted Users is null or empty";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                response.ErrorMessage = "Exception in DeleteByTwitchUserID";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }
        /// <summary>
        /// Delete a Twitch Verified Link using a Youtube Tag
        /// </summary>
        [ResonanceAuth("TwitchVerifiedLink", requiredPermissionsAny: ConstantsPermissions.Amp.ServiceUpdate)]
        [HttpPost("worker/twitch-verified-link/delete-by-youtube-tag")]
        public JsonResult DeleteByYoutubeTag([FromBody]TwitchDeletedUserListModel twitchDeletedUsers)
        {
            var response = new ApiResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (Request.Headers != null)
                {
                    var keys = Request.Headers
                        .Where(x => x.Key.StartsWith("web-"))
                        .Select(x => $"{x.Key}={x.Value}");
                    if (keys.Count() > 0)
                    {
                        var headers = string.Join(",", keys);
                        Log.Info($"Received Delete Request: {headers}");
                    }
                }

                if (twitchDeletedUsers != null && twitchDeletedUsers.DeletedUsers != null && twitchDeletedUsers.DeletedUsers.Length > 0)
                {
                    var linkdata = twitchVerifiedLink.DeleteTwitchVerifiedLinkByYoutubeTags(twitchDeletedUsers);
                    if (linkdata != false)
                    {
                        response.Success = true;
                    }
                    else
                    {
                        response.ErrorMessage = "Link data is false";
                    }
                }
                else
                {
                    response.ErrorMessage = "Twitch Deleted User is null or empty";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                response.ErrorMessage = "Exception in DeleteByYoutubeTag";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }

        /// <summary>
        /// Rename a Twitch Verified Link user using the user ID.
        /// Process also updates the twitch_user_lookup table
        /// </summary>
        [ResonanceAuth("TwitchVerifiedLink", requiredPermissionsAny: ConstantsPermissions.Amp.ServiceUpdate)]
        [HttpPost("worker/twitch-verified-link/rename-twitch-login-by-twitch-user-id")]
        public JsonResult RenameTwitchLoginByTwitchUserID([FromBody]TwitchRenamedUserListModel data)
        {
            var response = new ApiListResponse<EmptyModel>()
            {
                RequestStartTime = DateTime.UtcNow,
                WorkerIdentifier = ConstantsWorker.WorkerIdentifier
            };

            try
            {
                if (Request.Headers != null)
                {
                    var keys = Request.Headers
                        .Where(x => x.Key.StartsWith("scanner- "))
                        .Select(x => $"{x.Key}={x.Value}");
                    if (keys.Count() > 0)
                    {
                        var headers = string.Join(",", keys);
                        Log.Info($"Received Bulk Rename Request: {headers}");
                    }
                }

                if (data != null && data.Data != null && data.Data.Length > 0)
                {
                    response.ReceivedRecords = data.Data.Length;

                    var processedCount = twitchVerifiedLink.RenameBulkTwitchVerifiedLink(Constants.DatabaseSchema, ref data);

                    response.ProcessedRecords = processedCount;
                    response.Success = true;

                    // Update the twitch user table with verified link users
                    var batchID = Guid.NewGuid().ToString().Replace("-", "");
                    var batchStart = DateTime.UtcNow;
                    var users = new TwitchUserListModel()
                    {
                        Data = data.Data
                        .Where(x => x.TwitchUserID > 0 && !string.IsNullOrWhiteSpace(x.NewTwitchLogin))
                        .Select(x => new TwitchUserModel()
                        {
                            TwitchUserID = x.TwitchUserID,
                            TwitchLogin = x.NewTwitchLogin,
                            IsActive = true,
                            InactiveReason = TwitchInactiveStatus.Active
                        })
                        .ToArray(),
                        WorkerIdentifier = ConstantsWorker.WorkerIdentifier,
                        BatchID = batchID,
                        BatchStart = batchStart
                    };
                    twitchUserMethods.InsertUpdateBulkTwitchUser(ConstantsWorker.WorkerIdentifier, batchID, batchStart, ConstantsWorker.WorkerS3Bucket, ConstantsWorker.WorkerS3Path, Constants.DatabaseSchema, users.Data.Distinct().ToArray());
                }
                else
                {
                    response.ErrorMessage = "Data is null or empty";
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                response.ErrorMessage = "Exception in RenameTwitchLoginByTwitchUserID";
            }
            finally
            {
                response.RequestEndTime = DateTime.UtcNow;
            }

            return new JsonResult(response);
        }
    }
}