﻿using Microsoft.VisualStudio.TestTools.UnitTesting;
using Newtonsoft.Json;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Models.ServiceModels.AtlasModels;
using System;
using Resonance.Core.Models.ApiModels;
using System.Diagnostics;
using Resonance.Core.Models.FilterModels;
using Resonance.Core.Models.DatabaseModels.AtlasModels;
using System.IO;
using Resonance.Core.Models.ApiModels.AtlasModels;
using System.Collections.Generic;
using Resonance.Core.Models.DatabaseModels.RequestModels;

namespace Resonance.Api.Test.Atlas
{
    /// <summary>
    /// For best results:
    /// - Have AMP, Atlas, Auth running
    /// - Be connected to Twitch VPN
    /// - Be connected to AWS VPN
    /// - Ensure your ApiKey from /permissions-admin/edit is located at the BearerToken path
    /// </summary>
    [TestClass]
    public class EventController
    {
        private static string BearerToken { get; set; }
        private static Random rand = new Random();

        public EventController()
        {
            BearerToken = File.ReadAllText(@"C:\Secrets\AnchorBearerToken.dat");
        }

        [TestMethod]
        public void GetProductList()
        {
            bool success = false;

            try
            {
                var postdata = new ListingFilter()
                {
                    Limit = 10,
                    Page = 0,
                    EventType = Core.Constants.AtlasInternalEventType.Product,
                    Columns = AtlasListingDefaults.ProductDefault(),
                    QueryFilters = new QueryFilter[1]
                    {
                        new QueryFilter()
                        {
                            FilterType = Core.Constants.FilterType.Contains,
                            Key = "product_name",
                            Value = "no"
                        }
                    },
                    SortOrder = new List<SortFilter>()
                    {
                        new SortFilter()
                        {
                            Column = "topic",
                            Ordinal = 0,
                            Ascending = true
                        },
                        new SortFilter()
                        {
                            Column = "sub_topic",
                            Ordinal = 1,
                            Ascending = true
                        },
                        new SortFilter()
                        {
                            Column = "format",
                            Ordinal = 2,
                            Ascending = true
                        },
                        new SortFilter()
                        {
                            Column = "sub_format",
                            Ordinal = 3,
                            Ascending = true
                        }
                    }.ToArray()
                };

                var response = WebRequestHelper.PostData<ListingFilter>($@"{TestConfig.ApiEndpointAtlas}event/get-product-list", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<AtlasListing>>(response.Data);
                    if(result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch(Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void EditProduct()
        {
            bool success = false;
            try
            {
                var data = new AtlasProduct()
                {
                    ProductName = $"Test Product {DateTime.UtcNow.ToString("yyyy-MM-dd")} {Guid.NewGuid().ToString("d")}",
                    ProductDescription = "Test Description",
                    Topic = "Test Topic",
                    SubTopic = "Test Sub Topic",
                    Format = "Test Format",
                    SubFormat = "Test sub topic",
                    ProductOwner = Environment.MachineName,
                    CreatedTime = DateTime.UtcNow,
                    CreatedUser = Environment.MachineName,
                    IsActive = true
                };
                var response = WebRequestHelper.PostData<AtlasProduct>(url: $@"{TestConfig.ApiEndpointAtlas}event/edit-product", bearer: BearerToken, data: ref data, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<EmptyModel>>(response.Data);
                    success = result.Success;
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetSeasonList()
        {
            bool success = false;

            try
            {
                var postdata = new ListingFilter()
                {
                    Limit = 10,
                    Page = 0,
                    EventType = Core.Constants.AtlasInternalEventType.Season,
                    Columns = AtlasListingDefaults.SeasonDefault(),
                    QueryFilters = new QueryFilter[1]
                    {
                        new QueryFilter()
                        {
                            FilterType = Core.Constants.FilterType.Contains,
                            Key = "season_name",
                            Value = "2020"
                        }
                    }
                };

                var response = WebRequestHelper.PostData<ListingFilter>($@"{TestConfig.ApiEndpointAtlas}event/get-season-list", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<AtlasListing>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void RequestUserAutoComplete()
        {
            bool success = false;

            try
            {
                var postdata = new RequestUserSearch()
                {
                    LoginNameSearch = "ryu"
                };

                var response = WebRequestHelper.PostData<RequestUserSearch>($@"{TestConfig.ApiEndpointAtlas}event/user-auto-complete", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiListResponse<RequestUserSearchUser>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetManagedChannels()
        {
            bool success = false;

            try
            {
                dynamic data = null;
                var response = WebRequestHelper.PostData<dynamic>($@"{TestConfig.ApiEndpointAtlas}event/get-managed-channels", bearer: BearerToken, data: ref data, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<GroupedOwnedChannelList>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetManagedCreators()
        {
            bool success = false;

            try
            {
                dynamic data = null;
                var response = WebRequestHelper.PostData<dynamic>($@"{TestConfig.ApiEndpointAtlas}event/get-managed-creators", bearer: BearerToken, data: ref data, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiListResponse<PremiumContentCreator>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void EditSeason()
        {
            bool success = false;
            try
            {
                var data = new AtlasSeason()
                {
                    ProductID = 1,
                    SeasonName = $"Test Season {DateTime.UtcNow.ToString("yyyy-MM-dd")} {Guid.NewGuid().ToString("d")}",
                    PrizePool = rand.Next(100, 100000),
                    PledgedHoursBroadcast = (float)rand.Next(0, 1000) / (float)rand.Next(1, 1000),
                    StartTime = DateTime.UtcNow,
                    EndTime = DateTime.UtcNow.Date.AddDays(1),
                    CreatedTime = DateTime.UtcNow,
                    CreatedUser = Environment.MachineName,
                    IsActive = true,
                    PreviousSeasonID = 42,
                    PreviousSeasonName = $"Test Previous Season {DateTime.UtcNow.ToString("yyyy-MM-dd")}"
                };
                var response = WebRequestHelper.PostData<AtlasSeason>($@"{TestConfig.ApiEndpointAtlas}event/edit-season", bearer: BearerToken, data: ref data, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<EmptyModel>>(response.Data);
                    success = result.Success;
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetEventList()
        {
            bool success = false;

            try
            {
                var postdata = new ListingFilter()
                {
                    Limit = 10,
                    Page = 0,
                    EventType = Core.Constants.AtlasInternalEventType.Event,
                    Columns = AtlasListingDefaults.EventDefault(),
                    QueryFilters = new QueryFilter[1]
                    {
                        new QueryFilter()
                        {
                            FilterType = Core.Constants.FilterType.Contains,
                            Key = "event_name",
                            Value = "a"
                        }
                    },
                    ExternalQueryFilters = new Dictionary<string, List<ExternalQueryFilter>>()
                    {
                        { "stream-channel-id", new List<ExternalQueryFilter>(){
                            new ExternalQueryFilter()
                            {
                                QueryFilters = new List<QueryFilter>()
                                {
                                    new QueryFilter()
                                    {
                                        Key = "channel_id",
                                        FilterType = Core.Constants.FilterType.Range,
                                        ValueArray = new dynamic[5]
                                        {
                                            1,
                                            2,
                                            3,
                                            4,
                                            5
                                        }
                                    }
                                }
                            }
                        } }
                    }
                };

                var response = WebRequestHelper.PostData<ListingFilter>($@"{TestConfig.ApiEndpointAtlas}event/get-event-list", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<AtlasListing>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void EditEvent()
        {
            bool success = false;
            try
            {
                var data = new AtlasEvent()
                {
                    EstimatedAverageCCV = rand.Next(1000, 100000),
                    EventName = $"Test Event {DateTime.UtcNow.ToString("yyyy-MM-dd")} {Guid.NewGuid().ToString("d")}",
                    EventType = "Demo",
                    PledgedHoursBroadcast = (float)rand.Next(0, 1000) / (float)rand.Next(1, 1000),
                    ProductID = rand.Next(0, 1000),
                    SeasonID = rand.Next(0, 1000),
                    StartTime = DateTime.UtcNow,
                    EndTime = DateTime.UtcNow.Date.AddDays(1),
                    CreatedTime = DateTime.UtcNow,
                    CreatedUser = Environment.MachineName,
                    Topic = "test topic",
                    SubTopic = "test sub topic",
                    Format = "test format",
                    SubFormat = "test sub format",
                    IsActive = true,
                    PremiumContentCreatorID = rand.Next(100000, 500000),
                    CostreamingSettings = "Test costream settings",
                    Distribution = true,
                    GameName = "Example game",
                    TwitchInvolvement = "Very involved"
                };
                var response = WebRequestHelper.PostData<AtlasEvent>($@"{TestConfig.ApiEndpointAtlas}event/edit-event", bearer: BearerToken, data: ref data, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<EmptyModel>>(response.Data);
                    success = result.Success;
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetSeasonEventList()
        {
            bool success = false;

            try
            {
                var postdata = new ListingFilter()
                {
                    Limit = 10,
                    Page = 0,
                    EventType = Core.Constants.AtlasInternalEventType.Season,
                    Columns = AtlasListingDefaults.SeasonDefault(),
                    QueryFilters = new QueryFilter[1]
                    {
                        new QueryFilter()
                        {
                            FilterType = Core.Constants.FilterType.Contains,
                            Key = "season",
                            Value = "2020"
                        }
                    }
                };

                var response = WebRequestHelper.PostData<ListingFilter>($@"{TestConfig.ApiEndpointAtlas}event/get-season-event-list", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<SeasonEventList>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetChannelEventList()
        {
            bool success = false;

            try
            {
                var postdata = new ListingFilter()
                {
                    Limit = 10,
                    Page = 0,
                    Columns = new string[1] { "*" },
                    QueryFilters = new QueryFilter[0] {}
                };

                var response = WebRequestHelper.PostData<ListingFilter>($@"{TestConfig.ApiEndpointAtlas}event/get-channel-event-list", bearer: BearerToken, data: ref postdata, timeout: 900, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<ChannelGroupedEventList>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetTopicEventList()
        {
            bool success = false;

            try
            {
                var postdata = new ListingFilter()
                {
                    Limit = 10,
                    Page = 0,
                    Columns = new string[1] { "*" },
                    QueryFilters = new QueryFilter[0] { }
                };

                var response = WebRequestHelper.PostData<ListingFilter>($@"{TestConfig.ApiEndpointAtlas}event/get-topic-event-list", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<GroupedEventList>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetFormatEventList()
        {
            bool success = false;

            try
            {
                var postdata = new ListingFilter()
                {
                    Limit = 10,
                    Page = 0,
                    Columns = new string[1] { "*" },
                    QueryFilters = new QueryFilter[0]{ }
                };

                var response = WebRequestHelper.PostData<ListingFilter>($@"{TestConfig.ApiEndpointAtlas}event/get-format-event-list", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<GroupedEventList>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetCategoryEventList()
        {
            bool success = false;

            try
            {
                var postdata = new ListingFilter()
                {
                    Limit = 10,
                    Page = 0,
                    Columns = new string[1] { "*" },
                    QueryFilters = new QueryFilter[0] {}
                };

                var response = WebRequestHelper.PostData<ListingFilter>($@"{TestConfig.ApiEndpointAtlas}event/get-category-event-list", bearer: BearerToken, data: ref postdata, timeout: 900, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<ChannelGroupedEventList>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetStreamList()
        {
            bool success = false;

            try
            {
                var postdata = new ListingFilter()
                {
                    Limit = 10,
                    Page = 0,
                    EventType = Core.Constants.AtlasInternalEventType.Stream,
                    Columns = AtlasListingDefaults.SeasonDefault(),
                    QueryFilters = new QueryFilter[1]
                    {
                        new QueryFilter()
                        {
                            FilterType = Core.Constants.FilterType.Contains,
                            Key = "stream_login",
                            Value = "no"
                        }
                    }
                };

                var response = WebRequestHelper.PostData<ListingFilter>($@"{TestConfig.ApiEndpointAtlas}event/get-stream-list", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<AtlasListing>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void EditStream()
        {
            bool success = false;
            try
            {
                var data = new AtlasStream()
                {
                    ChannelID = rand.Next(100000, 500000),
                    ChannelType = "partner",
                    EventID = rand.Next(0, 1000),
                    GameName = $"Test Game {DateTime.UtcNow.ToString("yyyy-MM-dd")} {Guid.NewGuid().ToString("d")}",
                    StreamLogin = "test login",
                    StartTime = DateTime.UtcNow,
                    EndTime = DateTime.UtcNow.Date.AddDays(1),
                    CreatedTime = DateTime.UtcNow,
                    CreatedUser = Environment.MachineName,
                    IsActive = true
                };
                var response = WebRequestHelper.PostData<AtlasStream>($@"{TestConfig.ApiEndpointAtlas}event/edit-stream", bearer: BearerToken, data: ref data, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<EmptyModel>>(response.Data);
                    success = result.Success;
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void EditContractAccountManagerMap()
        {
            bool success = false;
            try
            {
                var data = new AtlasContractAccountManagerMap()
                {
                    AccountManagerID = 1,
                    ContractID = 1,
                    CreatedTime = DateTime.UtcNow,
                    CreatedUser = Environment.MachineName,
                    IsActive = true
                };
                var response = WebRequestHelper.PostData<AtlasContractAccountManagerMap>($@"{TestConfig.ApiEndpointAtlas}event/edit-contract-account-manager-map", bearer: BearerToken, data: ref data, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<EmptyModel>>(response.Data);
                    success = result.Success;
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }


        [TestMethod]
        public void GetEventTypes()
        {
            bool success = false;

            try
            {
                var response = WebRequestHelper.GetData($@"{TestConfig.ApiEndpointAtlas}event/get-event-types", bearer: BearerToken, timeout: 30, isToken: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<Dictionary<string, int>>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetTopicsFormats()
        {
            bool success = false;

            try
            {
                var response = WebRequestHelper.GetData($@"{TestConfig.ApiEndpointAtlas}event/get-topics-formats", bearer: BearerToken, timeout: 30, isToken: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiListResponse<TopicListEntry>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetPremiumContentCreator()
        {
            bool success = false;

            try
            {
                var postdata = new ListingFilter()
                {
                    Limit = 10,
                    Page = 0,
                    EventType = Core.Constants.AtlasInternalEventType.PremiumContentCreator,
                    Columns = AtlasListingDefaults.PremiumContentCreatorDefault(),
                    QueryFilters = new QueryFilter[1]
                    {
                        new QueryFilter()
                        {
                            FilterType = Core.Constants.FilterType.Contains,
                            Key = "is_active",
                            Value = true
                        }
                    }
                };

                var response = WebRequestHelper.PostData<ListingFilter>($@"{TestConfig.ApiEndpointAtlas}event/get-premium-content-creator-list", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<AtlasListing>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void EditPremiumContentCreator()
        {
            bool success = false;
            try
            {
                var postdata = new PremiumContentCreator()
                {
                    PremiumContentCreatorName = $"Test {DateTime.UtcNow.ToString("yyyy-MM-dd")}",
                    PremiumContentCreatorType = "Test",
                    IsActive = true
                };
                var response = WebRequestHelper.PostData<PremiumContentCreator>($@"{TestConfig.ApiEndpointAtlas}event/edit-premium-content-creator", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<EmptyModel>>(response.Data);
                    success = result.Success;
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetAccountManager()
        {
            bool success = false;

            try
            {
                var postdata = new ListingFilter()
                {
                    Limit = 10,
                    Page = 0,
                    EventType = Core.Constants.AtlasInternalEventType.AccountManager,
                    Columns = AtlasListingDefaults.AccountManagerDefault(),
                    QueryFilters = new QueryFilter[1]
                    {
                        new QueryFilter()
                        {
                            FilterType = Core.Constants.FilterType.Contains,
                            Key = "account_manager_ldap_name",
                            Value = "r"
                        }
                    }
                };

                var response = WebRequestHelper.PostData<ListingFilter>($@"{TestConfig.ApiEndpointAtlas}event/get-account-manager-list", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<AtlasListing>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void EditAccountManager()
        {
            bool success = false;
            try
            {
                var postdata = new AtlasAccountManager()
                {
                    AccountManagerLdapName = "nstewart",
                    AccountManagerFirstName = "Nathan",
                    AccountManagerLastName = "Stewart",
                    ApprovingManagerAmazonID = "nsinkule",
                    AccountManagerEmail = "nstewart@twitch.tv",
                    ApprovingManagerName = "Nic Sinkule",
                    IsActive = true
                };
                var response = WebRequestHelper.PostData<AtlasAccountManager>($@"{TestConfig.ApiEndpointAtlas}event/edit-account-manager", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<EmptyModel>>(response.Data);
                    success = result.Success;
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetPccAmMap()
        {
            bool success = false;

            try
            {
                var postdata = new ListingFilter()
                {
                    Limit = 10,
                    Page = 0,
                    EventType = Core.Constants.AtlasInternalEventType.PccToAmMap,
                    Columns = AtlasListingDefaults.PccToAmMapDefault(),
                    QueryFilters = new QueryFilter[1]
                    {
                        new QueryFilter()
                        {
                            FilterType = Core.Constants.FilterType.Contains,
                            Key = "is_active",
                            Value = true
                        }
                    }
                };

                var response = WebRequestHelper.PostData<ListingFilter>($@"{TestConfig.ApiEndpointAtlas}event/get-pcc-am-map-list", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<AtlasListing>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void EditPccAmMap()
        {
            bool success = false;
            try
            {
                var postdata = new PremiumContentCreatorToAccountManagerMap()
                {
                    AccountManagerID = 1,
                    PremiumContentCreatorID = 1,
                    IsActive = true
                };
                var response = WebRequestHelper.PostData<PremiumContentCreatorToAccountManagerMap>($@"{TestConfig.ApiEndpointAtlas}event/edit-pcc-am-map", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<EmptyModel>>(response.Data);
                    success = result.Success;
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetPccChannelMap()
        {
            bool success = false;

            try
            {
                var postdata = new ListingFilter()
                {
                    Limit = 10,
                    Page = 0,
                    EventType = Core.Constants.AtlasInternalEventType.PccToChannelMap,
                    Columns = AtlasListingDefaults.PccToChannelMapDefault(),
                    QueryFilters = new QueryFilter[0] { }
                };

                var response = WebRequestHelper.PostData<ListingFilter>($@"{TestConfig.ApiEndpointAtlas}event/get-pcc-channel-map-list", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<AtlasListing>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void EditPccChannelMap()
        {
            bool success = false;
            try
            {
                var postdata = new PremiumContentCreatorToChannelMap()
                {
                    ChannelID = 1,
                    PremiumContentCreatorID = 1,
                    IsActive = true
                };
                var response = WebRequestHelper.PostData<PremiumContentCreatorToChannelMap>($@"{TestConfig.ApiEndpointAtlas}event/edit-pcc-channel-map", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<EmptyModel>>(response.Data);
                    success = result.Success;
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetContractList()
        {
            bool success = false;

            try
            {
                var postdata = new ListingFilter()
                {
                    Limit = 10,
                    Page = 0,
                    EventType = Core.Constants.AtlasInternalEventType.Contract,
                    Columns = AtlasListingDefaults.ContractDefault(),
                    QueryFilters = new QueryFilter[1]
                    {
                        new QueryFilter()
                        {
                            FilterType = Core.Constants.FilterType.Contains,
                            Key = "is_active",
                            Value = true
                        }
                    }
                };

                var response = WebRequestHelper.PostData<ListingFilter>($@"{TestConfig.ApiEndpointAtlas}event/get-product-list", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<AtlasListing>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void EditContract()
        {
            bool success = false;
            try
            {
                var postdata = new AtlasContract()
                {
                    ContractName = $"Test {DateTime.UtcNow.ToString("yyyy-MM-dd")} {Guid.NewGuid().ToString("d")}",
                    AdDensityPerHour = 0.15f,
                    AdPrerollsEnabled = true,
                    AdRevenueContractType = "cpm",
                    AdRevenueShareAmountTier1 = 75.0f,
                    AdRevenueShareAmountTier2 = 50.0f,
                    AdRevenueShareAmountTier3 = 25.0f,
                    AdRevenueShareTypeTier1 = "percent",
                    AdRevenueShareTypeTier2 = "percent",
                    AdRevenueShareTypeTier3 = "percent",
                    BitsRevenueSharePercent = 75.0f,
                    ContractType = "Test",
                    CreatedTime = DateTime.UtcNow,
                    CreatedUser = "Test",
                    EndDate = DateTime.UtcNow.AddDays(5).Date,
                    ExclusivityEnabled = true,
                    PremiumContentCreatorID = 1,
                    RevenueShareEnabled = true,
                    SponsorshipEnabled = true,
                    StartDate = DateTime.UtcNow.Date,
                    SubsRevenueSharePercent = 100.0f,
                    IsActive = true
                };
                var response = WebRequestHelper.PostData<AtlasContract>($@"{TestConfig.ApiEndpointAtlas}event/edit-contract", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<EmptyModel>>(response.Data);
                    success = result.Success;
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void GetContractChannelList()
        {
            bool success = false;

            try
            {
                var postdata = new ListingFilter()
                {
                    Limit = 10,
                    Page = 0,
                    EventType = Core.Constants.AtlasInternalEventType.ContractChannel,
                    Columns = AtlasListingDefaults.ContractChannelDefault(),
                    QueryFilters = new QueryFilter[1]
                    {
                        new QueryFilter()
                        {
                            FilterType = Core.Constants.FilterType.Contains,
                            Key = "channel_id",
                            Value = "1"
                        }
                    }
                };

                var response = WebRequestHelper.PostData<ListingFilter>($@"{TestConfig.ApiEndpointAtlas}event/get-season-event-list", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<AtlasListing>>(response.Data);
                    if (result.Metrics != null && !string.IsNullOrWhiteSpace(result.Metrics.MetricName) && response.Data != null)
                    {
                        success = result.Success;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }

        [TestMethod]
        public void EditContractChannel()
        {
            bool success = false;
            try
            {
                var postdata = new AtlasContractWithChannelList()
                {
                    AccountManagerIDs = new List<int>()
                    {
                        1
                    },
                    Channels = new List<AtlasContractChannelEntry>()
                    {
                        new AtlasContractChannelEntry()
                        {
                            Channel = $"Test Channel {DateTime.UtcNow.ToString("yyyy-MM-dd")}",
                            ChannelID = 1,
                            CustomStartDate = DateTime.UtcNow.Date.AddDays(1),
                            CustomEndDate = DateTime.UtcNow.AddDays(2)
                        }
                    },
                    Contract = new AtlasContract()
                    {
                        AdDensityPerHour = 0.5f,
                        AdPrerollsEnabled = true,
                        AdRevenueContractType = "TEST",
                        AdRevenueShareAmountTier1 = 1.55f,
                        AdRevenueShareAmountTier2 = 1.0f,
                        AdRevenueShareAmountTier3 = 1.0f,
                        AdRevenueShareTypeTier1 = "cpm",
                        AdRevenueShareTypeTier2 = "percent",
                        AdRevenueShareTypeTier3 = "percent",
                        BitsRevenueSharePercent = 1.0f,
                        ContractName = $"Test Contract {DateTime.UtcNow.ToString("yyyy-MM-dd")}",
                        ContractType = "Test",
                        ExclusivityEnabled = true,
                        IsActive = true,
                        PremiumContentCreatorID = 1,
                        RevenueShareEnabled = true,
                        SponsorshipEnabled = true,
                        SubsRevenueSharePercent = 1.0f,
                        StartDate = DateTime.UtcNow.AddDays(-1),
                        EndDate = DateTime.UtcNow.AddDays(3),
                        ContractID = 1
                    }
                };
                var response = WebRequestHelper.PostData<AtlasContractWithChannelList>($@"{TestConfig.ApiEndpointAtlas}event/edit-contract-with-channels", bearer: BearerToken, data: ref postdata, timeout: 30, isToken: true, needsAuth: true);
                if (response != null && response.IsSuccess)
                {
                    var result = JsonConvert.DeserializeObject<ApiResponse<EmptyModel>>(response.Data);
                    success = result.Success;
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
            }

            Assert.IsTrue(success);
        }
    }
}
