﻿using Microsoft.VisualStudio.TestTools.UnitTesting;
using Resonance.Api.Test.Jobs.Atlas.ContractAutoExtender.Mocks;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.DatabaseModels.AtlasModels;
using Resonance.Jobs.Atlas.ContractAutoExtender.Methods;
using Resonance.Jobs.Atlas.Contracts;
using System;
using System.Collections.Generic;

namespace Resonance.Api.Test.Jobs.Atlas.ContractAutoExtender
{
    [TestClass]
    public class ContractAutoExtenderJobTests
    {
        private ExtenderMethods Methods;
        private ContractAutoExtenderJob Job;
        public ContractAutoExtenderJobTests()
        {
            Methods = new ExtenderMethods(new MockExtenderQuery(), new MockActivityLogger());
            Job = new ContractAutoExtenderJob(new Core.Models.ConfigurationModels.Jobs.JobConfiguration { });
        }

        [TestMethod]
        public void NextRunIsSetTo24HoursAfterLastRun()
        {
            Job.RunJob(Methods, null, null);

            //next run should be 24 hours after last run
            Assert.IsTrue(Job.Config.NextRunTime.Value == Job.Config.LastRunTime.Value.AddHours(24));
        }

        [TestMethod]
        public void IsRunningIsFalseBeforeAndAfterRun()
        {
            Assert.IsFalse(Job.Config.IsRunning);
            Job.RunJob(Methods, null, null);
            Assert.IsFalse(Job.Config.IsRunning);
        }

        [TestMethod]
        public void JobReturnsIfNoContractsFound()
        {
            var infolist = new List<string>();
            var errorlist = new List<Exception>();

            Action<string> info = msg => { infolist.Add(msg); };
            Action<Exception> error = ex => { errorlist.Add(ex); };
            
            var methods = new ExtenderMethods(new MockExtenderQuery(new List<AtlasContract>()), new MockActivityLogger());                        
            var result = Job.RunJob(methods, info, error);

            Assert.AreEqual(result.ContractsUpdated, 0);
            Assert.AreEqual(result.ContractsFound, 0);
            Assert.AreEqual(infolist[0], ContractAutoExtenderJob.QueryingContractsMsg);
            Assert.AreEqual(infolist[1], ContractAutoExtenderJob.NoContractsFoundMsg);
            Assert.AreEqual(infolist[2], ContractAutoExtenderJob.DoneMsg);
        }

        [TestMethod]
        public void RunJobSucceeds()
        {
            var infolist = new List<string>();
            var errorlist = new List<Exception>();

            Action<string> info = msg => { infolist.Add(msg); };
            Action<Exception> error = ex => { errorlist.Add(ex); };

            DateTime end = DateTime.Today.AddDays(-7);
            DateTime start = end.AddDays(-30);

            List<AtlasContract> contracts = new List<AtlasContract>()
            {
                new AtlasContract { ContractID = 1, StartDate = start, EndDate = end, AutoRenew = true },
                new AtlasContract { ContractID = 2, StartDate = start, EndDate = end, AutoRenew = true },
                new AtlasContract { ContractID = 3, StartDate = start, EndDate = end, AutoRenew = true },
            };

            ExtenderMethods methods = new ExtenderMethods(new MockExtenderQuery(contracts), null);

            var result = Job.RunJob(methods, info, error);            

            Assert.AreEqual(result.ContractsFound, 3);
            Assert.AreEqual(result.ContractsExtended, 3);
            Assert.AreEqual(result.ContractsUpdated, 3);
        }

        [TestMethod]
        public void JobDoesNotExtendExpiredContractsWhenAutoRenewIsFalse()
        {
            DateTime end = DateTime.Today.AddDays(-7);
            DateTime start = end.AddDays(-30);

            List<AtlasContract> contracts = new List<AtlasContract>()
            {
                new AtlasContract { ContractID = 1, StartDate = start, EndDate = end, AutoRenew = false },
                new AtlasContract { ContractID = 2, StartDate = start, EndDate = end, AutoRenew = false },
                new AtlasContract { ContractID = 3, StartDate = start, EndDate = end, AutoRenew = false },
            };

            ExtenderMethods methods = new ExtenderMethods(new MockExtenderQuery(contracts), null);

            var result = Job.RunJob(methods, null, null);

            Assert.AreEqual(result.ContractsFound, 3);
            Assert.AreEqual(result.ContractsExtended, 0);
            Assert.AreEqual(result.ContractsUpdated, 0);
        }

        [TestMethod]
        public void JobDoesNotExtendNonExpiredContractsWhenAutoRenewIsTrue()
        {
            DateTime end = DateTime.Today.AddDays(7);
            DateTime start = end.AddDays(-30);

            List<AtlasContract> contracts = new List<AtlasContract>()
            {
                new AtlasContract { ContractID = 1, StartDate = start, EndDate = end, AutoRenew = true },
                new AtlasContract { ContractID = 2, StartDate = start, EndDate = end, AutoRenew = true },
                new AtlasContract { ContractID = 3, StartDate = start, EndDate = end, AutoRenew = true },
            };

            ExtenderMethods methods = new ExtenderMethods(new MockExtenderQuery(contracts), null);

            var result = Job.RunJob(methods, null, null);

            Assert.AreEqual(result.ContractsFound, 3);
            Assert.AreEqual(result.ContractsExtended, 0);
            Assert.AreEqual(result.ContractsUpdated, 0);
        }
    

        //Do a test run where it skips autorenew = false and non expired contracts
    }
}
