﻿using Microsoft.VisualStudio.TestTools.UnitTesting;
using Resonance.Api.Test.Jobs.Atlas.ContractAutoExtender.Mocks;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.DatabaseModels.AtlasModels;
using Resonance.Jobs.Atlas.ContractAutoExtender.Interfaces;
using Resonance.Jobs.Atlas.ContractAutoExtender.Methods;
using System;
using System.Collections.Generic;

namespace Resonance.Api.Test.Jobs.Atlas.ContractAutoExtender
{
    [TestClass]
    public class ExtenderMethodsTests
    {
        private ExtenderMethods Methods { get; set; }
        public ExtenderMethodsTests()
        {
            Methods = new ExtenderMethods(new MockExtenderQuery(), new MockActivityLogger());
        }

        [TestMethod]
        public void CorrectNumberOfContractsAreReturned()
        {
            IList<AtlasContract> contracts = new List<AtlasContract>
            {
                new AtlasContract
                {
                    ContractID = 1,
                    StartDate = new DateTime(2020, 1, 1),
                    EndDate = new DateTime(2020, 1, 31),
                    AutoRenew = true,
                },
                new AtlasContract
                {
                    ContractID = 2,
                    StartDate = new DateTime(2020, 1, 1),
                    EndDate = new DateTime(2020, 1, 31),
                    AutoRenew = true,
                }
            };

            IList<AtlasContract> extendedContracts = Methods.ExtendContracts(contracts);

            Assert.IsTrue(extendedContracts.Count == 2);
        }

        [TestMethod]
        public void ExtendContractsOriginalStartDateIsPopulated()
        {
            DateTime originalStartDate = new DateTime(2019, 1, 1);
            IList<AtlasContract> contracts = new List<AtlasContract>
            {
                new AtlasContract
                {
                    ContractID = 1,
                    StartDate = originalStartDate,
                    EndDate = new DateTime(2020,1,1),
                    AutoRenew = true,
                }
            };

            IList<AtlasContract> extendedContracts = Methods.ExtendContracts(contracts);

            Assert.IsNotNull(extendedContracts[0].OriginalStartDate);
            Assert.IsTrue(extendedContracts[0].OriginalStartDate == originalStartDate);
        }

        [TestMethod]
        public void ExtendContractsOriginalEndDateIsPopulated()
        {
            DateTime originalEndDate = new DateTime(2020, 1, 1);
            IList<AtlasContract> contracts = new List<AtlasContract>
            {
                new AtlasContract
                {
                    ContractID = 1,
                    StartDate = new DateTime(2019,1,1),
                    EndDate = originalEndDate,
                    AutoRenew = true,
                }
            };

            IList<AtlasContract> extendedContracts = Methods.ExtendContracts(contracts);

            DateTime expectedEndDate = contracts[0].EndDate.AddYears(1);

            Assert.IsNotNull(extendedContracts[0].OriginalEndDate);
            Assert.AreEqual(extendedContracts[0].OriginalEndDate, originalEndDate);
        }

        [TestMethod]
        public void ExtendContractsEndDateIsUpdated()
        {
            IList<AtlasContract> contracts = new List<AtlasContract>
            {
                new AtlasContract
                {
                    ContractID = 1,
                    StartDate = new DateTime(2019,1,1),
                    EndDate = new DateTime(2020,1,1),
                    AutoRenew = true,
                }
            };

            IList<AtlasContract> extendedContracts = Methods.ExtendContracts(contracts);
            DateTime expectedNewEndDate = new DateTime(2021, 1, 1);
            Assert.IsTrue(extendedContracts[0].EndDate == expectedNewEndDate);
        }

        [TestMethod]
        public void ExtendContractsDoesNotErrorOnNullContractsList()
        {
            Methods.ExtendContracts(null);
        }

        [TestMethod]
        public void NonAutoRenewContractsShouldNotModifyDates()
        {
            DateTime start = new DateTime(2020, 1, 1);
            DateTime end = new DateTime(2020, 1, 31);

            IList<AtlasContract> contracts = new List<AtlasContract>
            {
                new AtlasContract
                {
                    ContractID = 1,
                    AutoRenew = false,
                    StartDate = start,
                    EndDate = end,
                }
            };

            IList<AtlasContract> extendedContracts = Methods.ExtendContracts(contracts);

            Assert.AreEqual(extendedContracts.Count, 0);
        }

        [TestMethod]
        public void NonExpiredContractsShouldNotBeRenewedIfAutoRenewIsTrue()
        {
            DateTime end = DateTime.UtcNow.AddDays(7);
            DateTime start = end.AddYears(-1);

            IList<AtlasContract> contracts = new List<AtlasContract>
            {
                new AtlasContract
                {
                    ContractID = 1,
                    AutoRenew = true,
                    StartDate = start,
                    EndDate = end,
                }
            };

            IList<AtlasContract> extendedContracts = Methods.ExtendContracts(contracts);
            Assert.AreEqual(extendedContracts.Count, 0);
        }

        [TestMethod]
        public void ContractAutoRenewShouldBeSetToFalseAfterRenewing()
        {
            DateTime end = DateTime.UtcNow.AddDays(-7);
            DateTime start = end.AddYears(-1);

            IList<AtlasContract> contracts = new List<AtlasContract>
            {
                new AtlasContract
                {
                    ContractID = 1,
                    AutoRenew = true,
                    StartDate = start,
                    EndDate = end,
                }
            };

            IList<AtlasContract> extendedContracts = Methods.ExtendContracts(contracts);
            int updated = extendedContracts.Count;

            Assert.AreEqual(extendedContracts[0].AutoRenew, false);
        }

        [TestMethod]
        public void ActivityIsLoggedOnDatabaseUpdate()
        {
            MockActivityLogger logger = new MockActivityLogger();
            IContractExtender extender = new ExtenderMethods(new MockExtenderQuery(), logger);
            IList<AtlasContract> contracts = new List<AtlasContract>()
            {
                new AtlasContract { ContractID = 1, StartDate = new DateTime(2019, 1, 1), OriginalEndDate=new DateTime(2019,12,31), EndDate = new DateTime(2020, 12, 31) },
                new AtlasContract { ContractID = 2, StartDate = new DateTime(2019, 1, 1), OriginalEndDate=new DateTime(2019,6,30), EndDate = new DateTime(2020, 6, 30) },
                new AtlasContract { ContractID = 3, StartDate = new DateTime(2020, 3, 1), OriginalEndDate=new DateTime(2020,3,30), EndDate = new DateTime(2021, 3, 30) },
            };

            int result = extender.UpdateDatabase(contracts);

            Assert.IsTrue(logger.Messages.Count == contracts.Count);

            for (int i = 0; i < contracts.Count; i++)
            {
                AtlasContract contract = contracts[i];
                string expected = ExtenderMethods.GetSuccessUpdateMessage(contract.ContractID, contract.OriginalEndDate, contract.EndDate);
                string got = logger.Messages[i];

                Assert.AreEqual(got, expected);
            }
        }

        [TestMethod]
        public void ActivityErrorIsLoggedOnDatabaseUpdate()
        {
            MockActivityLogger logger = new MockActivityLogger();
            IContractExtender extender = new ExtenderMethods(new MockExtenderQuery(), logger);
            IList<AtlasContract> contracts = new List<AtlasContract>()
            {
                new AtlasContract { ContractID = 501, StartDate = new DateTime(2019, 1, 1), OriginalEndDate=new DateTime(2019,12,31), EndDate = new DateTime(2020, 12, 31) },
                new AtlasContract { ContractID = 502, StartDate = new DateTime(2019, 1, 1), OriginalEndDate=new DateTime(2019,6,30), EndDate = new DateTime(2020, 6, 30) },
                new AtlasContract { ContractID = 503, StartDate = new DateTime(2020, 3, 1), OriginalEndDate=new DateTime(2020,3,30), EndDate = new DateTime(2021, 3, 30) },
            };

            int result = extender.UpdateDatabase(contracts);

            Assert.IsTrue(logger.Messages.Count == contracts.Count);

            for (int i = 0; i < contracts.Count; i++)
            {
                AtlasContract contract = contracts[i];
                string expected = ExtenderMethods.GetFailedUpdateMessage(contract.ContractID);
                string got = logger.Messages[i];

                Assert.AreEqual(got, expected);
            }
        }
    }
}
