﻿using Amazon.CloudWatch;
using Microsoft.AspNetCore.Mvc.Filters;
using Resonance.Core.Exceptions;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.AuthHelpers;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StatsDHelpers;
using Resonance.Core.Helpers.StringHelpers;
using Resonance.Core.Models.AuthModels;
using System;
using System.Diagnostics;
using System.Linq;

namespace Resonance.Core.Attributes
{
    public class ResonanceAuth : ActionFilterAttribute
    {
        /// <summary>
        /// A user must belong to all of these permissions to pass authorization
        /// </summary>
        private string[] RequiredPermissions { get; set; }
        /// <summary>
        /// A user must belong to any of these permissions to pass authorization
        /// </summary>
        private string[] RequiredPermissionsAny { get; set; }
        
        private string ServiceName { get; set; }

        //private IDictionary<string, string> ApiKeysByIP { get; set; } = new Dictionary<string, string>();
        private static PermissionHelper permissionHelper { get; set; }

        public ResonanceAuth(string serviceName, string requiredPermissions = null, string requiredPermissionsAny = null) : base()
        {
            RequiredPermissions = requiredPermissions?.Split(',', StringSplitOptions.RemoveEmptyEntries);
            RequiredPermissionsAny = requiredPermissionsAny?.Split(',', StringSplitOptions.RemoveEmptyEntries);
            ServiceName = serviceName;
            permissionHelper = new PermissionHelper();
            permissionHelper.Initialize();
        }

        public override void OnResultExecuted(ResultExecutedContext context)
        {
            CloudwatchHelper.EnqueueMetricRequest("resonance_auth_onresultexecuted", 1, context.HttpContext, StandardUnit.Count);

            Log.Info("Executed: " + context.HttpContext.Response.ContentLength);
            base.OnResultExecuted(context);
        }

        public override void OnActionExecuting(ActionExecutingContext context)
        {
            var request = context.HttpContext.Request;
            Stopwatch actionExecutingStopWatch = new Stopwatch();
            actionExecutingStopWatch.Start();
            try
            {
                var resonancePermissions = permissionHelper.GetUserPermissions(context.HttpContext);
                
                // Resonance requires at least one role whitelisting, either through api keys or whitelisted ldap content
                if (resonancePermissions == null || resonancePermissions.Count == 0)
                {
                    CloudwatchHelper.EnqueueMetricRequest("auth_exception", 1, context.HttpContext, StandardUnit.Count);
                    throw new AuthException(context);
                }
                if (RequiredPermissions != null && RequiredPermissions.Length > 0 && RequiredPermissions.Any(g => !resonancePermissions.Contains(g)))
                {
                    CloudwatchHelper.EnqueueMetricRequest("auth_exception", 1, context.HttpContext, StandardUnit.Count);
                    throw new AuthException(context);
                }
                if (RequiredPermissionsAny != null && RequiredPermissionsAny.Length > 0 && RequiredPermissionsAny.Where(g => resonancePermissions.Contains(g)).Count() == 0)
                {
                    CloudwatchHelper.EnqueueMetricRequest("auth_exception", 1, context.HttpContext, StandardUnit.Count);
                    throw new AuthException(context);
                }
            }
            catch (AuthException ex)
            {
                Log.Error(ex, "resonance-auth-onactionexecuting", context: context.HttpContext);
                CloudwatchHelper.EnqueueMetricRequest("auth_exception", 1, context.HttpContext, StandardUnit.Count);
                throw;
            }
            // We shouldn't hit unhandled exceptions unless it's truly unhandled. Throw a relevant auth error if it's an expected failure option
            catch (Exception ex)
            {
                CloudwatchHelper.EnqueueMetricRequest("auth", 1, context.HttpContext, StandardUnit.Count);
                Log.Error(ex, "resonance-auth-onactionexecuting", context: context.HttpContext);
                throw new AuthFailureException(context);
            }
            finally
            {
                actionExecutingStopWatch.Stop();
                var statsdname = $"{request.Path.Value}";
                if (!string.IsNullOrWhiteSpace(statsdname))
                {
                    statsdname = statsdname.Length > 1
                        ? statsdname.Replace("/", "_").Replace(".", "_").Substring(1)
                        : "root";
                    string originalpath = (context.HttpContext.Items["originalPath"] as string) ?? "";
                    AuthTokenData token = ((AuthTokenData)context.HttpContext.Items[UserAuthDataContext.AuthTokenDataKey]) ?? null;
                    CloudwatchHelper.EnqueueMetricRequest("auth_check", actionExecutingStopWatch.Elapsed.Milliseconds, context.HttpContext, StandardUnit.Milliseconds);
                    StatsDHelper.Counter(measurement: $"auth_check", measurementType: "access", val: 1, location: statsdname, additionalTags: $",http_method={request.Method},original_path={originalpath},user_id={token?.User}");
                    StatsDHelper.Counter(measurement: $"auth_check", measurementType: "elapsed", val: actionExecutingStopWatch.Elapsed.Milliseconds, location: statsdname, additionalTags: $",http_method={request.Method},original_path={originalpath},user_id={token?.User}");
                    Log.Info("Auth Check took " + actionExecutingStopWatch.ElapsedMilliseconds.ToString() + " MS");
                }
            }
            base.OnActionExecuting(context);
        }
    }
}
