﻿using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.FileHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ConfigurationModels;
using Resonance.Core.Models.ConfigurationModels.Permissions;
using System;
using System.ComponentModel;
using System.Text.RegularExpressions;

namespace Resonance.Core
{
    public static class Constants
    {
        private const string unloadManifestPattern = @"(?:s3:\/\/crs-data-export\/)(.*)?";
        public static Regex UnloadManifestRegex = new Regex(unloadManifestPattern, RegexOptions.Compiled | RegexOptions.Singleline | RegexOptions.IgnoreCase);
        public static string ApplicationVersion = "x.x.x.x";
        public static ApplicationConfigurationModel AppConfig { get; set; }
        //public static string Environment { get; set; }
        public static string ServiceName { get; set; }
        public static string DatabaseSchema { get; set; }
        public const string RedshiftDateFormat = "YYYY-MM-DD HH24:MI:SS";
        public const string RedshiftDateFormatToString = "yyyy-MM-dd HH:mm:ss";
        public static string ApiKey { get; set; }
        public static PermissionConfigurationModel Permissions { get; set; }
        private static PermissionHelper permissionHelper { get; set; }
        public static string GqlEndpoint { get { return "https://gql.twitch.tv/gql"; } }

        public static void InitConstants(ApplicationConfigurationModel config, string serviceName)
        {
            AppConfig = config;
            ServiceName = serviceName;
            ApiKey = null;

            if (config.Application.Environment == "Development")
            {
                DatabaseSchema = "development.";
            }
            else if (config.Application.Environment == "Staging")
            {
                DatabaseSchema = "staging.";
            }
            else if (config.Application.Environment == "Production")
            {
                DatabaseSchema = "production.";
            }
            else
            {
                throw new NotImplementedException($@"{config.Application.Environment} does not have defined handling in Resonance.Core.Constants");
            }

            permissionHelper = new PermissionHelper();
            permissionHelper.Initialize();
            Permissions = permissionHelper.GetPermissionsFromS3(AppConfig.Application.Name);
            try
            {
                var path = FileHelper.GetRelativePath("version.conf");
                if (System.IO.Directory.Exists(System.IO.Path.GetDirectoryName(path)) && System.IO.File.Exists(path))
                {
                    ApplicationVersion = System.IO.File.ReadAllText(path);
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
        }

        public enum DBProjectSqlServer
        {
            Unknown = 0,
            Yttrium = 1,
            MoneyBall = 2,
            ImpCn = 3
        }

        public enum DBProjectRedshift
        {
            Unknown = 0,
            CrsData = 1,
            TwitchScience = 2,
            TwitchPayments = 3
        }

        public enum TwitchInactiveStatus
        {
            Active = -1,
            Unknown = 0,
            SelfDeleted = 1,
            BannedUndefined = 2,
            DMCA = 3,
            TOS = 4
        }

        public enum AggregationType
        {
            Lifetime = -1,
            Unknown = 0,
            Week = 7,
            ThirtyDay = 30,
            SixtyDay = 60,
            NintyDay = 90,
            BankerYear = 360,
            Year = 365
        }

        public enum AggregationGranularity
        {
            Unknown = 0,
            Daily = 1,
            Weekly = 2,
            Monthly = 3,
            Quarterly = 4,
            Yearly = 5
        }

        public enum FilterType
        {
            Unknown = 0,
            Exact = 1,
            Range = 2,
            StartsWith = 3,
            Contains = 4,
            In = 5,
            LessThan = 6,
            GreaterThan = 7,
            Regex = 8,
            Json = 9,
            NotEqual = 10
        }

        public enum PermissionFormType
        {
            Unknown = 0,
            
            NewRole = 10,
            EditRole = 11,
            DeleteRole = 12,

            NewPermission = 20,
            EditPermission = 21,
            DeletePermission = 22,
            
            NewLdapGroup = 30,
            EditLdapGroup = 31,
            DeleteLdapGroup = 32,

            NewLdapName = 40,
            EditLdapName = 41,
            DeleteLdapName = 42,

            NewToken = 51,
            EditToken = 52,
            DeleteToken = 53,

            NewRolePermissionMap =  61,
            EditRolePermissionMap = 61,
            DeleteRolePermissionMap = 63,

            NewRoleLdapGroupMap = 71,
            EditRoleLdapGroupMap = 72,
            DeleteRoleLdapGroupMap = 73,

            NewRoleTokenMap = 81,
            EditRoleTokenMap = 82,
            DeleteRoleTokenMap = 83,

            NewRoleLdapNameMap = 91,
            EditRoleLdapNameMap = 92,
            DeleteRoleLdapNameMap = 93,

            NewModel = 101,
            EditModel = 102,
            DeleteModel = 103
        }

        public enum ScorecardPriority
        {
            None = 0,
            Low = 1,
            Normal = 2,
            High = 3
        }

        public enum SiteID
        {
            Unknown = 0,
            Amp = 1,
            Atlas = 2,
            Explorer = 3,
        }

        public enum OAuthType
        {
            Unknown = 0,
            TwitchSSO = 1,
            TwitchOAuth = 2,
            AmazonOAuth = 3,
            GoogleOAuth = 4
        }

        public enum AssetType
        {
            Unknown = 0,
            Image = 1,
            Video = 2,
            Html = 3,
            Js = 4,
            Css = 5
        }

        public enum AtlasInternalEventType
        {
            Unknown = 0,
            Product = 1,
            Season = 2,
            Event = 3,
            Stream = 4,
            AccountManager = 5,
            PremiumContentCreator = 6,
            PccToAmMap = 7,
            PccToChannelMap = 8,
            Contract = 9,
            ContractChannel = 10,
            ContractAccountManagerMap = 11
        }

        public enum AtlasEventChannelType
        {
            [Description("Undefined")]
            Unknown = 0,
            [Description("A channel owned and operated by the organizer, used exclusively for airing an organizer's content")]
            PCCOwnedPrimary = 1,
            [Description("A channel owned and operated by the organizer, used exclusively for airing an organizer's content, specialized into a different language from a primary channel")]
            PCCOwnedSupplemental = 2,
            [Description("A non-owned channel who has partnered with the organizer to provide the official languange localization of otherwise unaltered broadcast content (either a mirror or a \"clean feed\")")]
            ThirdPartyNonUGCSupplemental = 3,
            [Description("A non-owned channel who has partnered with the organizer to provide original commentary for an event")]
            CoStreamAuthorized = 4,
            [Description("A channel run by a UGC-creator who is participating in the associated event and broadcasting original/first-person gameplay and providing original content (e.g. Twitch Rivals participants)")]
            UGCParticipant = 5,
            [Description("A non-owned channel who, independently of the organizer - is providing original/organic commentary using the event as a subject.")]
            CoStreamOrganic = 6,
            [Description("A channel run by a UGC-creator who is acting as a host, co-host or interviewee for the primary broadcast (not co-streamed) (e.g. T-Pain hosting GlitchCon)")]
            UGCHost = 7,
        }

        public enum AtlasEventStatus
        {
            [Description("Undefined")]
            Unknown = 0,
            [Description("An event planned for a future date.")]
            Planned = 1,
            [Description("A future-dated event which, barring extraneous circumstances, is happening.")]
            Confirmed = 2,
            [Description("An event which has already taken place.")]
            Completed = 3
        }

        public enum AtlasEventType
        {
            [Description("Undefined")]
            Unknown = 0,
            [Description("Predictable and frequently-occurring (e.g., weekly) set of matches played for league standings.")]
            RegularSeason = 1,
            [Description("Tournament to determine whether a player/team will lose their place in a league to another up-and-coming team. (e.g. \"Relegation\")")]
            RegularSeasonQualifier = 2,
            [Description("Tournament to determine eligibility for a seed in the playoffs or for tournament standing. (e.g. \"Group Stage\", \"Open Division\")")]
            TournamentPlayoffQualifier = 3,
            [Description("Series of matches (either the culmination of a season, or a free-standing event) leading to a Finals and ultimate victor.")]
            TournamentPlayoffFinals = 4,
            [Description("A zero-stakes competitive event often featuring celebrities / all-stars / retired ethletes, or perhaps using a novel game mode.")]
            ExhibitionShowmatch = 5,
            [Description("A broadcast featuring a convention or expo. (Game selection not required for this event type)")]
            ConsAndExpos = 6,
            [Description("An announcement stream or showcase around upcoming game content, which may include competitive gameplay, but is zero-stakes.")]
            DeveloperUpdate = 7,
            [Description("Pre and post-event programming which serves as a companion to live events.")]
            ShoulderContent = 8
        }

        /// <summary>
        /// Possible states an upload can be in.
        /// </summary>
        public enum TwitchUploadServiceStatus
        {
            /// <summary>
            /// Create() has been called, but the file has not been updated yet.
            /// </summary>
            REQUESTED = 0,
            /// <summary>
            /// A worker has picked up the job to postprocess the file.
            /// </summary>
            POSTPROCESS_STARTED = 1,
            /// <summary>
            /// Postprocessing has completed and the callback has been made, waiting for the feature
            /// service to call SetStatus(COMPLETE).
            /// </summary>
            POSTPROCESS_COMPLETE = 2,
            /// <summary>
            /// The whole process is complete. Feature services should set this status when they are done
            /// processing the callback and the whole process is complete. We will notify over pubsub when this status is
            /// received over SetStatus. Please ensure your `message` parameter is safe for being user-visible.
            /// </summary>
            COMPLETE = 3,
            /// <summary>
            /// This status is deprecated and will not be set, see below
            /// </summary>
            VALIDATION_FAILED = 4,
            /// <summary>
            /// This status is deprecated and will not be set, see below
            /// </summary>
            POSTPROCESS_FAILED = 5,
            /// <summary>
            /// Postprocessing has failed and will be retried.
            /// </summary>
            POSTPROCESS_RETRYING = 6,
            /// <summary>
            /// Feature services should set this status if there is a failure on their end that will not be retried.
            /// We will notify over pubsub when this status is received over SetStatus.
            /// Please ensure your `message` parameter is safe for being user-visible.
            /// </summary>
            FEATURE_SERVICE_FAILED = 7,
            /// <summary>
            /// Failure Codes:
            ///  x1x is an image-related failure
            ///  x2x is a non-image-related failure
            ///  1xx is a validation failure
            ///  2xx is a processing failure
            /// User error / validation failures: we are confident the input is bad.
            /// These errors will not be retried
            /// An image is required but a non-image was uploaded. Triggered by validations or transformations
            /// </summary>
            IS_IMAGE_VALIDATION_FAILED = 110,
            /// <summary>
            /// Aspect ratio validation failed. Aspect ratio did not follow constraints
            /// </summary>
            ASPECT_RATIO_VALIDATION_FAILED = 111,
            /// <summary>
            /// Width validation failed. Width did not follow constraints
            /// </summary>
            WIDTH_VALIDATION_FAILED = 112,
            /// <summary>
            /// Height validation failed. Height did not follow constraints
            /// </summary>
            HEIGHT_VALIDATION_FAILED = 113,
            /// <summary>
            /// Image format validation failed. The file was an image of the wrong format
            /// </summary>
            IMAGE_FORMAT_VALIDATION_FAILED = 114,
            /// <summary>
            /// Image format validation failed. The file was an image of the wrong format
            /// </summary>
            ZIP_FORMAT_VALIDATION_FAILED = 115,
            /// <summary>
            /// File size validation failed. The file size was too large
            /// </summary>
            FILE_SIZE_VALIDATION_FAILED = 120,
            /// <summary>
            /// Processing errors: probably not the user's fault. Could be parameters to CreateUpload or an error on our end.
            /// Internally, the errors causing this status will be retried several times in case they are transient. During
            /// that process the status will be POSTPROCESS_RETRYING.
            /// Once one of these statuses has been set, we are done retrying.
            ///​
            /// Transformations on a valid image failed
            /// </summary>
            TRANSFORMATION_FAILED = 210,
            /// <summary>
            /// The worker failed for reasons other than validations or transformations, including file I/O and AWS calls
            /// </summary>
            PROCESSING_FAILED = 220
        }

        /// <summary>
        /// Remove transparency information from the image, used with Transcode message.
        /// TRUE will cause 1%-99% transparency to become 100% transparency, and
        /// 0% transparency to become black. WHITE will overlay the image on a white background,
        /// meaning that transparency gradients will be preserved, and 0% transparency will be white.
        /// </summary>
        public enum TwitchUploadServiceRemoveAlpha
        {
            /// <summary>
            /// Default behavior: TRUE if format = jp(e)g.
            /// </summary>
            DEFAULT = 0,
            /// <summary>
            /// Do not remove alpha channel.
            /// </summary>
            FALSE = 1,
            /// <summary>
            /// Remove alpha channel.
            /// </summary>
            TRUE = 2,
            /// <summary>
            /// Overlay the image on a black background.
            /// </summary>
            BLACK = 3,
            /// <summary>
            /// Overlay the image on a white background.
            /// </summary>
            WHITE = 4
        }

        public enum ValidEnvironments
        {
            Unknown = 0,
            Development = 1,
            Staging = 2,
            Production = 3,
            DevelopmentBeta = 4
        }
    }
}
