﻿using Resonance.Core.Models.ConfigurationModels.Permissions;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;

namespace Resonance.Core.Extensions
{
    public static class PermissionExtensions
    {
        private const string rawPermissionsPattern = "([a-z0-9-]+)\\|(.*?)\\|(.*?)$";
        private static Regex rawPermissionsRegex = new Regex(rawPermissionsPattern, RegexOptions.Compiled | RegexOptions.Singleline | RegexOptions.IgnoreCase);
        private const string rawTokensPattern = "([a-z0-9-]+)\\|(.*?)\\|(.*?)\\|(.*?)$";
        private static Regex rawTokensRegex = new Regex(rawTokensPattern, RegexOptions.Compiled | RegexOptions.Singleline | RegexOptions.IgnoreCase);
        private const string rawRolesPattern = "([a-z0-9-]+)\\|(.*?)\\|(.*?)$";
        private static Regex rawRolesRegex = new Regex(rawRolesPattern, RegexOptions.Compiled | RegexOptions.Singleline | RegexOptions.IgnoreCase);
        private const string rawLdapGroupsPattern = "([a-z0-9-]+)\\|(.*?)\\|(.*?)$";
        private static Regex rawLdapGroupsRegex = new Regex(rawLdapGroupsPattern, RegexOptions.Compiled | RegexOptions.Singleline | RegexOptions.IgnoreCase);
        private const string rawLdapUsersPattern = "([a-z0-9-]+)\\|(.*?)\\|(.*?)$";
        private static Regex rawLdapUsersRegex = new Regex(rawLdapUsersPattern, RegexOptions.Compiled | RegexOptions.Singleline | RegexOptions.IgnoreCase);
        private const string rawColumnPermissionsPattern = "([a-z0-9-]+)\\|(.*?)\\|(.*?)\\|(.*?)$";
        private static Regex rawColumnPermissionsRegex = new Regex(rawColumnPermissionsPattern, RegexOptions.Compiled | RegexOptions.Singleline | RegexOptions.IgnoreCase);
        private const string mappingRoleLdapGroupPattern = "([a-z0-9-]+)\\|(.*?)$";
        private static Regex mappingRoleLdapGroupRegex = new Regex(mappingRoleLdapGroupPattern, RegexOptions.Compiled | RegexOptions.Singleline | RegexOptions.IgnoreCase);
        private const string mappingRoleLdapUserPattern = "([a-z0-9-]+)\\|(.*?)$";
        private static Regex mappingRoleLdapUserRegex = new Regex(mappingRoleLdapUserPattern, RegexOptions.Compiled | RegexOptions.Singleline | RegexOptions.IgnoreCase);
        private const string mappingRoleTokenPattern = "([a-z0-9-]+)\\|(.*?)$";
        private static Regex mappingRoleTokenRegex = new Regex(mappingRoleTokenPattern, RegexOptions.Compiled | RegexOptions.Singleline | RegexOptions.IgnoreCase);
        private const string mappingRolePermissionPattern = "([a-z0-9-]+)\\|(.*?)$";
        private static Regex mappingRolePermissionRegex = new Regex(mappingRolePermissionPattern, RegexOptions.Compiled | RegexOptions.Singleline | RegexOptions.IgnoreCase);

        public static List<PermissionModel> GetRawPermissions(this string data)
        {
            var permissionList = new List<PermissionModel>(); // Should not contain duplicates

            if (!string.IsNullOrWhiteSpace(data))
            {
                var lines = data.Split(new[] { "\r\n", "\r", "\n" }, StringSplitOptions.None);
                foreach(var line in lines)
                {
                    if(!string.IsNullOrWhiteSpace(line) && rawPermissionsRegex.IsMatch(line))
                    {
                        var match = rawPermissionsRegex.Match(line);
                        if (match.Success)
                        {
                            var permission = new PermissionModel()
                            {
                                PermissionID = match?.Groups[1]?.Value,
                                PermissionName = match?.Groups[2]?.Value,
                                PermissionDescription = match?.Groups[3]?.Value
                            };
                            if (!string.IsNullOrWhiteSpace(permission.PermissionID) && !string.IsNullOrWhiteSpace(permission.PermissionName))
                            {
                                permissionList.Add(permission);
                            }
                        }
                    }
                }
            }

            return permissionList;
        }

        public static List<TokenModel> GetRawTokens(this string data)
        {
            var tokenList = new List<TokenModel>();
            if (!string.IsNullOrWhiteSpace(data))
            {
                var lines = data.Split(new[] { "\r\n", "\r", "\n" }, StringSplitOptions.None);
                foreach (var line in lines)
                {
                    if (!string.IsNullOrWhiteSpace(line) && rawTokensRegex.IsMatch(line))
                    {
                        var match = rawTokensRegex.Match(line);
                        if (match.Success)
                        {
                            var token = new TokenModel()
                            {
                                TokenID = match?.Groups[1]?.Value,
                                TokenName = match?.Groups[2]?.Value,
                                TokenDescription = match?.Groups[3]?.Value,
                                WhitelistedIPSource = match?.Groups[4]?.Value?.Split(',')
                            };
                            if (!string.IsNullOrWhiteSpace(token.TokenID) && !string.IsNullOrWhiteSpace(token.TokenName))
                            {
                                tokenList.Add(token);
                            }
                        }
                    }
                }
            }
            return tokenList;
        }

        public static List<RoleModel> GetRawRoles(this string data)
        {
            var roleList = new List<RoleModel>();
            if (!string.IsNullOrWhiteSpace(data))
            {
                var lines = data.Split(new[] { "\r\n", "\r", "\n" }, StringSplitOptions.None);
                foreach (var line in lines)
                {
                    if (!string.IsNullOrWhiteSpace(line) && rawRolesRegex.IsMatch(line))
                    {
                        var match = rawRolesRegex.Match(line);
                        if (match.Success)
                        {
                            var role = new RoleModel()
                            {
                                RoleID = match?.Groups[1]?.Value,
                                RoleName = match?.Groups[2]?.Value,
                                RoleDescription = match?.Groups[3]?.Value
                            };
                            if (!string.IsNullOrWhiteSpace(role.RoleID) && !string.IsNullOrWhiteSpace(role.RoleName))
                            {
                                roleList.Add(role);
                            }
                        }
                    }
                }
            }
            return roleList;
        }

        public static List<LdapGroupModel> GetRawLdapGroups(this string data)
        {
            var ldapGroupList = new List<LdapGroupModel>();
            if (!string.IsNullOrWhiteSpace(data))
            {
                var lines = data.Split(new[] { "\r\n", "\r", "\n" }, StringSplitOptions.None);
                foreach (var line in lines)
                {
                    if (!string.IsNullOrWhiteSpace(line) && rawLdapGroupsRegex.IsMatch(line))
                    {
                        var match = rawLdapGroupsRegex.Match(line);
                        if (match.Success)
                        {
                            var group = new LdapGroupModel()
                            {
                                LdapGroupId = match?.Groups[1]?.Value,
                                LdapGroupName = match?.Groups[2]?.Value,
                                LdapGroupDescription = match?.Groups[3]?.Value
                            };
                            if (!string.IsNullOrWhiteSpace(group.LdapGroupId) && !string.IsNullOrWhiteSpace(group.LdapGroupName))
                            {
                                ldapGroupList.Add(group);
                            }
                        }
                    }
                }
            }
            return ldapGroupList;
        }

        public static List<LdapNameModel> GetRawLdapUsers(this string data)
        {
            var ldapUserList = new List<LdapNameModel>();
            if (!string.IsNullOrWhiteSpace(data))
            {
                var lines = data.Split(new[] { "\r\n", "\r", "\n" }, StringSplitOptions.None);
                foreach (var line in lines)
                {
                    if (!string.IsNullOrWhiteSpace(line) && rawLdapUsersRegex.IsMatch(line))
                    {
                        var match = rawLdapUsersRegex.Match(line);
                        if (match.Success)
                        {
                            var user = new LdapNameModel()
                            {
                                LdapNameID = match?.Groups[1]?.Value,
                                LdapName = match?.Groups[2]?.Value,
                                LdapNameDescription = match?.Groups[3]?.Value
                            };
                            if (!string.IsNullOrWhiteSpace(user.LdapNameID) && !string.IsNullOrWhiteSpace(user.LdapName))
                            {
                                ldapUserList.Add(user);
                            }
                        }
                    }
                }
            }
            
            return ldapUserList;
        }

        public static List<ColumnPermissionModel> GetRawColumnPermissions(this string data)
        {
            var columnPermissionsList = new List<ColumnPermissionModel>();
            if (!string.IsNullOrWhiteSpace(data))
            {
                var lines = data.Split(new[] { "\r\n", "\r", "\n" }, StringSplitOptions.None);
                foreach (var line in lines)
                {
                    if (!string.IsNullOrWhiteSpace(line) && rawColumnPermissionsRegex.IsMatch(line))
                    {
                        var match = rawColumnPermissionsRegex.Match(line);
                        if (match.Success)
                        {
                            var columnPermission = new ColumnPermissionModel()
                            {
                                ModelID = match?.Groups[1]?.Value,
                                ColumnName = match?.Groups[2]?.Value,
                                ModelDescription = match?.Groups[3]?.Value,
                                RequiredPermissionList = match?.Groups[4]?.Value?.Split(",")?.ToList()
                            };
                            if (!string.IsNullOrWhiteSpace(columnPermission.ModelID) && !string.IsNullOrWhiteSpace(columnPermission.ColumnName))
                            {
                                columnPermissionsList.Add(columnPermission);
                            }
                        }
                    }
                }
            }
            return columnPermissionsList;
        }

        public static Dictionary<string, List<string>> GetRoleToLdapGroupMappings(this string data)
        {
            var roleToLdapGroupMapping = new Dictionary<string, List<string>>();
            if (!string.IsNullOrWhiteSpace(data))
            {
                var lines = data.Split(new[] { "\r\n", "\r", "\n" }, StringSplitOptions.None);
                foreach (var line in lines)
                {
                    if (!string.IsNullOrWhiteSpace(line) && mappingRoleLdapGroupRegex.IsMatch(line))
                    {
                        var match = mappingRoleLdapGroupRegex.Match(line);
                        if (match.Success)
                        {
                            var key = match?.Groups[1]?.Value;
                            var value = match?.Groups[2]?.Value?.Split(",").ToList();
                            if (!string.IsNullOrWhiteSpace(key))
                            {
                                roleToLdapGroupMapping.Add(key, value);
                            }
                        }
                    }
                }
            }
            return roleToLdapGroupMapping;
        }

        public static Dictionary<string, List<string>> GetRoleToLdapUserMappings(this string data)
        {
            var roleToLdapUserMapping = new Dictionary<string, List<string>>();
            if (!string.IsNullOrWhiteSpace(data))
            {
                var lines = data.Split(new[] { "\r\n", "\r", "\n" }, StringSplitOptions.None);
                foreach (var line in lines)
                {
                    if (!string.IsNullOrWhiteSpace(line) && mappingRoleLdapUserRegex.IsMatch(line))
                    {
                        var match = mappingRoleLdapUserRegex.Match(line);
                        if (match.Success)
                        {
                            var key = match?.Groups[1]?.Value;
                            var value = match?.Groups[2]?.Value?.Split(",").ToList();
                            if (!string.IsNullOrWhiteSpace(key))
                            {
                                roleToLdapUserMapping.Add(key, value);
                            }
                        }
                    }
                }
            }
            return roleToLdapUserMapping;
        }

        public static Dictionary<string, List<string>> GetRoleToTokenMappings(this string data)
        {
            var roleToTokenMapping = new Dictionary<string, List<string>>();
            if (!string.IsNullOrWhiteSpace(data))
            {
                var lines = data.Split(new[] { "\r\n", "\r", "\n" }, StringSplitOptions.None);
                foreach (var line in lines)
                {
                    if (!string.IsNullOrWhiteSpace(line) && mappingRoleTokenRegex.IsMatch(line))
                    {
                        var match = mappingRoleTokenRegex.Match(line);
                        if (match.Success)
                        {
                            var key = match?.Groups[1]?.Value;
                            var value = match?.Groups[2]?.Value?.Split(",").ToList();
                            if (!string.IsNullOrWhiteSpace(key))
                            {
                                roleToTokenMapping.Add(key, value);
                            }
                        }
                    }
                }
            }
            return roleToTokenMapping;
        }

        public static Dictionary<string, List<string>> GetRoleToPermissionMappings(this string data)
        {
            var roleToPermissionMapping = new Dictionary<string, List<string>>();
            if (!string.IsNullOrWhiteSpace(data))
            {
                var lines = data.Split(new[] { "\r\n", "\r", "\n" }, StringSplitOptions.None);
                foreach (var line in lines)
                {
                    if (!string.IsNullOrWhiteSpace(line) && mappingRolePermissionRegex.IsMatch(line))
                    {
                        var match = mappingRolePermissionRegex.Match(line);
                        if (match.Success)
                        {
                            var key = match?.Groups[1]?.Value;
                            var value = match?.Groups[2]?.Value?.Split(",").ToList();
                            if (!string.IsNullOrWhiteSpace(key))
                            {
                                roleToPermissionMapping.Add(key, value);
                            }
                        }
                    }
                }
            }
            return roleToPermissionMapping;
        }
    }
}
