﻿using Microsoft.AspNetCore.Http;
using Microsoft.Extensions.Primitives;
using Resonance.Core.Helpers.LoggingHelpers;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Resonance.Core.Helpers.ApiHelpers
{
    public static class HeaderHelper
    {
        public static string GetRequestIP(HttpContext context, bool tryUseXForwardHeader = true)
        {
            string ip = null;

            if (tryUseXForwardHeader)
            {
                ip = GetHeaderValueAs<string>(context, "X-Forwarded-For").SplitCsv().FirstOrDefault();
            }

            if (string.IsNullOrWhiteSpace(ip) && context.Connection?.RemoteIpAddress != null)
            {
                ip = context.Connection.RemoteIpAddress.MapToIPv4().ToString();
            }

            if (string.IsNullOrWhiteSpace(ip))
            {
                ip = GetHeaderValueAs<string>(context, "REMOTE_ADDR");
            }

            if (string.IsNullOrWhiteSpace(ip))
            {
                throw new Exception("Unable to determine caller's IP.");
            }

            return ip;
        }

        public static string GetUserAgent(HttpContext context)
        {
            try
            {
                if (context.Request.Headers.ContainsKey("User-Agent"))
                {
                    var headers = context.Request.Headers["User-Agent"];

                    StringBuilder sb = new StringBuilder();

                    foreach (var header in headers)
                    {
                        sb.Append(header);

                        // Re-add spaces stripped when user agent string was split up.
                        sb.Append(" ");
                    }

                    var userAgent = sb.ToString().Trim();
                    return userAgent;
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }
            return "Unknown";
        }

        public static T GetHeaderValueAs<T>(HttpContext context, string headerName)
        {
            StringValues values;

            if (context.Request?.Headers?.TryGetValue(headerName, out values) ?? false)
            {
                string rawValues = values.ToString();   // writes out as Csv when there are multiple.

                if (!string.IsNullOrWhiteSpace(rawValues))
                {
                    return (T)Convert.ChangeType(values.ToString(), typeof(T));
                }
            }
            return default(T);
        }

        public static List<string> SplitCsv(this string csvList, bool nullOrWhitespaceInputReturnsNull = false)
        {
            if (string.IsNullOrWhiteSpace(csvList))
            {
                return nullOrWhitespaceInputReturnsNull ? null : new List<string>();
            }

            return csvList
                .TrimEnd(',')
                .Split(',')
                .AsEnumerable<string>()
                .Select(s => s.Trim())
                .ToList();
        }
    }

}
