﻿using Amazon;
using Amazon.Lambda.APIGatewayEvents;
using Amazon.Lambda.Model;
using Microsoft.AspNetCore.Http;
using Newtonsoft.Json;
using Resonance.Core;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.DateTimeHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StringHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.ApiModels.SalesforceModels;
using Resonance.Core.Models.ApiModels.TwitchModels;
using Resonance.Core.Models.DatabaseModels.TwitchUserListingModels;
using Resonance.Core.Models.ServiceModels.TwitchModels;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Threading.Tasks;
using System.Web;

namespace Resonance.Core.Helpers.ApiHelpers
{
    public static class TwitchUserDetailsHelper
    {
        private static int queryTimeout = 5;

        public static async Task<Dictionary<string, List<ChannelDetailsClip>>> GetClips(long channelID, ElapsedTimeModel metrics = null)
        {
            Log.Verbose($@"Starting GetClips");
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var clipdata = new Dictionary<string, List<ChannelDetailsClip>>();
            try
            {

                var tasks = new ConcurrentBag<Task>()
                {
                    GetClipData(channelID, new KeyValuePair<string, DateTime?>("1d", DateTime.UtcNow.AddDays(-1))).ContinueWith((data) =>
                    {
                        if (data.IsCompletedSuccessfully)
                        {
                            lock (clipdata)
                            {
                                try
                                {
                                    clipdata["1d"] = data.Result;
                                }
                                catch(Exception ex)
                                {
                                    Log.Error(ex);
                                }
                            }
                        }
                    }),
                    GetClipData(channelID, new KeyValuePair<string, DateTime?>("7d", DateTime.UtcNow.AddDays(-7))).ContinueWith((data) =>
                    {
                        if (data.IsCompletedSuccessfully)
                        {
                            lock (clipdata)
                            {
                                try
                                {
                                    clipdata["7d"] = data.Result;
                                }
                                catch(Exception ex)
                                {
                                    Log.Error(ex);
                                }
                            }
                        }
                    }),
                    GetClipData(channelID, new KeyValuePair<string, DateTime?>("30d", DateTime.UtcNow.AddDays(-30))).ContinueWith((data) =>
                    {
                        if (data.IsCompletedSuccessfully)
                        {
                            lock (clipdata)
                            {
                                try
                                {
                                    clipdata["30d"] = data.Result;
                                }
                                catch(Exception ex)
                                {
                                    Log.Error(ex);
                                }
                            }
                        }
                    }),
                    GetClipData(channelID, new KeyValuePair<string, DateTime?>("all", null)).ContinueWith((data) =>
                    {
                        if (data.IsCompletedSuccessfully)
                        {
                            lock (clipdata)
                            {
                                try
                                {
                                    clipdata["all"] = data.Result;
                                }
                                catch(Exception ex)
                                {
                                    Log.Error(ex);
                                }
                            }
                        }
                    })
                };

                await Task.WhenAll(tasks);
            }
            finally
            {
                stopwatch.Stop();
                Log.Info($@"Clips took {stopwatch.ElapsedMilliseconds}ms");
                if(metrics != null && metrics.ChildrenMetrics != null)
                {
                    metrics.ChildrenMetrics.Add(new ElapsedTimeModel()
                    {
                        MetricName = "clip_data",
                        ElapsedMS = stopwatch.ElapsedMilliseconds
                    });
                }
            }

            return clipdata;
        }

        private static async Task<List<ChannelDetailsClip>> GetClipData(long channelID, KeyValuePair<string, DateTime?> timeframe)
        {
            try
            {
                using (var client = new HttpClient())
                {
                    var url = $"http://clips.amp.xarth.tv/api/v3/clips/top?channel_id={channelID}&limit=5{(timeframe.Value.HasValue ? "&start=" + timeframe.Value.Value.ToString("yyyy-MM-ddThh:mm:ssZ") : "")}";
                    var response = await client.GetAsync(url);
                    response.EnsureSuccessStatusCode();
                    var content = await response.Content.ReadAsStringAsync();
                    dynamic clipData = JsonConvert.DeserializeObject<dynamic>(content);
                    List<ChannelDetailsClip> clips = new List<ChannelDetailsClip>();
                    foreach (dynamic clipDataEntry in clipData.clips)
                    {
                        ChannelDetailsClip clip = new ChannelDetailsClip
                        {
                            Title = clipDataEntry.title,
                            Views = clipDataEntry.views,
                            Created = clipDataEntry.created_at,
                            Thumbnail = clipDataEntry.preview_image,
                            Url = clipDataEntry.url
                        };
                        clips.Add(clip);
                    }
                    return clips;
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return new List<ChannelDetailsClip>();
        }

        public static async Task<TwitchUserListingModel> GetChannelListingData(string login)
        {
            TwitchUserListingModel model = new TwitchUserListingModel();

            try
            {
                using (var conn = await DBManagerMysql.GetConnectionAsync(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = queryTimeout;
                        command.CommandText = $@"Select valuescoreoverallweighted from {Constants.DatabaseSchema}microservice_twitch_user_listing_past_360_days where login=@login and aggregationtype=360";

                        command.Parameters.AddWithValue("@login", login);
                        using (var reader = await command.ExecuteReaderAsync())
                        {
                            if (reader.HasRows)
                            {
                                if (await reader.ReadAsync())
                                {
                                    model.ValueScoreOverallWeighted = reader["valuescoreoverallweighted"] != DBNull.Value ? (float?)reader["valuescoreoverallweighted"] : (float?)null;
                                }
                            }
                        }
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }

            return model;
        }

        public static async Task<UserScorecardModel> GetScorecard(long channelID, ElapsedTimeModel metrics = null)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var scorecard = new UserScorecardModel() { ScorecardTask = new List<ScorecardTaskModel>() };

            try
            {
                using (var conn = await DBManagerMysql.GetConnectionAsync(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = queryTimeout;
                        command.CommandText =
                        $@"
                            select
                                TwitchUserID,
                                Login,
                                DataSourced,
                                coalesce(HasProfileImage, False) as HasProfileImage, 
                                coalesce(HasUsedEvents, False) as HasUsedEvents,
                                coalesce(MeetsOrExceedsChannelPanelCount, False) as MeetsOrExceedsChannelPanelCount,
                                coalesce(HasUsedRules, False) as HasUsedRules,
                                coalesce(MeetsOrExceedsCollectionsCount, False) as MeetsOrExceedsCollectionsCount,
                                coalesce(AutomodEnabled, False) as AutomodEnabled,
                                coalesce(MeetsOrExceedsVodCount, False) as MeetsOrExceedsVodCount,
                                coalesce(MeetsOrExceedsHighlightCount, False) as MeetsOrExceedsHighlightCount,
                                coalesce(MeetsOrExceedsUploadCount, False) as MeetsOrExceedsUploadCount,
                                coalesce(MeetsOrExceedsEmoteCount, False) as MeetsOrExceedsEmoteCount,
                                coalesce(MeetsOrExceedsCheermoteCount, False) as MeetsOrExceedsCheermoteCount,
                                coalesce(MeetsOrExceedsClipCount, False) as MeetsOrExceedsClipCount,
                                coalesce(MeetsOrExceedsTenureBadgeCount, False) as MeetsOrExceedsTenureBadgeCount,
                                coalesce(MeetsOrExceedsExtensionInstallCount, False) as MeetsOrExceedsExtensionInstallCount,
                                coalesce(HasBio, False) as HasBio,
                                coalesce(HasStandardBackground, False) as HasStandardBackground,
                                coalesce(HasVideoPlayerBanner, False) as HasVideoPlayerBanner,
                                coalesce(HasChatBot, False) as HasChatBot,
                                coalesce(MeetsOrExceedsTagCount, False) as MeetsOrExceedsTagCount,
                                coalesce(MeetsOrExceedsModeratorCount, False) as MeetsOrExceedsModeratorCount,
                                coalesce(MeetsOrExceedsChatVanityCount, False) as MeetsOrExceedsChatVanityCount,
                                coalesce(MeetsOrExceedsEditorCount, False) as MeetsOrExceedsEditorCount,
                                coalesce(MeetsOrExceedsAutohostCount, False) as MeetsOrExceedsAutohostCount,
                                coalesce(HasAlerts, False) as HasAlerts,
                                coalesce(HasOverlays, False) as HasOverlays,
                                coalesce(HasAudioBalance, False) as HasAudioBalance,
                                coalesce(HasLightingBalance, False) as HasLightingBalance,
                                coalesce(HasDistinctFocalPoints, False) as HasDistinctFocalPoints,
                                coalesce(HasDiscordServer, False) as HasDiscordServer,
                                coalesce(HasRituals, False) as HasRituals,
                                coalesce(HasSocialMedia, False) as HasSocialMedia,
                                coalesce(HasCategory, False) as HasCategory
                            from {Constants.DatabaseSchema}microservice_twitch_user_scorecard
                            where TwitchUserID = @twitchUserID
                            limit 1;
                        ";
                        command.Parameters.AddWithValue("@twitchUserID", channelID);
                        using (var reader = await command.ExecuteReaderAsync())
                        {
                            if (reader.HasRows)
                            {
                                while (await reader.ReadAsync())
                                {
                                    scorecard.TwitchUserID = reader.GetInt64(0);
                                    scorecard.Login = reader.GetString(1);
                                    scorecard.DataSourced = reader.GetDateTime(2);
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "has_profile_image",
                                        DisplayName = "Has Profile Image",
                                        Description = "User has a profile image",
                                        RecommendedAction = "Set a description",
                                        IsComplete = reader.GetBoolean(3),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "has_used_events",
                                        DisplayName = "Has Used Events",
                                        Description = "User has used the event scheduling system",
                                        RecommendedAction = "Schedule an exciting event",
                                        IsComplete = reader.GetBoolean(4),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Low
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "meets_or_exceeds_channel_panel_count",
                                        DisplayName = "Sufficient Channel Panels",
                                        Description = "User utilizes channel panels",
                                        RecommendedAction = "Add some channel panels",
                                        IsComplete = reader.GetBoolean(5),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "has_used_rules",
                                        DisplayName = "Has Used Rules",
                                        Description = "User utilizes rules",
                                        RecommendedAction = "Add some chat rules",
                                        IsComplete = reader.GetBoolean(6),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "meets_or_exceeds_collections_count",
                                        DisplayName = "Has Video Collections",
                                        Description = "User utilizes video collections",
                                        RecommendedAction = "Create some video collections",
                                        IsComplete = reader.GetBoolean(7),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "automod_enabled",
                                        DisplayName = "Automod Enabled",
                                        Description = "User has automod enabled",
                                        RecommendedAction = "Enable auomod at your preferred level",
                                        IsComplete = reader.GetBoolean(8),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "meets_or_exceeds_vod_count",
                                        DisplayName = "Utilizes Vods",
                                        Description = "User utilizes vods",
                                        RecommendedAction = "Enable vod recordings",
                                        IsComplete = reader.GetBoolean(9),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "meets_or_exceeds_highlight_count",
                                        DisplayName = "Utilizes Highlights",
                                        Description = "User utilizes highlights",
                                        RecommendedAction = "Create some highlights",
                                        IsComplete = reader.GetBoolean(10),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "meets_or_exceeds_upload_count",
                                        DisplayName = "Utilizes Uploads",
                                        Description = "User utilizes video uploads",
                                        RecommendedAction = "Upload some awesome videos",
                                        IsComplete = reader.GetBoolean(11),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Low
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "meets_or_exceeds_emote_count",
                                        DisplayName = "Utilizes Emotes",
                                        Description = "User utilizes emotes",
                                        RecommendedAction = "Create some awesome emotes",
                                        IsComplete = reader.GetBoolean(12),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.High
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "meets_or_exceeds_cheermote_count",
                                        DisplayName = "Utilizes Cheermotes",
                                        Description = "User utilizes cheermotes",
                                        RecommendedAction = "Create some dazzling cheermotes",
                                        IsComplete = reader.GetBoolean(13),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.High
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "meets_or_exceeds_clip_count",
                                        DisplayName = "Utilizes Clips",
                                        Description = "Users viewers utilize clips",
                                        RecommendedAction = "Encourage viewers to clip the action",
                                        IsComplete = reader.GetBoolean(14),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Low
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "meets_or_exceeds_tenure_badge_count",
                                        DisplayName = "Utilizes Tenure Badges",
                                        Description = "User utilizes tenure badges",
                                        RecommendedAction = "Create badges that perfectly represent your viewers loyalty",
                                        IsComplete = reader.GetBoolean(15),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.High
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "meets_or_exceeds_extension_install_count",
                                        DisplayName = "Utilizes Extensions",
                                        Description = "User utilizes extensions",
                                        RecommendedAction = "Install some cool extensions",
                                        IsComplete = reader.GetBoolean(16),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "has_bio",
                                        DisplayName = "Has Bio",
                                        Description = "User has entered a bio",
                                        RecommendedAction = "Enter a bio",
                                        IsComplete = reader.GetBoolean(17),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Low
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "has_standard_background",
                                        DisplayName = "Has Standard Background",
                                        Description = "User has a standard background",
                                        RecommendedAction = "Set a background",
                                        IsComplete = reader.GetBoolean(18),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Low
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "has_video_player_banner",
                                        DisplayName = "Has Video Player Banner",
                                        Description = "User has a video player banner",
                                        RecommendedAction = "Create a video player banner",
                                        IsComplete = reader.GetBoolean(19),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "has_chat_bot",
                                        DisplayName = "Has Chat Bot",
                                        Description = "User has a chat bot",
                                        RecommendedAction = "Set up a chat bot",
                                        IsComplete = reader.GetBoolean(20),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "meets_or_exceeds_tag_count",
                                        DisplayName = "Utilizes Tags",
                                        Description = "User utilizes tags",
                                        RecommendedAction = "Set some tags, and if you don't stream for a while be sure to set them again!",
                                        IsComplete = reader.GetBoolean(21),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "meets_or_exceeds_chat_vanity_count",
                                        DisplayName = "Utilizes Vanity Badges",
                                        Description = "User utilizes chat vanity badges such as VIP",
                                        RecommendedAction = "Set VIP on your VIPs",
                                        IsComplete = reader.GetBoolean(22),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "meets_or_exceeds_editor_count",
                                        DisplayName = "Utilizes Editors",
                                        Description = "User utilizes video editors",
                                        RecommendedAction = "Get some help with your timeline! Set an editor",
                                        IsComplete = reader.GetBoolean(23),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Low
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "meets_or_exceeds_autohost_count",
                                        DisplayName = "Utilizes Autohost",
                                        Description = "User utilizes autohost",
                                        RecommendedAction = "Work with others and set some auto hosts while you're away",
                                        IsComplete = reader.GetBoolean(24),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "has_alerts",
                                        DisplayName = "Has Alerts",
                                        Description = "User has on screen alerts and notifications",
                                        RecommendedAction = "Create some alerts when bits, subs, donations, gifts, and more happen",
                                        IsComplete = reader.GetBoolean(25),
                                        IsManual = true,
                                        Priority = Constants.ScorecardPriority.High
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "has_overlays",
                                        DisplayName = "Has Overlays",
                                        Description = "User has on screen overlays",
                                        RecommendedAction = "Use overlays that match your games theme",
                                        IsComplete = reader.GetBoolean(26),
                                        IsManual = true,
                                        Priority = Constants.ScorecardPriority.High
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "has_audio_balance",
                                        DisplayName = "Has Audio Balance",
                                        Description = "User has a nice audio balance",
                                        RecommendedAction = "Set a nice audio balance to save your viewers ears",
                                        IsComplete = reader.GetBoolean(27),
                                        IsManual = true,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "has_lighting_balance",
                                        DisplayName = "Utilizes Lighting Balance",
                                        Description = "User has a nice lighting balance",
                                        RecommendedAction = "Get some lighting to help people see your sparkling smile",
                                        IsComplete = reader.GetBoolean(28),
                                        IsManual = true,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "has_distinct_focal_points",
                                        DisplayName = "Utilizes Focal Points",
                                        Description = "User utilizes distinct focal points to draw viewer attention",
                                        RecommendedAction = "Draw your viewers attention to specific locations where you want them to watch",
                                        IsComplete = reader.GetBoolean(29),
                                        IsManual = true,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "has_discord_server",
                                        DisplayName = "Has Discord Server",
                                        Description = "User has a discord server",
                                        RecommendedAction = "Set up a discord server for your community to come together",
                                        IsComplete = reader.GetBoolean(30),
                                        IsManual = true,
                                        Priority = Constants.ScorecardPriority.Low
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "has_rituals",
                                        DisplayName = "Utilizes Rituals",
                                        Description = "User has rituals that the viewers can expect",
                                        RecommendedAction = "Create log-on, log-off, break, sub, bits, gift rituals to help engage the audience",
                                        IsComplete = reader.GetBoolean(31),
                                        IsManual = true,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "has_social_media",
                                        DisplayName = "Utilizes Social Media",
                                        Description = "User has social media like Facebook, Instagram, Twitter, or Reddit",
                                        RecommendedAction = "Create social media accounts to help get your name out there",
                                        IsComplete = reader.GetBoolean(32),
                                        IsManual = true,
                                        Priority = Constants.ScorecardPriority.Normal
                                    });
                                    scorecard.ScorecardTask.Add(new ScorecardTaskModel()
                                    {
                                        Name = "has_category",
                                        DisplayName = "Utilizes Category",
                                        Description = "User has set a category they are streaming to",
                                        RecommendedAction = "Set a category so that your viewers can find you",
                                        IsComplete = reader.GetBoolean(34),
                                        IsManual = false,
                                        Priority = Constants.ScorecardPriority.High
                                    });
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                Log.Info($@"Scorecard took {stopwatch.ElapsedMilliseconds}ms");
                if (metrics != null && metrics.ChildrenMetrics != null)
                {
                    metrics.ChildrenMetrics.Add(new ElapsedTimeModel()
                    {
                        MetricName = "scorecard_data",
                        ElapsedMS = stopwatch.ElapsedMilliseconds
                    });
                }
            }
            return scorecard;
        }

        public static async Task<TwitchUserCalendarDisplayModel> GetCalendar(long channelID, ElapsedTimeModel metrics = null)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var calendar = new TwitchUserCalendarDisplayModel()
            {
                Data = new Dictionary<string, string>()
            };

            try
            {
                using (var conn = await DBManagerMysql.GetConnectionAsync(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = queryTimeout;
                        command.CommandText =
                        $@"
                            select
                                TwitchUserID,
                                Month,
                                MonthData
                            from {Constants.DatabaseSchema}microservice_twitch_user_calendar
                            where TwitchUserID = @twitchUserID
                            order by TwitchUserID, Month
                            limit 12;
                        ";
                        command.Parameters.AddWithValue("@twitchUserID", channelID);
                        using (var reader = await command.ExecuteReaderAsync())
                        {
                            if (reader.HasRows)
                            {
                                while (await reader.ReadAsync())
                                {
                                    var twitchUserID = reader.GetInt64(0);
                                    var month = reader["Month"] == DBNull.Value ? null : ((string)reader["Month"]).ToLower();
                                    var data = reader["MonthData"] == DBNull.Value ? null : (string)reader["MonthData"];
                                    calendar.Data.Add(month, data.Base64DecodeGzipInflate());
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                Log.Info($@"Calendar took {stopwatch.ElapsedMilliseconds}ms");
                if (metrics != null && metrics.ChildrenMetrics != null)
                {
                    metrics.ChildrenMetrics.Add(new ElapsedTimeModel()
                    {
                        MetricName = "calendar_data",
                        ElapsedMS = stopwatch.ElapsedMilliseconds
                    });
                }
            }
            return calendar;
        }

        public static async Task<TwitchViewbotModel> GetViewbot(long channelID, ElapsedTimeModel metrics = null)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var viewbot = new TwitchViewbotModel();

            try
            {
                using (var conn = await DBManagerMysql.GetConnectionAsync(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = queryTimeout;
                        command.CommandText =
                        $@"
                            select
                                BeforeAvg,
                                MergedAvg,
                                BestGuessAvg,
                                Data
                            from {Constants.DatabaseSchema}microservice_twitch_viewbot
                            where ChannelID = @twitchUserID
                            limit 1;
                        ";
                        command.Parameters.AddWithValue("@twitchUserID", channelID);
                        using (var reader = await command.ExecuteReaderAsync())
                        {
                            if (reader.HasRows)
                            {
                                while (await reader.ReadAsync())
                                {
                                    viewbot.BeforeAvg = reader["BeforeAvg"] == DBNull.Value ? null : (float?)(float)reader["BeforeAvg"];
                                    viewbot.MergedAvg = reader["MergedAvg"] == DBNull.Value ? null : (float?)(float)reader["MergedAvg"];
                                    viewbot.BestGuessAvg = reader["BestGuessAvg"] == DBNull.Value ? null : (float?)(float)reader["BestGuessAvg"];
                                    viewbot.Data = reader["Data"] == DBNull.Value ? null : ((string)reader["Data"]).Base64DecodeGzipInflate();
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                Log.Info($@"Viewbot took {stopwatch.ElapsedMilliseconds}ms");
                if (metrics != null && metrics.ChildrenMetrics != null)
                {
                    metrics.ChildrenMetrics.Add(new ElapsedTimeModel()
                    {
                        MetricName = "viewbot_data",
                        ElapsedMS = stopwatch.ElapsedMilliseconds
                    });
                }
            }
            return viewbot;
        }

        public static async Task<TwitchUserWeeklySummaryModel> GetWeeklySummary(long channelID, ElapsedTimeModel metrics = null)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var weekly = new TwitchUserWeeklySummaryModel();

            try
            {
                using (var conn = await DBManagerMysql.GetConnectionAsync(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = queryTimeout;
                        command.CommandText =
                        $@"
                            select LastUpdateDate, Data
                            from {Constants.DatabaseSchema}microservice_twitch_user_weekly_summary
                            where TwitchUserID = @twitchUserID
                            limit 1;
                        ";
                        command.Parameters.AddWithValue("@twitchUserID", channelID);
                        using (var reader = await command.ExecuteReaderAsync())
                        {
                            if (reader.HasRows)
                            {
                                while (await reader.ReadAsync())
                                {
                                    weekly = new TwitchUserWeeklySummaryModel
                                    {
                                        LastUpdateDate = reader?.GetDateTime(0) ?? new DateTime(1900, 1, 1, 0, 0, 0, DateTimeKind.Utc),
                                        Data = reader?.GetString(1)?.Base64DecodeGzipInflate()
                                    };
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                Log.Info($@"Weekly Summary took {stopwatch.ElapsedMilliseconds}ms");
                if (metrics != null && metrics.ChildrenMetrics != null)
                {
                    metrics.ChildrenMetrics.Add(new ElapsedTimeModel()
                    {
                        MetricName = "weekly_summary_data",
                        ElapsedMS = stopwatch.ElapsedMilliseconds
                    });
                }
            }
            return weekly;
        }

        public static async Task<ChannelCountry> GetTopCountries(long channelID, ElapsedTimeModel metrics = null)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var data = new ChannelCountry()
            {
                CountryDetails = new List<ChannelCountryDetails>()
            };


            try
            {
                using (var conn = await DBManagerMysql.GetConnectionAsync(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = queryTimeout;
                        command.CommandText =
                        $@"
                            select
                                channel_id, 
                                country_code, 
                                country_name, 
                                total_count, 
                                total_count_known, 
                                total_count_unknown, 
                                total_known_percent, 
                                total_unknown_percent, 
                                country_percent, 
                                rank, 
                                last_update_date
                            from {Constants.DatabaseSchema}microservice_staging_twitch_user_top_countries
                            where 
                                channel_id = @twitchUserID
                                and rank <= 5
                            order by rank
                            limit 5;
                        ";
                        command.Parameters.AddWithValue("@twitchUserID", channelID);
                        using (var reader = await command.ExecuteReaderAsync())
                        {
                            if (reader.HasRows)
                            {
                                var first = true;
                                while (await reader.ReadAsync())
                                {
                                    if (first)
                                    {
                                        data.TotalCount = reader["total_count"] == DBNull.Value ? (long?)null : (long?)reader["total_count"];
                                        data.TotalKnownCount = reader["total_count_known"] == DBNull.Value ? (long?)null : (long?)reader["total_count_known"];
                                        data.TotalUnknownCount = reader["total_count_unknown"] == DBNull.Value ? (long?)null : (long?)reader["total_count_unknown"];
                                        data.TotalKnownPercent = reader["total_known_percent"] == DBNull.Value ? (float?)null : (float?)reader["total_known_percent"];
                                        data.TotalUnknownPercent = reader["total_unknown_percent"] == DBNull.Value ? (float?)null : (float?)reader["total_unknown_percent"];
                                        data.LastUpdateDate = reader["last_update_date"] == DBNull.Value ? (DateTime?)null : (DateTime?)reader["last_update_date"];
                                        first = false;
                                    }

                                    var item = new ChannelCountryDetails()
                                    {
                                        CountryCode = reader["country_code"] == DBNull.Value ? null : (string)reader["country_code"],
                                        CountryName = reader["country_name"] == DBNull.Value ? null : (string)reader["country_name"],
                                        CountryPercent = reader["country_percent"] == DBNull.Value ? (float?)null : (float?)reader["country_percent"],
                                        Rank = reader["rank"] == DBNull.Value ? (int?)null : (int?)reader["rank"]
                                    };
                                    data.CountryDetails.Add(item);
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                Log.Info($@"Top Countries took {stopwatch.ElapsedMilliseconds}ms");
                if (metrics != null && metrics.ChildrenMetrics != null)
                {
                    metrics.ChildrenMetrics.Add(new ElapsedTimeModel()
                    {
                        MetricName = "top_countries_data",
                        ElapsedMS = stopwatch.ElapsedMilliseconds
                    });
                }
            }
            return data;
        }


        public static async Task<List<ChannelDetailsStream>> GetStreams(long channelID, string login, ElapsedTimeModel metrics = null)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var streamsData = new List<ChannelDetailsStream>();
            var proxyWatch = new Stopwatch();
            proxyWatch.Start();
            try
            {
                APIGatewayProxyRequest proxyRequest = new APIGatewayProxyRequest()
                {
                    HttpMethod = "POST",
                    IsBase64Encoded = false,
                    Path = "/twirp/twitch.chemtrail.ChemtrailService/GetStreamSummaries",
                    Headers = new Dictionary<string, string>() { { "Content-Type", "application/json" } },
                    Body = JsonConvert.SerializeObject(new { channel_id = channelID.ToString(), limit = 5, viewer_id = channelID.ToString() })
                };

                Amazon.Lambda.AmazonLambdaClient lambdaClient = new Amazon.Lambda.AmazonLambdaClient(RegionEndpoint.USWest2);

                InvokeRequest invokeRequest = new InvokeRequest()
                {
                    FunctionName = "arn:aws:lambda:us-west-2:873624430394:function:ChemtrailStack-APIFunctionDD57B14D-1L8ODXNF8YVGW:Live",
                    Payload = JsonConvert.SerializeObject(proxyRequest)
                };
                Dictionary<string, List<ChannelDetailsStream>> streamsByBroadcastID = new Dictionary<string, List<ChannelDetailsStream>>();

                var lambdaResponse = await lambdaClient.InvokeAsync(invokeRequest);
                using (TextReader reader = new StreamReader(lambdaResponse.Payload))
                {
                    string lambdaResponseString = await reader.ReadToEndAsync();
                    APIGatewayProxyResponse response = JsonConvert.DeserializeObject<APIGatewayProxyResponse>(lambdaResponseString);
                    dynamic results = JsonConvert.DeserializeObject<dynamic>(response.Body);
                    streamsData = new List<ChannelDetailsStream>();

                    if(results != null && results.stream_summaries != null)
                    {
                        foreach (dynamic streamEntry in results?.stream_summaries)
                        {
                            ChannelDetailsStream stream = new ChannelDetailsStream();
                            stream.AverageCCU = streamEntry.average_viewers ?? 0;
                            stream.PeakCCU = streamEntry.max_viewers ?? 0;
                            stream.Followers = streamEntry.new_followers ?? 0;                            
                            stream.StartDate = streamEntry.started_at;                            
                            stream.EndDate = streamEntry.ended_at;
                            stream.Views = streamEntry.unique_views ?? 0;
                            stream.Categories = new HashSet<string>();
                            List<string> broadcastIDs = new List<string>();

                            if(streamEntry.broadcast_ids != null)
                            {
                                foreach (string broadcastID in streamEntry?.broadcast_ids)
                                {
                                    broadcastIDs.Add(broadcastID);
                                    if (!streamsByBroadcastID.ContainsKey(broadcastID))
                                    {
                                        streamsByBroadcastID.Add(broadcastID, new List<ChannelDetailsStream>());
                                    }
                                    streamsByBroadcastID[broadcastID].Add(stream);
                                }
                            }
                            stream.BroadcastIDs = broadcastIDs;
                            stream.DropInUrl = $@"https://dashboard.twitch.tv/u/{login}/stream-summary/{stream.StartDate.ToUnixTime() * 1000}-{stream.EndDate.ToUnixTime() * 1000}";
                            streamsData.Add(stream);
                        }
                        proxyWatch.Stop();
                        Log.Info($@"GetStreams Proxy Total: {proxyWatch.ElapsedMilliseconds}ms");
                        try
                        {
                            WebClient client = new WebClient();
                            client.Headers.Add("Content-Type", "application/json");
                            string broadcastMetaDataResponse = client.UploadString(@"http://vodapi.amp.xarth.tv/twirp/vodapi.VodApi/InternalGetVodsByBroadcastIDs", "POST", JsonConvert.SerializeObject(new { broadcast_ids = streamsData.SelectMany(x => x.BroadcastIDs).Distinct() }));
                            dynamic broadcastMetaData = JsonConvert.DeserializeObject<dynamic>(broadcastMetaDataResponse);
                            foreach (dynamic broadcast in broadcastMetaData.vods)
                            {
                                string broadcastID = broadcast.broadcast_id;
                                string preview = broadcast.preview_template;
                                string title = broadcast.title;
                                preview = preview.Replace("%{width}", "80").Replace("%{height}", "45");
                                string game = broadcast.game;
                                List<ChannelDetailsStream> streams = null;
                                if (streamsByBroadcastID.TryGetValue(broadcastID, out streams))
                                {
                                    foreach (ChannelDetailsStream stream in streams)
                                    {
                                        if (stream.Title == null)
                                        {
                                            stream.Title = title;
                                            stream.Thumbnail = preview;
                                        }
                                        stream.Categories.Add(game);
                                    }
                                }
                            }
                        }
                        catch (Exception ex)
                        {
                            Log.Error(ex);
                        }
                    }
                }
                streamsData = streamsData.Where(x => x.Title != null).ToList();
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                Log.Info($@"Streams took {stopwatch.ElapsedMilliseconds}ms");
                if (metrics != null && metrics.ChildrenMetrics != null)
                {
                    metrics.ChildrenMetrics.Add(new ElapsedTimeModel()
                    {
                        MetricName = "stream_data",
                        ElapsedMS = stopwatch.ElapsedMilliseconds
                    });
                }
            }
            return streamsData;
        }

        public static async Task<IEnumerable<UserCase>> GetPartnershipAuditData(long channelID, ElapsedTimeModel metrics = null)
        {
            List<UserCase> cases = new List<UserCase>();
            var stopwatch = new Stopwatch();
            stopwatch.Start();
            try
            {
                var basequery = $"Select Id,TwitchUserID__c,Status,Original_Close_Date__c,CreatedDate,ClosedDate from Case";
                var filterquery = $@" where TwitchUserID__c='{channelID}'";
                var salesforceResults = await SalesforceHelpers.QueryDataAsync(basequery + filterquery);
                foreach (dynamic salesforceEntry in salesforceResults)
                {
                    string closedDateString = salesforceEntry.ClosedDate;
                    string createdDateString = salesforceEntry.CreatedDate;
                    string originalCreatedDateString = salesforceEntry.Original_Close_Date__c;
                    DateTime? closedDate = null;
                    DateTime closedDateSource;
                    DateTime createdDate;
                    DateTime? originalCreatedDate = null;
                    DateTime originalCreatedDateSource;
                    if(DateTime.TryParse(closedDateString, out closedDateSource))
                    {
                        closedDate = closedDateSource;
                    }
                    DateTime.TryParse(createdDateString, out createdDate);
                    if(DateTime.TryParse(originalCreatedDateString, out originalCreatedDateSource))
                    {
                        originalCreatedDate = originalCreatedDateSource;
                    }
                    cases.Add(new UserCase() { ID = salesforceEntry.Id, Status = salesforceEntry.Status, ClosedDate = closedDate, CreatedDate=createdDate, OriginalClosedDate=originalCreatedDate });
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                Log.Info($@"GetPartnershipAuditData took {stopwatch.ElapsedMilliseconds}ms");
                if (metrics != null && metrics.ChildrenMetrics != null)
                {
                    metrics.ChildrenMetrics.Add(new ElapsedTimeModel()
                    {
                        MetricName = "partnership_audit_data",
                        ElapsedMS = stopwatch.ElapsedMilliseconds
                    });
                }
            }
            return cases;
        }

        public static async Task<TenureBadgeSets> GetTenureBadges(long channelID, HttpContext context = null, ElapsedTimeModel metrics = null)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var badges = new TenureBadgeSets();

            try
            {
                var result = await WebRequestHelper.GetDataAsync($"https://prod.badges.twitch.a2z.com/v1/badges/channels/{channelID}/display", null, 5, false, context);
                if(result != null && result.IsSuccess)
                {
                    badges = JsonConvert.DeserializeObject<TenureBadgeSets>(result.Data);
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                Log.Info($@"GetTenureBadges took {stopwatch.ElapsedMilliseconds}ms");
                if (metrics != null && metrics.ChildrenMetrics != null)
                {
                    metrics.ChildrenMetrics.Add(new ElapsedTimeModel()
                    {
                        MetricName = "tenure_data",
                        ElapsedMS = stopwatch.ElapsedMilliseconds
                    });
                }
            }

            return badges;
        }

        public static async Task<List<TwitchEmoteDisplayModel>> GetEmotes(long channelID, HttpContext context = null, ElapsedTimeModel metrics = null)
        {
            var emotes = new List<TwitchEmoteDisplayModel>();
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            try
            {
                using (var conn = await DBManagerMysql.GetConnectionAsync(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = queryTimeout;
                        command.CommandText =
                        $@"
                            select
	                            Regex,
                                TierType,
                                BaseUrl,
                                SmallUrlFragment,
                                MediumUrlFragment,
                                LargeUrlFragment
                            from {Constants.DatabaseSchema}microservice_twitch_user_emotes 
                            where ChannelID = @twitchUserID;
                        ";
                        command.Parameters.AddWithValue("@twitchUserID", channelID);
                        using (var reader = await command.ExecuteReaderAsync())
                        {
                            if (reader.HasRows)
                            {
                                while(await reader.ReadAsync())
                                {
                                    try
                                    {
                                        var item = new TwitchEmoteDisplayModel()
                                        {
                                            Regex = reader.GetString(0),
                                            TierType = reader.GetString(1),
                                            BaseUrl = reader.GetString(2),
                                            SmallUrlFragment = reader.GetString(3),
                                            MediumUrlFragment = reader.GetString(4),
                                            LargeUrlFragment = reader.GetString(5)
                                        };
                                        emotes.Add(item);
                                    }
                                    catch (Exception ex)
                                    {
                                        Log.Error(ex);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                Log.Info($@"GetEmotes took {stopwatch.ElapsedMilliseconds}ms");
                if (metrics != null && metrics.ChildrenMetrics != null)
                {
                    metrics.ChildrenMetrics.Add(new ElapsedTimeModel()
                    {
                        MetricName = "emote_data",
                        ElapsedMS = stopwatch.ElapsedMilliseconds
                    });
                }
            }

            return emotes;
        }

        public static async Task<CheermoteActions> GetCheermotes(long channelID, HttpContext context = null, ElapsedTimeModel metrics = null)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            var cheermotes = new CheermoteActions();
            try
            {
                var url =
                    Constants.AppConfig.Application.Environment == "Development"
                    ? $"https://main.us-west-2.beta.payday.twitch.a2z.com/api/actions?channel_id={channelID}"
                    : $"https://prod.payday.twitch.a2z.com/api/actions?channel_id={channelID}";
                var result = await WebRequestHelper.GetDataAsync(url, null, 5, false);
                if(result != null && result.IsSuccess)
                {
                    cheermotes = JsonConvert.DeserializeObject<CheermoteActions>(result.Data);
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                stopwatch.Stop();
                Log.Info($@"GetCheermotes took {stopwatch.ElapsedMilliseconds}ms");
                if (metrics != null && metrics.ChildrenMetrics != null)
                {
                    metrics.ChildrenMetrics.Add(new ElapsedTimeModel()
                    {
                        MetricName = "cheermote_data",
                        ElapsedMS = stopwatch.ElapsedMilliseconds
                    });
                }
            }

            return cheermotes;
        }
    }
}
