﻿using Microsoft.AspNetCore.Http;
using Newtonsoft.Json;
using Resonance.Core.Helpers.ErrorHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels;
using System;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace Resonance.Core.Helpers.ApiHelpers
{
    public static class WebRequestHelper
    {
        private static string apptoken { get; set; } = null;
        public static string AppToken
        {
            get
            {
                return apptoken;
            }
        }
        private static string servicetoken { get; set; } = null;
        public static string ServiceToken
        {
            get
            {
                return servicetoken;
            }
        }

        static WebRequestHelper()
        {
            if(Constants.Permissions != null && Constants.Permissions.RoleToTokenMap != null)
            {
                apptoken = Constants.Permissions.RoleToTokenMap.ContainsKey("AppServiceToken")
                    ? Constants.Permissions?.RoleToTokenMap["AppServiceToken"]?.FirstOrDefault() ?? ""
                    : ""
                ;
                servicetoken = Constants.Permissions.RoleToTokenMap.ContainsKey("JobServiceUpdate")
                    ? Constants.Permissions?.RoleToTokenMap["JobServiceUpdate"]?.FirstOrDefault() ?? ""
                    : ""
                ;
            }
            else
            {
                apptoken = "";
                servicetoken = "";
            }
        }

        public static WebRequestResponse GetData(string url, string bearer, int timeout = 86400, bool isToken = false, HttpContext context = null)
        {
            WebRequestResponse response = new WebRequestResponse();
            try
            {
                using (var httpClientHandler = new HttpClientHandler())
                {
                    // The line below disables SSL chain validation
                    httpClientHandler.ServerCertificateCustomValidationCallback = (message, cert, chain, errors) => { return true; };
                    // The line above disables SSL chain validation
                    using (var client = new HttpClient(httpClientHandler))
                    {
                        httpClientHandler.AutomaticDecompression = DecompressionMethods.GZip | DecompressionMethods.Deflate;
                        client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/json"));

                        if (!string.IsNullOrWhiteSpace(bearer))
                        {
                            if (isToken)
                            {
                                client.DefaultRequestHeaders.Add("Authorization", $"ApiKey {bearer}");
                            }
                            else
                            {
                                client.DefaultRequestHeaders.Add("Authorization", $"Bearer {bearer}");
                            }
                        }
                        client.Timeout = TimeSpan.FromSeconds(timeout);
                        try
                        {
                            var result = client.GetAsync(url);
                            result.Wait();
                            var resultTask = result.Result.Content.ReadAsStringAsync();
                            resultTask.Wait();
                            if(resultTask.IsCompletedSuccessfully)
                            {
                                response.Data = resultTask.Result;
                                response.IsSuccess = true;
                                response.Error = null;
                                response.IsError = false;
                                response.HttpStatusCode = (int)result.Result.StatusCode;
                            }
                        }
                        catch (WebException ex)
                        {
                            response.HttpStatusCode = 500;
                            response.IsSuccess = false;
                            response.IsError = true;
                            response.IsSuccess = false;
                            response.ErrorCode = ErrorCodeHelper.CalculateErrorCode(context?.TraceIdentifier ?? Guid.NewGuid().ToString());

                            var responsedata = ex.Response as HttpWebResponse;
                            if (responsedata != null)
                            {
                                response.HttpStatusCode = (int)responsedata.StatusCode;
                            }
                            try
                            {
                                using (var exResponse = new StreamReader(ex.Response.GetResponseStream()))
                                {
                                    response.Error = exResponse.ReadToEnd();
                                }
                            }
                            catch (Exception ex2)
                            {
                                response.Error = ex2.ToString();
                            }
                        }
                        catch (Exception ex)
                        {
                            response.IsSuccess = false;
                            response.IsError = true;
                            response.Error = ex.ToString();
                            response.ErrorCode = ErrorCodeHelper.CalculateErrorCode(context?.TraceIdentifier ?? Guid.NewGuid().ToString());
                            response.HttpStatusCode = 500;
                        }
                    }
                }
            }
            catch(Exception ex)
            {
                response.IsSuccess = false;
                response.IsError = true;
                response.Error = ex.ToString();
                response.ErrorCode = ErrorCodeHelper.CalculateErrorCode(context?.TraceIdentifier ?? Guid.NewGuid().ToString());
                response.HttpStatusCode = 500;
            }
            return response;
        }

        public static async Task<WebRequestResponse> GetDataAsync(string url, string bearer, int timeout = 86400, bool isToken = false, HttpContext context = null)
        {
            WebRequestResponse response = new WebRequestResponse();
            try
            {
                using (var httpClientHandler = new HttpClientHandler())
                {
                    // The line below disables SSL chain validation
                    httpClientHandler.ServerCertificateCustomValidationCallback = (message, cert, chain, errors) => { return true; };
                    // The line above disables SSL chain validation
                    using (var client = new HttpClient(httpClientHandler))
                    {
                        httpClientHandler.AutomaticDecompression = DecompressionMethods.GZip | DecompressionMethods.Deflate;
                        client.DefaultRequestHeaders.Accept.Add(new MediaTypeWithQualityHeaderValue("application/json"));

                        if (!string.IsNullOrWhiteSpace(bearer))
                        {
                            if (isToken)
                            {
                                client.DefaultRequestHeaders.Add("Authorization", $"ApiKey {bearer}");
                            }
                            else
                            {
                                client.DefaultRequestHeaders.Add("Authorization", $"Bearer {bearer}");
                            }
                        }
                        client.Timeout = TimeSpan.FromSeconds(timeout);
                        try
                        {
                            var result = await client.GetAsync(url);
                            var resultTask = await result.Content.ReadAsStringAsync();
                            response.Data = resultTask;
                            response.IsSuccess = true;
                            response.IsError = false;
                            response.Error = null;
                            response.HttpStatusCode = (int)result.StatusCode;
                        }
                        catch (WebException ex)
                        {
                            response.IsError = true;
                            response.IsSuccess = false;
                            response.ErrorCode = ErrorCodeHelper.CalculateErrorCode(context?.TraceIdentifier ?? Guid.NewGuid().ToString());
                            response.HttpStatusCode = 500;
                            var responsedata = ex.Response as HttpWebResponse;
                            if (responsedata != null)
                            {
                                response.HttpStatusCode = (int)responsedata.StatusCode;
                            }
                            try
                            {
                                using (var exResponse = new StreamReader(ex.Response.GetResponseStream()))
                                {
                                    response.Error = exResponse.ReadToEnd();
                                }
                            }
                            catch (Exception ex2)
                            {
                                response.Error = ex2.ToString();
                            }
                        }
                        catch (Exception ex)
                        {
                            response.IsSuccess = false;
                            response.IsError = true;
                            response.Error = ex.ToString();
                            response.ErrorCode = ErrorCodeHelper.CalculateErrorCode(context?.TraceIdentifier ?? Guid.NewGuid().ToString());
                            response.HttpStatusCode = 500;
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                response.HttpStatusCode = 500;
                response.IsSuccess = false;
                response.IsError = true;
                response.ErrorCode = ErrorCodeHelper.CalculateErrorCode(context?.TraceIdentifier ?? Guid.NewGuid().ToString());
                response.Error = ex.ToString();
            }
            return response;
        }        

        public static WebRequestResponse PostData<T>(string url, string bearer, ref T data, int timeout = 86400, bool isToken = false, bool needsAuth = false, HttpContext context = null)
        {
            WebRequestResponse response = new WebRequestResponse();
            try
            {
                using (var client = new ResonanceWebClient(timeout))
                {
                    if (needsAuth)
                    {
                        if (!string.IsNullOrWhiteSpace(bearer))
                        {
                            if (isToken)
                            {
                                client.Headers.Add("Authorization", $"ApiKey {bearer}");
                            }
                            else
                            {
                                client.Headers.Add("Authorization", $"Bearer {bearer}");
                            }
                        }
                    }

                    int retryCount = 3;
                    bool success = false;
                    for (int retry = 0; retry <= 3; retry++)
                    {
                        if (success)
                        {
                            break;
                        }
                        try
                        {
                            response.Data = client.UploadString(url, "POST", JsonConvert.SerializeObject(data));
                            response.IsSuccess = true;
                            response.IsError = false;
                            response.Error = null;
                            success = true;
                            response.HttpStatusCode = 200;
                        }
                        catch (OperationCanceledException ex)
                        {
                            if (retry == retryCount)
                            {
                                response.HttpStatusCode = 500;
                                response.IsSuccess = false;
                                response.IsError = true;
                                response.Error = $"{response.ErrorCode} :: {ex.ToString()} {url} {JsonConvert.SerializeObject(data)}";
                                response.ErrorCode = ErrorCodeHelper.CalculateErrorCode(context?.TraceIdentifier ?? Guid.NewGuid().ToString());
                            }
                            Thread.Sleep(500);
                        }
                        catch (WebException ex)
                        {
                            if (retry == retryCount)
                            {
                                response.HttpStatusCode = 500;
                                response.IsSuccess = false;
                                response.IsError = true;

                                var responsedata = ex.Response as HttpWebResponse;
                                if (responsedata != null)
                                {
                                    response.HttpStatusCode = (int)responsedata.StatusCode;
                                }
                                response.ErrorCode = ErrorCodeHelper.CalculateErrorCode(context?.TraceIdentifier ?? Guid.NewGuid().ToString());

                                using (var exResponse = new StreamReader(ex.Response.GetResponseStream()))
                                {
                                    response.Error = exResponse.ReadToEnd();
                                }
                            }
                            Thread.Sleep(500);
                        }
                        catch (Exception ex)
                        {
                            if (retry == retryCount)
                            {
                                response.IsSuccess = false;
                                response.IsError = true;
                                response.HttpStatusCode = 500;
                                response.ErrorCode = ErrorCodeHelper.CalculateErrorCode(context?.TraceIdentifier ?? Guid.NewGuid().ToString());
                                response.Error = $@"{response.ErrorCode} :: {ex.ToString()} {url} {JsonConvert.SerializeObject(data)}";
                            }
                            Thread.Sleep(500);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                response.IsSuccess = false;
                response.IsError = true;
                response.HttpStatusCode = 500;
                response.ErrorCode = ErrorCodeHelper.CalculateErrorCode(context?.TraceIdentifier ?? Guid.NewGuid().ToString());
                response.Error = ex.ToString();
            }

            return response;
        }

        public static WebRequestResponse PostData(string url, string bearer, byte[] data, int timeout = 86400, bool isToken = false, bool needsAuth = false, HttpContext context = null)
        {
            WebRequestResponse response = new WebRequestResponse();
            try
            {
                using (var client = new ResonanceWebClient(timeout))
                {
                    if (needsAuth)
                    {
                        if (!string.IsNullOrWhiteSpace(bearer))
                        {
                            if (isToken)
                            {
                                client.Headers.Add("Authorization", $"ApiKey {bearer}");
                            }
                            else
                            {
                                client.Headers.Add("Authorization", $"Bearer {bearer}");
                            }
                        }
                    }

                    int retryCount = 3;
                    bool success = false;
                    for (int retry = 0; retry <= 3; retry++)
                    {
                        if (success)
                        {
                            break;
                        }
                        try
                        {
                            var result = client.UploadData(url, data);
                            response.Data = Encoding.UTF8.GetString(result);
                            response.IsSuccess = true;
                            response.IsError = false;
                            response.Error = null;
                            success = true;
                            response.HttpStatusCode = 200;
                        }
                        catch (OperationCanceledException ex)
                        {
                            if (retry == retryCount)
                            {
                                response.HttpStatusCode = 500;
                                response.IsSuccess = false;
                                response.IsError = true;
                                response.Error = $"{response.ErrorCode} :: {ex.ToString()} {url} {JsonConvert.SerializeObject(data)}";
                                response.ErrorCode = ErrorCodeHelper.CalculateErrorCode(context?.TraceIdentifier ?? Guid.NewGuid().ToString());
                            }
                            Thread.Sleep(500);
                        }
                        catch (WebException ex)
                        {
                            if (retry == retryCount)
                            {
                                response.HttpStatusCode = 500;
                                response.IsSuccess = false;
                                response.IsError = true;

                                var responsedata = ex.Response as HttpWebResponse;
                                if (responsedata != null)
                                {
                                    response.HttpStatusCode = (int)responsedata.StatusCode;
                                }
                                response.ErrorCode = ErrorCodeHelper.CalculateErrorCode(context?.TraceIdentifier ?? Guid.NewGuid().ToString());

                                using (var exResponse = new StreamReader(ex.Response.GetResponseStream()))
                                {
                                    response.Error = exResponse.ReadToEnd();
                                }
                            }
                            Thread.Sleep(500);
                        }
                        catch (Exception ex)
                        {
                            if (retry == retryCount)
                            {
                                response.IsSuccess = false;
                                response.IsError = true;
                                response.HttpStatusCode = 500;
                                response.ErrorCode = ErrorCodeHelper.CalculateErrorCode(context?.TraceIdentifier ?? Guid.NewGuid().ToString());
                                response.Error = $@"{response.ErrorCode} :: {ex.ToString()} {url} {JsonConvert.SerializeObject(data)}";
                            }
                            Thread.Sleep(500);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                response.IsSuccess = false;
                response.IsError = true;
                response.HttpStatusCode = 500;
                response.ErrorCode = ErrorCodeHelper.CalculateErrorCode(context?.TraceIdentifier ?? Guid.NewGuid().ToString());
                response.Error = ex.ToString();
            }

            return response;
        }

    }
}
