﻿using Microsoft.AspNetCore.Http;
using Newtonsoft.Json;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.FormHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StringHelpers;
using Resonance.Core.Models.ServiceModels.AtlasModels;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using static Resonance.Core.Constants;

namespace Resonance.Core.Helpers.AssetHelpers
{
    public static class AssetHelper
    {
        private const int mbSize = 1024;
        private const int serverMaxUserImageAssetSize = 150 * mbSize;
        /// <summary>
        /// http://www.libpng.org/pub/png/spec/1.2/PNG-Structure.html Section 3.1
        /// </summary>
        private static int[] pngHeader = new int[8] { 137, 80, 78, 71, 13, 10, 26, 10 };
        private static int[] jpgHeader = new int[4] { 255, 216, 255, 224 };
        public static Dictionary<string, ProductImageRestriction> GetImageSlotRestrictions()
        {            
            return imageSlotRestrictions;         
        }
        private static Dictionary<string, ProductImageRestriction> imageSlotRestrictions = new Dictionary<string, ProductImageRestriction>()
        {
            { "front-page-promotional-banner-slot", new ProductImageRestriction() {
                MinHeight = 156,
                MinWidth = 280,
                MaxHeight = 156,
                MaxWidth = 280,
                MaxFileSize = serverMaxUserImageAssetSize
            } },
            { "marketing-banner-medium-rectangle-slot", new ProductImageRestriction() {
                MinHeight = 250,
                MinWidth = 300,
                MaxHeight = 250,
                MaxWidth = 300,
                MaxFileSize = serverMaxUserImageAssetSize
            } },
            { "marketing-banner-super-leaderboard-slot", new ProductImageRestriction() {
                MinHeight = 66,
                MinWidth = 970,
                MaxHeight = 66,
                MaxWidth = 970,
                MaxFileSize = serverMaxUserImageAssetSize
            } }
        };

        public static bool UserHasPermissionToEditProduct(int productID, HttpContext httpContext)
        {
            // Todo: Permission check for user
            return true;
        }

        private static Dictionary<string, ProductVideoRestrictions> videoSlotRestrictions = new Dictionary<string, ProductVideoRestrictions>()
        {
            { "video-ad-slot", new ProductVideoRestrictions() {
                MaxFileSize = serverMaxUserVideoAssetSize
            } }
        };
        private const int serverMaxUserVideoAssetSize = 100000000;

        public static bool ValidateImageFile(string fileName, string contentType, byte[] file)
        {
            var valid = true;
            try
            {
                valid = ValidateSharedImageData(fileName, file);

                if (valid)
                {
                    switch (contentType)
                    {
                        case "image/png":
                        {
                            valid = ValidatePngHeader(file);
                            valid = ValidateImageSize(fileName, file);
                            break;
                        }
                        case "image/jpeg":
                        {
                            valid = ValidateJpgHeader(file);
                            valid = ValidateImageSize(fileName, file);
                            break;
                        }
                        default:
                        {
                            valid = false;
                            break;
                        }
                    }
                }
            }
            catch(Exception ex)
            {
                valid = false;
                Log.Error(ex);
            }
            return valid;
        }

        private static bool ValidateImageSize(string fileName, byte[] bytes)
        {
            var memoryStream = new MemoryStream(bytes);
            var valid = true;
            try
            {
                Image image = Image.FromStream(memoryStream);
                if
                (
                    image.Height < (imageSlotRestrictions[fileName].MinHeight ?? 0)
                    || image.Width < (imageSlotRestrictions[fileName].MinWidth ?? 0)
                    || image.Height > imageSlotRestrictions[fileName].MaxHeight
                    || image.Width > imageSlotRestrictions[fileName].MaxWidth
                )
                {
                    valid = false;
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                memoryStream.Dispose();
            }
            return valid;
        }

        private static bool ValidateSharedImageData(string fileName, byte[] file)
        {
            var valid = true;
            if (!imageSlotRestrictions.ContainsKey(fileName))
            {
                valid = false;
            }
            if (file.Length > serverMaxUserImageAssetSize)
            {
                valid = false;
            }
            if(imageSlotRestrictions.ContainsKey(fileName) && file.Length > imageSlotRestrictions[fileName].MaxFileSize)
            {
                valid = false;
            }
            return valid;
        }

        private static bool ValidatePngHeader(byte[] bytes)
        {
            var valid = true;
            for (var i = 0; i < pngHeader.Length; i++)
            {
                if (pngHeader[i] != bytes[i])
                {
                    valid = false;
                }
                if(i >= pngHeader.Length + 1)
                {
                    break;
                }
            }
            return valid;
        }

        private static bool ValidateJpgHeader(byte[] bytes)
        {
            var valid = true;

            if (bytes.Length > serverMaxUserImageAssetSize)
            {
                valid = false;
            }

            if (valid)
            {
                for (var i = 0; i < jpgHeader.Length; i++)
                {
                    if (jpgHeader[i] != bytes[i])
                    {
                        valid = false;
                    }
                    if (i > jpgHeader.Length + 1)
                    {
                        break;
                    }
                }
            }
            return valid;
        }

        public static bool ValidateVideoFile(string fileName, byte[] file)
        {
            var valid = true;

            try
            {
                if (file.Length > serverMaxUserVideoAssetSize)
                {
                    valid = false;
                }

                if (!videoSlotRestrictions.ContainsKey(fileName))
                {
                    valid = false;
                }
                else
                {
                    if (file.Length > videoSlotRestrictions[fileName].MaxFileSize)
                    {
                        valid = false;
                    }
                }
                // Todo: Validate h.264 mp4 file format and any others we may support
            }
            catch (Exception ex)
            {
                valid = false;
                Log.Error(ex);
            }
            return valid;
        }

        public static bool SaveVideoAssetToS3(int productID, string fileName, string fileType, string assetID, byte[] bytes, string bucket, string keypath, HttpContext context)
        {
            var success = false;
            var memoryStream = new MemoryStream(bytes);
            try
            {
                //var bytes = FormFileHelper.ReadFileBytes(file);// Leave for debugging
                S3Helper.UploadToS3(memoryStream, bucket, keypath, kmsKeyID: null, encryptionMethod: null);
                success = SaveAssetToDatabase(productID, assetID, AssetType.Video, fileName, fileType, bucket, keypath, context);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                memoryStream.Dispose();
            }
            return success;
        }

        public static string GenerateAssetID()
        {
            return Guid.NewGuid().ToString("d");
        }

        public static bool SaveImageAssetToS3(int productID, string assetID, string fileName, string fileType, string contentType, byte[] bytes, string bucket, string keypath, HttpContext context)
        {
            var success = false;
            var memoryStream = new MemoryStream(bytes);
            try
            {
                S3Helper.UploadToS3(memoryStream, bucket, keypath, kmsKeyID: null, encryptionMethod: null);
                success = SaveAssetToDatabase(productID, assetID, AssetType.Image, fileName, fileType, bucket, keypath, context);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                memoryStream.Dispose();
            }
            return success;
        }

        public static List<ProductAssetResponseData> GetAssetsByProductID(int productID, HttpContext context)
        {
            var assets = new List<ProductAssetResponseData>();
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText = $@"select product_id, asset_id, asset_type, asset_location, asset_file_type, slot_data from {Constants.DatabaseSchema}microservice_twitch_atlas_product_assets where product_id = @productID;";
                        command.Parameters.AddWithValue("@productID", productID);
                        using (var reader = new DataReaderWithMeasurements(command, context, "get_assets_by_product_id").MysqlReader)
                        {
                            if (reader.HasRows)
                            {
                                while (reader.Read())
                                {
                                    try
                                    {
                                        var item = new ProductAssetResponseData()
                                        {
                                            ProductID = reader.GetInt32(0),
                                            AssetID = reader.GetString(1),
                                            AssetType = (Constants.AssetType)reader.GetInt32(2),
                                            AssetLocation = reader.GetString(3),
                                            AssetFileType = reader.GetString(4),
                                        };
                                        assets.Add(item);
                                    }
                                    catch (Exception ex)
                                    {
                                        Log.Error(ex);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
                assets = null;
            }
            return assets;
        }

        public static ProductAssetResponseData GetAssetsByProductIDAndLocation(int productID, string assetLocation, HttpContext context)
        {
            ProductAssetResponseData asset = null;
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText = $@"select product_id, asset_id, asset_type, asset_location, asset_file_type from {Constants.DatabaseSchema}microservice_twitch_atlas_product_assets where product_id = @productID and asset_location = @assetLocation;";
                        command.Parameters.AddWithValue("@productID", productID);
                        command.Parameters.AddWithValue("@assetLocation", assetLocation);
                        using (var reader = new DataReaderWithMeasurements(command, context, "get_assets_by_product_id_location").MysqlReader)
                        {
                            if (reader.HasRows)
                            {
                                while (reader.Read())
                                {
                                    asset = new ProductAssetResponseData()
                                    {
                                        ProductID = reader.GetInt32(0),
                                        AssetID = reader.GetString(1),
                                        AssetType = (Constants.AssetType)reader.GetInt32(2),
                                        AssetLocation = reader.GetString(3),
                                        AssetFileType = reader.GetString(4)
                                    };
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            return asset;
        }

        private static bool SaveAssetToDatabase(int productID, string assetID, AssetType assetType, string assetLocation, string assetFileType, string bucket, string keypath, HttpContext context)
        {
            var success = false;
            try
            {
                string data = StringHelpers.StringHelper.GzipBase64Encode(
                    JsonConvert.SerializeObject(
                        new ProductAssetData()
                        {
                            S3Bucket = bucket,
                            S3Keypath = keypath,
                            CreatedBy = context?.User?.Identity?.Name,
                            CreatedOn = DateTime.UtcNow
                        })
                    , 8000);
                using (var conn = DBManagerMysql.GetConnection())
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText =
                        $@"
                            replace into {Constants.DatabaseSchema}microservice_twitch_atlas_product_assets
                            (product_id, asset_id, asset_type, asset_location, asset_file_type, slot_data, hash)
                            select 
                                @productID as product_id, 
                                @assetID as asset_id, 
                                @assetType as asset_type,
                                @assetLocation as asset_location,
                                @assetFileType as asset_file_type,
                                @data as slot_data,
                                @hash as hash
                            ;
                        ";
                        command.Parameters.AddWithValue("@productID", productID);
                        command.Parameters.AddWithValue("@assetID", assetID);
                        command.Parameters.AddWithValue("@assetType", (int)assetType);
                        command.Parameters.AddWithValue("@assetLocation", assetLocation);
                        command.Parameters.AddWithValue("@assetFileType", assetFileType);
                        command.Parameters.AddWithValue("@data", data);
                        command.Parameters.AddWithValue("@hash", HashHelper.SimpleRandomLetterSequence.RandomHash());
                        command.ExecuteNonQueryWithMeasurements("save-asset", context: context);
                    }
                }
                success = true;
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }
            return success;
        }
    }
}
