﻿using Microsoft.AspNetCore.Http;
using Newtonsoft.Json;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ApiModels;
using Resonance.Core.Models.AuthModels;
using Resonance.Core.Models.FilterModels;
using Resonance.Core.Models.MetaDataModels;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;

namespace Resonance.Core.Helpers.AuthHelpers
{
    public class AtlasEndpointAuthHelper
    {
        public AuthPermittedModel VerifyUser(HttpContext context, ConcurrentBag<ElapsedTimeModel> metrics, List<int> pccIDs)
        {
            if (Constants.AppConfig.Application.BypassAllOwnershipChecks)
            {
                return new AuthPermittedModel()
                {
                    StatusCode = 200,
                    IsVerified = true
                };
            }

            AuthPermittedModel result = new AuthPermittedModel();

            bool continueEvaluating = true;
            try
            {
                if (pccIDs == null || pccIDs.Count == 0)
                {
                    result.IsVerified = false;
                    result.StatusCode = 400;
                    result.ErrorMessage = "No data provided for 'premium_content_creator_id'";
                    continueEvaluating = false;
                }
            }
            catch (Exception ex)
            {
                result.IsVerified = false;
                result.StatusCode = 400;
                result.ErrorMessage = "An error occurred while processing the requested filters";
                Log.Error(ex);
                continueEvaluating = false;
            }

            var authresult = ProcessPremiumContentCreatorIDList(ref result, ref continueEvaluating, ref pccIDs, context, metrics);

            // Final evaluation of permissions
            if (authresult.AuthResults.Values.Any(x => x != 200))
            {
                authresult.StatusCode = 403;
                authresult.ErrorMessage = $@"Requested Premium Content Creator is denied. {result.ErrorMessage}";
            }

            return authresult;
        }

        public AuthPermittedModel VerifyUser(HttpContext context, ConcurrentBag<ElapsedTimeModel> metrics, int premiumContentCreatorID)
        {
            if(premiumContentCreatorID == 0)
            {
                return new AuthPermittedModel()
                {
                    StatusCode = 400,
                    IsVerified = false,
                    ErrorMessage = "Premium Content Creator ID is invalid"
                };
            }
            if (Constants.AppConfig.Application.BypassAllOwnershipChecks)
            {
                return new AuthPermittedModel()
                {
                    StatusCode = 200,
                    IsVerified = true
                };
            }

            AuthPermittedModel result = new AuthPermittedModel();

            bool continueEvaluating = true;
            var pccIDs = new List<int>();
            try
            {
                if (premiumContentCreatorID <= 0)
                {
                    result.IsVerified = false;
                    result.StatusCode = 400;
                    result.ErrorMessage = "Invalid 'premium_content_creator_id'";
                    continueEvaluating = false;
                }
                pccIDs.Add(premiumContentCreatorID);
            }
            catch (Exception ex)
            {
                result.IsVerified = false;
                result.StatusCode = 400;
                result.ErrorMessage = "An error occurred while processing the requested filters";
                Log.Error(ex);
                continueEvaluating = false;
            }

            var authresult = ProcessPremiumContentCreatorIDList(ref result, ref continueEvaluating, ref pccIDs, context, metrics);

            // Final evaluation of permissions
            if (authresult.AuthResults.Values.Any(x => x != 200))
            {
                authresult.StatusCode = 403;
                authresult.ErrorMessage = $@"Requested Premium Content Creator is denied. {result.ErrorMessage}";
            }

            return authresult;
        }


        /// <summary>
        /// User Authorized, RestrictToOwner
        /// </summary>
        public AuthPermittedModel VerifyUser(HttpContext context, ConcurrentBag<ElapsedTimeModel> metrics, ref ListingFilter filter)
        {
            if (Constants.AppConfig.Application.BypassAllOwnershipChecks)
            {
                return new AuthPermittedModel()
                {
                    StatusCode = 200,
                    IsVerified = true
                };
            }
            AuthPermittedModel result = new AuthPermittedModel();
            var pccIDs = new List<int>();

            bool continueEvaluating = true;

            try
            {
                if (!filter?.QueryFilters?.Any(x => x.Key == "premium_content_creator_id") == true)
                {
                    result.IsVerified = false;
                    result.StatusCode = 400;
                    result.ErrorMessage = "This endpoint requires at least one filter that contains 'premium_content_creator_id'";
                    continueEvaluating = false;
                }
                foreach(var pccFilter in filter.QueryFilters)
                {
                    if(pccFilter.Key == "premium_content_creator_id")
                    {
                        if(pccFilter.Value != null)
                        {
                            pccIDs.Add(Convert.ToInt32(pccFilter.Value));
                        }
                        else
                        {
                            pccIDs.AddRange(pccFilter.ValueArray.Cast<int>());
                        }
                    }
                }
                if(pccIDs.Count > 0)
                {
                    pccIDs = pccIDs.Distinct().ToList();
                }
                else
                {
                    result.IsVerified = false;
                    result.StatusCode = 400;
                    result.ErrorMessage = "No Premium Content Creators were found after pre-processing step";
                    continueEvaluating = false;
                }
            }
            catch (Exception ex)
            {
                result.IsVerified = false;
                result.StatusCode = 400;
                result.ErrorMessage = "An error occurred while processing the requested filters";
                Log.Error(ex);
                continueEvaluating = false;
            }

            var authresult = ProcessPremiumContentCreatorIDList(ref result, ref continueEvaluating, ref pccIDs, context, metrics);

            // Final evaluation of permissions
            if (authresult.AuthResults.Values.Any(x => x != 200))
            {
                authresult.StatusCode = 403;
                authresult.ErrorMessage = $@"Requested Premium Content Creator is denied. {result.ErrorMessage}";
            }

            return authresult;
        }

        private AuthPermittedModel ProcessPremiumContentCreatorIDList(ref AuthPermittedModel result, ref bool continueEvaluating, ref List<int> pccIDs, HttpContext context, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            if(pccIDs == null || pccIDs.Count == 0)
            {
                result.IsVerified = false;
                result.StatusCode = 400;
                result.ErrorMessage = "No premium content creator ids found";
                continueEvaluating = false;
            }
            var stopwatch = new Stopwatch();
            stopwatch.Start();

            try
            {
                if (continueEvaluating)
                {
                    try
                    {
                        var hasAnyAccounts = false;
                        // If the user is an administrator, handle potential special cases
                        if (!Constants.AppConfig.Application.BypassAdminOwnershipChecks)
                        {
                            if (AdministratorGroup(context, metrics))
                            {
                                result.IsVerified = true;
                                // Mark all pcc as allowed
                                foreach (var pcc in pccIDs)
                                {
                                    result.AuthResults.Add(pcc, 200);
                                }
                                result.StatusCode = 200;
                                continueEvaluating = false;

                                Log.Metric("bypass_ownership_check_admin", context, true);
                            }
                        }

                        if (continueEvaluating)
                        {
                            // If the user can manage any PCC, we skip a lot of the ownership checks
                            if (CanManageAnyPcc(context, metrics))
                            {
                                result.IsVerified = true;
                                // Mark all pcc as allowed
                                foreach (var pcc in pccIDs)
                                {
                                    result.AuthResults.Add(pcc, 200);
                                }
                                result.StatusCode = 200;
                                continueEvaluating = false;
                                Log.Metric($@"bypass_ownership_check_cmap", context, true);
                            }
                        }

                        if (continueEvaluating)
                        {
                            if (VerifyDatabaseHasAnyAccountOwnership(context, metrics))
                            {
                                hasAnyAccounts = true;
                            }

                            if (hasAnyAccounts)
                            {
                                var ownershipResults = VerifyDatabaseSpecificAccountOwnership(context, metrics, pccIDs);
                                result.AuthResults = ownershipResults;
                                // If any of the scenarios in this if are true, the entire request should fail with an unauthorized notice
                                if (ownershipResults == null || ownershipResults.Keys.Count < pccIDs.Count || ownershipResults.Values.Any(x => x != 200))
                                {
                                    result.IsVerified = false;
                                    result.StatusCode = 403;
                                    if (ownershipResults == null)
                                    {
                                        result.ErrorMessage = $@"No information found for ownership";
                                    }
                                    if (ownershipResults != null && ownershipResults.Keys.Count != pccIDs.Count)
                                    {
                                        result.ErrorMessage = $@"Premium content creator id count mismatch";
                                    }
                                    if (ownershipResults != null && ownershipResults.Values.Any(x => x != 200))
                                    {
                                        result.ErrorMessage = $@"The following Premium Content Creator IDs were unauthorized: {string.Join(",", ownershipResults.Where(x => x.Value != 200).Select(x => x.Key).ToArray())}";
                                    }
                                    continueEvaluating = false;
                                }
                                // Explicit check to make sure no accidental else-fall-through-permitted
                                if (ownershipResults != null && ownershipResults.Keys.Count > 0 && ownershipResults.Keys.Count == pccIDs.Count && ownershipResults.Values.All(x => x == 200))
                                {
                                    result.IsVerified = true;
                                    result.StatusCode = 200;
                                    result.ErrorMessage = null;
                                    continueEvaluating = false;
                                }
                            }
                            else
                            {
                                result.StatusCode = 403;
                                result.IsVerified = false;
                                result.ErrorMessage = "User is not assigned to any accounts";
                                continueEvaluating = false;
                            }
                        }
                    }
                    catch (Exception ex)
                    {
                        result.IsVerified = false;
                        result.StatusCode = 400;
                        result.ErrorMessage = "An error occurred while processing user ownership";
                        Log.Error(ex);
                        continueEvaluating = false;
                    }
                }
            }
            catch (Exception ex)
            {
                result.IsVerified = false;
                result.StatusCode = 400;
                result.ErrorMessage = "An error occurred while processing ownership";
                Log.Error(ex);
                continueEvaluating = false;
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "verify-ownership-check",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return result;
        }

        private bool AdministratorGroup(HttpContext context, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var stopwatch = new Stopwatch();
            var success = false;
            try
            {
                stopwatch.Start();
                if (PermissionHelper.HasPermission(context, ConstantsPermissions.Atlas.Administrator))
                {
                    success = true;
                }
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "administrator-group-check",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return success;
        }

        private bool CanManageAnyPcc(HttpContext context, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var stopwatch = new Stopwatch();
            var success = false;
            try
            {
                stopwatch.Start();
                if (PermissionHelper.HasPermission(context, ConstantsPermissions.Atlas.CanManageAnyPremiumContentCreators))
                {
                    success = true;
                }
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "administrator-group-check",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
            return success;
        }

        /// <summary>
        /// Queries the database using the information provided to determine whether the user should have access to the data
        /// </summary>
        private bool VerifyDatabaseHasAnyAccountOwnership(HttpContext context, ConcurrentBag<ElapsedTimeModel> metrics)
        {
            var path = context.Request.Path.Value;
            AuthTokenData token = ((AuthTokenData)context.Items[UserAuthDataContext.AuthTokenDataKey]) ?? null;
            if(token == null || string.IsNullOrWhiteSpace(token.User))
            {
                return false;
            }
            var stopwatch = new Stopwatch();
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText =
                        $@"
                            select
	                            case when coalesce(sum(is_account_owner), 0) > 0
		                            then true
		                            else false
                                end as is_account_owner
                            from
                            (
	                            select
		                            (
		                                select count(*) 
		                                from {Constants.DatabaseSchema}microservice_twitch_atlas_pcc_to_am_map as b 
		                                where 
                                            a.account_manager_id = b.account_manager_id
	                                )
	                            as is_account_owner
	                            from {Constants.DatabaseSchema}microservice_twitch_atlas_account_manager as a
	                            where account_manager_ldap_name = @account_manager_ldap_name
                            ) as a
                        ";
                        command.Parameters.AddWithValue("@account_manager_ldap_name", token.User);
                        try
                        {
                            var result = Convert.ToBoolean(command.ExecuteScalarWithMeasurements<long>("ownership_check", context: context));
                            return result;
                        }
                        catch (Exception ex)
                        {
                            Log.Error(ex);
                            return false;
                        }
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
                return false;
            }
            finally
            {
                stopwatch.Stop();
                metrics.Add(new ElapsedTimeModel()
                {
                    MetricName = "ownership-check",
                    ElapsedMS = stopwatch.ElapsedMilliseconds
                });
            }
        }

        /// <summary>
        /// Queries the database to determine whether the user should have access to the requested PCC ID
        /// </summary>
        private Dictionary<int, int> VerifyDatabaseSpecificAccountOwnership(HttpContext context, ConcurrentBag<ElapsedTimeModel> metrics, List<int> pccIDs)
        {
            var pccResults = new Dictionary<int, int>();

            try
            {
                var query = new StringBuilder();

                // Default all PCC to forbidden
                foreach (var pcc in pccIDs)
                {
                    if(query.Length > 0)
                    {
                        query.AppendLine("union all");
                    }
                    pccResults.Add(pcc, 403);
                    query.AppendLine(
                    $@"
                        select {pcc} as premium_content_creator_id, 200 as status_code 
                        from {Constants.DatabaseSchema}microservice_twitch_atlas_account_manager as a
                        inner join {Constants.DatabaseSchema}microservice_twitch_atlas_pcc_to_am_map as b
	                        on a.account_manager_id = b.account_manager_id
                        where 
	                        b.premium_content_creator_id = {pcc}
	                        and a.account_manager_ldap_name = @ldapname
                    ");
                }
                if(query.Length > 0)
                {
                    var continueProcessing = false;
                    AuthTokenData token = ((AuthTokenData)context.Items[UserAuthDataContext.AuthTokenDataKey]) ?? null;
                    if (token == null || string.IsNullOrWhiteSpace(token.User))
                    {
                        continueProcessing = false;
                    }
                    else
                    {
                        continueProcessing = true;
                    }

                    if (continueProcessing)
                    {
                        using (var conn = DBManagerMysql.GetConnection(true))
                        {
                            using (var command = conn.GetCommand())
                            {
                                command.CommandText = $@"{query.ToString()};"; // NOTE: There is a ';' here inside the quotes intentionally.
                                Log.Verbose($@"{query.ToString().Replace("@ldapname", $"'{token.User}'")}");
                                command.Parameters.AddWithValue("@ldapname", token.User);
                                using (var reader = new DataReaderWithMeasurements(command, context, "get_specific_account_ownership").MysqlReader)
                                {
                                    if (reader.HasRows)
                                    {
                                        while (reader.Read())
                                        {
                                            var pcc = reader.GetInt32(0);
                                            var statuscode = reader.GetInt32(1);
                                            if (!pccResults.ContainsKey(pcc))
                                            {
                                                throw new ArgumentOutOfRangeException("Premium Content Creator ID is missing from list");
                                            }
                                            else
                                            {
                                                pccResults[pcc] = statuscode;
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex);
            }
            // Null results in the event something happened in the wash
            if(pccResults.Keys.Count != pccIDs.Count)
            {
                pccResults = null;
            }
            return pccResults;
        }

    }
}
