﻿using Amazon.CloudWatch;
using Microsoft.AspNetCore.Http;
using Microsoft.IdentityModel.Tokens;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IdentityModel.Tokens.Jwt;
using System.Text;

namespace Resonance.Core.Helpers.ApiHelpers
{
    public class JWTData
    {
        public string Profile { get; set; }
        public List<string> Groups { get; set; }
        public string Token { get; set; }
        public DateTime ExpireDate { get; set; }
    }

    public static class JWTHelper
    {
        private static string privateKey = "";
        private static TimeSpan tokenValidTimeSpan = TimeSpan.FromDays(30);
        private static SigningCredentials defaultCredentials;
        static JWTHelper()
        {
            privateKey = S3Helper.ReadTextFromS3("resonance-configuration", $"{Constants.AppConfig.Application.Environment}/credentials/auth/jwt");
            defaultCredentials = GetSigningCredentials();
        }

        public static JwtPayload VerifyJWT(string token, HttpContext context)
        {
            try
            {
                var credentials = defaultCredentials;
                var handler = new JwtSecurityTokenHandler();
                var securityToken = handler.ReadToken(token) as JwtSecurityToken;
                return securityToken.Payload;
            }
            catch(Exception ex)
            {
                Log.Error(ex, "jwthelper_verifyjwt_error", context);
                CloudwatchHelper.EnqueueMetricRequest("jwthelper_verifyjwt_error", 1, context, StandardUnit.Count);

                throw;
            }
        }

        public static SigningCredentials GetSigningCredentials(string suppliedKey = null, string algo = null, HttpContext context = null)
        {
            try
            {
                var securityKey = new SymmetricSecurityKey(Encoding.UTF8.GetBytes(suppliedKey ?? privateKey));
                var credentials = new SigningCredentials(securityKey, algo ?? SecurityAlgorithms.HmacSha256Signature);
                return credentials;
            }
            catch (Exception ex)
            {
                Log.Error(ex, "jwthelper_getsigncred_error", context);
                CloudwatchHelper.EnqueueMetricRequest("jwthelper_getsigncred_error", 1, context, StandardUnit.Count);

                throw;
            }
        }

        public static JWTData MakeAuthJWT(string siteID, dynamic authInfo, HttpContext context)
        {
            try
            {
                SigningCredentials credentials = GetSigningCredentials();

                var header = new JwtHeader(credentials);
                List<string> groups = new List<string>();
                if (authInfo.memberOf != null)
                {
                    try
                    {
                        string singleGroup = authInfo.memberOf;
                        groups.Add(singleGroup);
                    }
                    catch (Exception)
                    {

                        foreach (dynamic group in authInfo.memberOf)
                        {
                            string groupInfo = group;
                            string[] groupParts = groupInfo.Split(',');
                            if (groupParts.Length >= 2)
                            {
                                string groupName = groupParts[0].Replace("cn=", "");
                                string groupType = groupParts[1].Replace("ou=", "");
                                if (groupType == "Groups")
                                {
                                    groups.Add(groupName);
                                }
                            }
                        }
                    }
                }

                string profile = authInfo.profile;
                string email = authInfo.email;

                DateTime expireDate = DateTime.UtcNow + tokenValidTimeSpan;

                var jwtPayload = new JwtPayload
                {
                    {
                        "site_id", siteID
                    },
                    {
                        "user", profile
                    },
                    {
                        "expires", expireDate
                    },
                    {
                        "email", email
                    },
                    {
                        "groups", groups
                    }
                };

                var secToken = new JwtSecurityToken(header, jwtPayload);
                var handler = new JwtSecurityTokenHandler();

                return new JWTData()
                {
                    ExpireDate = expireDate,
                    Profile = profile,
                    Groups = groups,
                    Token = handler.WriteToken(secToken)
                };
            }
            catch (Exception ex)
            {
                Log.Error(ex, "jwthelper_makeauthjwt_error", context);
                CloudwatchHelper.EnqueueMetricRequest("jwthelper_makeauthjwt_error", 1, context, StandardUnit.Count);

                throw;
            }
        }        
    }
}
