﻿using Amazon.CloudWatch;
using Microsoft.AspNetCore.Http;
using Newtonsoft.Json;
using Resonance.Core;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.ConfigurationModels;
using Resonance.Core.Models.ConfigurationModels.Application;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Threading.Tasks;
using System.Web;

namespace Resonance.Core.Helpers.ApiHelpers
{
    public class OAuthHelpers
    {
        /// <summary>
        /// This ARN is intentionally hard coded. Map IAM permissions as appropriate through the web UI to support
        /// multi-project access. This ARN exists outside the regular permission system.
        /// The manifest file is protected the ManifestArn, but the individual oauth services are protected by their service ARN.
        /// </summary>
        private const string oauthManifestArn = "";
        private const string oauthBucket = "resonance-configuration";
        private string oauthManifestKeypath = $"{Constants.AppConfig.Application.Environment}/credentials/resonance/oauth/manifest.json.gz";
        private OAuthModel oauth { get; set; }
        private string oauthSecret { get; set; }

        public OAuthHelpers(Constants.SiteID siteId)
        {
            var s3Data = S3Helper.ReadTextFromS3(oauthBucket, oauthManifestKeypath, isCompressed: true);
            var manifest = JsonConvert.DeserializeObject<Dictionary<string, OAuthManifestModel>>(s3Data);
            var key = Enum.GetName(typeof(Constants.SiteID), siteId) ?? "";
            if (!manifest.ContainsKey(key))
            {
                throw new KeyNotFoundException($"No oauth entry was found for {Constants.AppConfig.Application.Name} in the manifest file. Have you added it using the Credential Creator?");
            }
            var oauthTarget = manifest?[key]?.S3OAuthKeypath;
            if (!string.IsNullOrWhiteSpace(oauthTarget))
            {
                var authData = S3Helper.ReadTextFromS3(oauthBucket, oauthTarget, isCompressed: true);
                var oauthData = JsonConvert.DeserializeObject<OAuthModel>(authData);
                oauth = oauthData;
                oauthSecret = oauthData.ClientSecret;
            }
            else
            {
                throw new InvalidOperationException(message: $@"OAuth S3 Keypath is blank. Site will be unable to load.");
            }
        }

        public string GetRedirectUrl(Constants.OAuthType oauthType)
        {
            string redirectURL = HttpUtility.UrlEncode(oauth.RedirectUrl);
            string scopes = HttpUtility.UrlEncode(oauth.Scopes);
            return $"{oauth.Endpoint}/as/authorization.oauth2?client_id={oauth.ClientID}&redirect_uri={redirectURL}&state={HttpUtility.UrlEncode(Guid.NewGuid().ToString().Replace("-", ""))}&scope={scopes}&response_type=code";
        }

        public string VerifyAuthorizeToken(string token, HttpContext context)
        {
            try
            {
                using (WebClient client = new WebClient())
                {
                    client.Headers.Add("Content-Type", "application/x-www-form-urlencoded");
                    string tokenEndPoint = $"{oauth.Endpoint}/as/token.oauth2";
                    string payload = $"grant_type=authorization_code&code={token}&redirect_uri={oauth.RedirectUrl}&client_id={oauth.ClientID}&client_secret={oauthSecret}";
                    try
                    {
                        return client.UploadString(tokenEndPoint, "POST", payload);
                    }
                    catch (WebException ex)
                    {
                        using (var exResponse = new StreamReader(ex.Response.GetResponseStream()))
                        {
                            string error = exResponse.ReadToEnd();
                            Log.Error($"{error} -- {ex}", "oauthhelpers_verifyauthtoken_error", context);
                        }
                    }
                }
                return null;
            }
            catch(Exception ex)
            {
                Log.Error(ex, "oauthhelpers_verifyauthtoken_error", context);
                CloudwatchHelper.EnqueueMetricRequest("oauthhelpers_verifyauthtoken_error", 1, context, StandardUnit.Count);

                throw;
            }
        }

        public string GetUserInfo(string authToken, HttpContext context)
        {
            try
            {
                using (WebClient client = new WebClient())
                {
                    client.Headers.Add("Authorization", $"Bearer {authToken}");
                    return client.UploadString($"{oauth.Endpoint}/idp/userinfo.openid", "POST", "");
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex, "oauthhelpers_getuserinfo_error", context);
                CloudwatchHelper.EnqueueMetricRequest("oauthhelpers_getuserinfo_error", 1, context, StandardUnit.Count);

                throw;
            }
        }

        public async Task<string> RequestTokenToAuthorizationServer(Uri uriAuthorizationServer, string clientId, string scope, string clientSecret, HttpContext context)
        {
            try
            {
                HttpResponseMessage responseMessage;
                using (HttpClient client = new HttpClient())
                {
                    HttpRequestMessage tokenRequest = new HttpRequestMessage(HttpMethod.Post, uriAuthorizationServer);
                    HttpContent httpContent = new FormUrlEncodedContent(
                    new[]
                    {
                        new KeyValuePair<string, string>("grant_type", "client_credentials"),
                        new KeyValuePair<string, string>("client_id", clientId),
                        new KeyValuePair<string, string>("scope", scope),
                        new KeyValuePair<string, string>("client_secret", clientSecret)
                    });
                    tokenRequest.Content = httpContent;
                    responseMessage = await client.SendAsync(tokenRequest);
                }
                return await responseMessage.Content.ReadAsStringAsync();
            }
            catch(Exception ex)
            {
                Log.Error(ex, "oauthhelpers_reqtokauthserv_error", context);
                CloudwatchHelper.EnqueueMetricRequest("oauthhelpers_reqtokauthserv_error", 1, context, StandardUnit.Count);

                throw;
            }
        }
    }
}
