﻿using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc.Filters;
using Newtonsoft.Json;
using Resonance.Core;
using Resonance.Core.Exceptions;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.AuthModels;
using Resonance.Core.Models.ConfigurationModels.Permissions;
using Resonance.Microservices.Exceptions;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using static Resonance.Core.Constants;
using Resonance.Core.Helpers.AuthHelpers;
using Amazon.CloudWatch;

namespace Resonance.Core.Helpers.ApiHelpers
{
    public class PermissionHelper
    {
        public void Initialize()
        {
        }

        public PermissionConfigurationModel ProcessNewItems(Dictionary<PermissionFormType, PermissionSingleItemModel> data)
        {
            try
            {
                PermissionConfigurationModel config = GetPermissionsFromS3(AppConfig.Application.Name);

                // Process new / deleted items
                if (data != null && data.Count > 0)
                {
                    var newRoles = data.Where(x => x.Key == PermissionFormType.NewRole).Select(x => x.Value)?.ToArray();
                    var newPermissions = data.Where(x => x.Key == PermissionFormType.NewPermission).Select(x => x.Value)?.ToArray();
                    var newLdapGroups = data.Where(x => x.Key == PermissionFormType.NewLdapGroup).Select(x => x.Value)?.ToArray();
                    var newLdapNames = data.Where(x => x.Key == PermissionFormType.NewLdapName).Select(x => x.Value)?.ToArray();
                    var newTokens = data.Where(x => x.Key == PermissionFormType.NewToken).Select(x => x.Value)?.ToArray();
                    var newModels = data.Where(x => x.Key == PermissionFormType.NewModel).Select(x => x.Value)?.ToArray();                    

                    var deleteRoles = data.Where(x => x.Key == PermissionFormType.DeleteRole).Select(x => x.Value)?.ToArray();
                    var deletePermissions = data.Where(x => x.Key == PermissionFormType.DeletePermission).Select(x => x.Value)?.ToArray();
                    var deleteLdapGroups = data.Where(x => x.Key == PermissionFormType.DeleteLdapGroup).Select(x => x.Value)?.ToArray();
                    var deleteLdapNames = data.Where(x => x.Key == PermissionFormType.DeleteLdapName).Select(x => x.Value)?.ToArray();
                    var deleteTokens = data.Where(x => x.Key == PermissionFormType.DeleteToken).Select(x => x.Value)?.ToArray();
                    var deleteModels = data.Where(x => x.Key == PermissionFormType.DeleteModel).Select(x => x.Value)?.ToArray();

                    foreach (var role in newRoles)
                    {
                        if (!config.Roles.Any(x => x.RoleName == role.NewValue))
                        {
                            config.Roles.Add(new RoleModel()
                            {
                                RoleID = role.ItemID,
                                RoleName = role.NewValue,
                                RoleDescription = role.ItemDescription
                            });
                        }
                    }

                    foreach (var permission in newPermissions)
                    {
                        if (!config.Permissions.Any(x => x.PermissionName == permission.NewValue))
                        {
                            config.Permissions.Add(new PermissionModel()
                            {
                                PermissionID = permission.ItemID,
                                PermissionName = permission.NewValue,
                                PermissionDescription = permission.ItemDescription
                            });
                        }
                    }

                    foreach (var ldap in newLdapGroups)
                    {
                        if (!config.LdapGroups.Any(x => x.LdapGroupName == ldap.NewValue))
                        {
                            config.LdapGroups.Add(new LdapGroupModel()
                            {
                                LdapGroupId = ldap.ItemID,
                                LdapGroupName = ldap.NewValue,
                                LdapGroupDescription = ldap.ItemDescription
                            });
                        }
                    }

                    foreach (var ldap in newLdapNames)
                    {
                        if (!config.LdapNames.Any(x => x.LdapName == ldap.NewValue))
                        {
                            config.LdapNames.Add(new LdapNameModel()
                            {
                                LdapNameID = ldap.ItemID,
                                LdapName = ldap.NewValue,
                                LdapNameDescription = ldap.ItemDescription
                            });
                        }
                    }

                    foreach (var token in newTokens)
                    {
                        if (!config.Tokens.Any(x => x.TokenName == token.NewValue))
                        {
                            config.Tokens.Add(new TokenModel()
                            {
                                TokenID = token.ItemID,
                                TokenName = token.NewValue,
                                TokenDescription = token.ItemDescription,
                                WhitelistedIPSource = token.IPWhitelist
                            });
                        }
                    }

                    foreach(var item in newModels)
                    {
                        var model = (PermissionColumnItemModel)item;
                        if (!config.ColumnPermissions.Any(x => x.ColumnName == model.NewValue))
                        {
                            config.ColumnPermissions.Add(new ColumnPermissionModel()
                            {
                                ModelID = model.ItemID,
                                ColumnName = model.NewValue,
                                ModelDescription = model.ItemDescription,
                                RequiredPermissionList = model.Columns                                
                            });
                        }
                    }

                    // Delete requested items
                    foreach (var role in deleteRoles)
                    {
                        config.Roles.RemoveAll(x => x.RoleID == role.ItemID);
                        foreach (var r in config.RoleToPermissionMap)
                        {
                            r.Value.RemoveAll(x => x == role.ItemID);
                        }
                    }

                    foreach (var permission in deletePermissions)
                    {
                        config.Permissions.RemoveAll(x => x.PermissionID == permission.ItemID);
                        foreach (var perm in config.RoleToPermissionMap)
                        {
                            perm.Value.RemoveAll(x => x == permission.ItemID);
                        }                        
                        
                        //When removing a permission remove any references to it in the ColumnPermission system
                        foreach(var item in config.ColumnPermissions)
                        {                            
                            item.RequiredPermissionList.RemoveAll(x => x == permission.ItemID);                            
                        }

                        //If a column has no assigned permissions left then just remove it
                        config.ColumnPermissions.RemoveAll(x => !x.RequiredPermissionList.Any());
                    }

                    foreach (var ldapGroup in deleteLdapGroups)
                    {
                        config.LdapGroups.RemoveAll(x => x.LdapGroupId == ldapGroup.ItemID);
                        foreach(var group in config.RoleToLdapGroupMap)
                        {
                            group.Value.RemoveAll(x => x == ldapGroup.ItemID);
                        }
                    }

                    foreach (var ldapName in deleteLdapNames)
                    {
                        config.LdapNames.RemoveAll(x => x.LdapNameID == ldapName.ItemID);
                        foreach (var name in config.RoleToLdapNameMap)
                        {
                            name.Value.RemoveAll(x => x == ldapName.ItemID);
                        }
                    }

                    foreach (var token in deleteTokens)
                    {
                        config.Tokens.RemoveAll(x => x.TokenID == token.ItemID);
                        foreach (var tok in config.RoleToTokenMap)
                        {
                            tok.Value.RemoveAll(x => x == token.ItemID);
                        }
                    }

                    foreach (var model in deleteModels)
                    {
                        config.ColumnPermissions.RemoveAll(x => x.ModelID == model.ItemID);
                        foreach (var m in config.RoleToTokenMap)
                        {
                            m.Value.RemoveAll(x => x == model.ItemID);
                        }
                    }

                    return config;
                }
                return null;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                return null;
            }
        }

        public PermissionConfigurationModel ProcessNewMaps(PermissionConfigurationModel data)
        {
            // Placeholder method for future work.
            return data;
        }

        public PermissionConfigurationModel GetPermissionsFromS3(string projectName, string environmentOverride = null)
        {
            string environment = Constants.AppConfig.Application.Environment;
            if (!string.IsNullOrWhiteSpace(environmentOverride))
            {
                environment = environmentOverride;
            }

            if(S3Helper.ExistsInS3("crs-data-export", $"{environment}/Permissions/{projectName}/Permissions.config.gz") == true)            
            {
                var data = JsonConvert.DeserializeObject<PermissionConfigurationModel>(S3Helper.ReadTextFromS3("crs-data-export", $"{environment}/Permissions/{projectName}/Permissions.config.gz", true));                                

                // Verify all items exist, in case new items have been added.
                if (data.Permissions == null)
                {
                    data.Permissions = new List<PermissionModel>();
                }
                if(data.Tokens == null)
                {
                    data.Tokens = new List<TokenModel>();
                }
                if(data.Roles == null)
                {
                    data.Roles = new List<RoleModel>();
                }
                if(data.LdapGroups == null)
                {
                    data.LdapGroups = new List<LdapGroupModel>();
                }
                if (data.LdapNames == null)
                {
                    data.LdapNames = new List<LdapNameModel>();
                }
                if (data.ColumnPermissions == null)
                {
                    data.ColumnPermissions = new List<ColumnPermissionModel>();
                }
                if (data.RoleToLdapGroupMap == null)
                {
                    data.RoleToLdapGroupMap = new Dictionary<string, List<string>>();
                }
                if (data.RoleToLdapNameMap == null)
                {
                    data.RoleToLdapNameMap = new Dictionary<string, List<string>>();
                }
                if (data.RoleToPermissionMap == null)
                {
                    data.RoleToPermissionMap = new Dictionary<string, List<string>>();
                }
                if(data.RoleToTokenMap == null)
                {
                    data.RoleToTokenMap = new Dictionary<string, List<string>>();
                }                
                return data;
            }
            else
            {
                return new PermissionConfigurationModel()
                {
                    Permissions = new List<PermissionModel>(),
                    Tokens = new List<TokenModel>(),
                    Roles = new List<RoleModel>(),
                    LdapGroups = new List<LdapGroupModel>(),
                    LdapNames = new List<LdapNameModel>(),
                    RoleToLdapGroupMap = new Dictionary<string, List<string>>(),
                    RoleToLdapNameMap = new Dictionary<string, List<string>>(),
                    RoleToPermissionMap = new Dictionary<string, List<string>>(),
                    RoleToTokenMap = new Dictionary<string, List<string>>(),
                    ColumnList = new List<string>(),
                    ColumnPermissions = new List<ColumnPermissionModel>()
                };
            }
        }        

        public bool? WritePermissionsToS3(string projectName, PermissionConfigurationModel config, HttpContext context = null, string environmentOverride = null)
        {
            bool? success = null;

            if (config != null && config != default(PermissionConfigurationModel))
            {
                try
                {
                    string environment = Constants.AppConfig.Application.Environment;
                    if (!string.IsNullOrWhiteSpace(environmentOverride))
                    {
                        environment = environmentOverride;
                    }
                    // Write primary and backup permissions
                    S3Helper.WriteStringToS3(JsonConvert.SerializeObject(config), "crs-data-export", $"{environment}/Permissions/{projectName}/Permissions.config.gz", true, Constants.AppConfig.Application.KmsArn);
                    S3Helper.WriteStringToS3(JsonConvert.SerializeObject(config), "crs-data-export", $"{environment}/Permissions/{projectName}/Archive/Permissions_{DateTime.UtcNow.ToString("yyyy-MM-dd_HH-mm-ss")}_{context?.User?.Identity?.Name ?? Environment.MachineName}.config.gz", true, Constants.AppConfig.Application.KmsArn);

                    /* This should include all active services */
                    Constants.Permissions = config;
                    switch (projectName)
                    {
                        case "Auth":
                        {
                            try { WebRequestHelper.GetData($"{AppConfig.Endpoints.Auth}permission/update-permission-roles", WebRequestHelper.ServiceToken); } catch (Exception) { }
                            break;
                        }
                        case "Amp":
                        {
                            try { WebRequestHelper.GetData($"{AppConfig.Endpoints.Amp}permission/update-permission-roles", WebRequestHelper.ServiceToken); } catch (Exception) { }
                            break;
                        }
                        case "Atlas":
                        {
                            try { WebRequestHelper.GetData($"{AppConfig.Endpoints.Atlas}permission/update-permission-roles", WebRequestHelper.ServiceToken); } catch (Exception) { }
                            break;
                        }
                    }
                    success = true;
                }
                catch(Exception ex)
                {
                    Log.Error(ex);
                    success = false;
                }
            }

            return success;
        }    
        
        public List<string> GetUserRoles(HttpContext context)
        {
            try
            {
                var stopwatch = new Stopwatch();
                stopwatch.Start();
                var data = new UserAuthDataContext(context);
                stopwatch.Stop();
                Log.Verbose($@"GetUserRoles took {stopwatch.ElapsedMilliseconds}ms");

                return data.Roles;
            }
            catch (Exception ex)
            {
                Log.Error(ex, "permission_helper_getuserroles_error", context);
                CloudwatchHelper.EnqueueMetricRequest("permission_helper_getuserroles_error", 1, context, StandardUnit.Count);

                throw;
            }
        }
        
        public List<string> GetUserPermissions(HttpContext context)
        {
            try
            {
                var stopwatch = new Stopwatch();
                stopwatch.Start();
                var data = new UserAuthDataContext(context);
                stopwatch.Stop();
                Log.Verbose($@"GetUserPermissions took {stopwatch.ElapsedMilliseconds}ms");

                return data.Permissions;
            }
            catch (Exception ex)
            {
                Log.Error(ex, "permission_helper_getuserpermissions_error", context);
                CloudwatchHelper.EnqueueMetricRequest("permission_helper_getuserpermissions_error", 1, context, StandardUnit.Count);

                throw;
            }
        }

        public static bool HasPermission(HttpContext context, string permissionName)
        {
            try
            {
                string key = UserAuthDataContext.UserAuthDataKey;
                return context.Items.ContainsKey(key) ? ((UserAuthData)context.Items[key]).Permissions.Contains(permissionName) : false;
            }
            catch (Exception ex)
            {
                Log.Error(ex, "permission_helper_haspermission_error", context);
                CloudwatchHelper.EnqueueMetricRequest("permission_helper_haspermission_error", 1, context, StandardUnit.Count);

                throw;
            }
        }

        public static IList<string> GetPermissions(HttpContext context)
        {
            try
            {
                string key = UserAuthDataContext.UserAuthDataKey;
                if (context.Items.ContainsKey(key))
                {
                    return ((UserAuthData)context.Items[key]).Permissions;
                }

                return null;
            }
            catch (Exception ex)
            {
                Log.Error(ex, "permission_helper_getpermissions_error", context);
                CloudwatchHelper.EnqueueMetricRequest("permission_helper_getpermissions_error", 1, context, StandardUnit.Count);

                throw;
            }
        }

        public bool VerifyToken(string token, string remoteAddress, HttpContext context)
        {
            try
            {                
                Match apiKeyMatch = UserAuthDataContext.APIKeyRegex.Match(token);
                if (apiKeyMatch.Success)
                {
                    string apiKey = apiKeyMatch.Groups[1].Value;

                    var tokendata = Constants.Permissions.Tokens.Where(x => x.TokenName == apiKey).FirstOrDefault();
                    if (tokendata != null)
                    {
                        if (tokendata.WhitelistedIPSource == null || tokendata.WhitelistedIPSource.Length == 0 || (tokendata.WhitelistedIPSource.Length == 1 && tokendata.WhitelistedIPSource[0] == ""))
                        {
                            return true;
                        }
                        else
                        {
                            if (tokendata.WhitelistedIPSource.Contains(remoteAddress))
                            {
                                return true;
                            }
                        }
                    }
                }   
            }
            catch (Exception ex)
            {
                Log.Error(ex, "permission_helper_verifytoken_error", context);
                CloudwatchHelper.EnqueueMetricRequest("permission_helper_verifytoken_error", 1, context, StandardUnit.Count);

                throw;
            }

            return false;
        }
    }
}
