﻿using Microsoft.AspNetCore.Http;
using Resonance.Core.Models.AuthModels;
using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.Linq;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Services.LdapService;
using System.Net;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.StringHelpers;
using Amazon.CloudWatch;
using Resonance.Core.Helpers.AwsHelpers;

namespace Resonance.Core.Helpers.AuthHelpers
{
    public class UserAuthDataContext
    {
        private static LdapService ldapHelper = new LdapService();
        public const string AuthorizationHeaderKey = @"Authorization";
        public const string MasqueradeHeaderKey = @"Masquerade";
        public static readonly Regex BearerRegex = new Regex("Bearer (.*?)$", RegexOptions.Compiled);
        public static readonly Regex APIKeyRegex = new Regex("ApiKey (.*?)$", RegexOptions.Compiled);
        // Old Authorization
        public const string AuthTokenDataKey = "AuthTokenData";
        public const string MasqueradeDataKey = "MasqueradeAuthTokenData";
        // New Authorization
        public const string UserAuthDataKey = "UserAuthData";
        public const string MasqueradeUserAuthDataKey = "MasqueradeUserAuthData";


        private UserAuthData _data;

        public List<string> Permissions
        {
            get
            {
                return _data?.Permissions ?? new List<string>();
            }            
        }

        public List<string> Roles
        {
            get
            {                
                return _data?.Roles ?? new List<string>();
            }
        }

        public UserAuthDataContext(HttpContext context)
        {
            _data = GetUserAuthData(context) ?? SetUserAuthData(context);
        }

        private UserAuthData GetUserAuthData(HttpContext context)
        {
            try
            {
                if (context.Items.ContainsKey(UserAuthDataKey))
                {
                    return (UserAuthData)context.Items[UserAuthDataKey];
                }

                return null;

            }
            catch (Exception ex)
            {
                Log.Error(ex, "userauthdatacontext_getuserauthdata_error", context);
                CloudwatchHelper.EnqueueMetricRequest("userauthdatacontext_getuserauthdata_error", 1, context, StandardUnit.Count);

                return null;
            }
        }

        private UserAuthData SetUserAuthData(HttpContext context)
        {
            try
            {
                UserAuthData data = PopulateUserAuthData(context);

                if (data != null)
                {
                    if (data.Permissions != null && data.Permissions.Any())
                    {
                        data.Permissions = data.Permissions.Distinct().ToList();
                    }

                    if (data.Roles != null && data.Roles.Any())
                    {
                        data.Roles = data.Roles.Distinct().ToList();
                    }

                    context.Items[UserAuthDataKey] = data;

                    return data;
                }

                return UserAuthData.Empty;
            }
            catch (Exception ex)
            {
                Log.Error(ex, "userauthdatacontext_setuserauthdata_error", context);
                CloudwatchHelper.EnqueueMetricRequest("userauthdatacontext_setuserauthdata_error", 1, context, StandardUnit.Count);

                return UserAuthData.Empty; ;
            }
        }

        private UserAuthData PopulateUserAuthData(HttpContext context)
        {
            try
            {
                string authHeader = context.Request.Headers[AuthorizationHeaderKey];

                if (string.IsNullOrWhiteSpace(authHeader))
                {
                    return UserAuthData.Empty;
                }

                UserAuthData data = null;
                UserAuthData masqueradeData = null;
                AuthTokenData tokenData = null;
                AuthTokenData masqueradeTokenData = null;
                UserAuthDataReaderConfig config = null;
                UserAuthDataReaderConfig masqueradeconfig = null;
                string masqueradeHeader = context.Request.Headers[MasqueradeHeaderKey];
                bool masqueradeRequested = !string.IsNullOrWhiteSpace(masqueradeHeader);

                var auth = CreateUserAuthDataReader(authHeader, context);

                if (auth != null)
                {
                    switch (auth.AuthType)
                    {
                        case UserAuthType.Token:
                        {
                            if (context.Items.ContainsKey(AuthTokenDataKey))
                            {
                                tokenData = (AuthTokenData)context.Items[AuthTokenDataKey];
                            }
                            else
                            {
                                string jwtToken = auth.AuthValue;
                                tokenData = ApiHelpers.AuthHelpers.ValidateAuthToken(Constants.AppConfig.Application.Name, ref jwtToken);
                            }

                            config = UserAuthDataReaderConfig.New().WithToken(tokenData);
                            break;
                        }
                        case UserAuthType.ApiKey:
                        {
                            string remoteAddress = HeaderHelper.GetRequestIP(context);
                            config = UserAuthDataReaderConfig.New().WithApiKey(auth.AuthValue, remoteAddress);
                            break;
                        }
                        default:
                        {
                            data = null;
                            break;
                        }
                    }
                }

                if (config != null)
                {
                    data = auth.ReadData(config);

                    if (!context.Items.ContainsKey(AuthTokenDataKey))
                    {
                        tokenData = tokenData ?? new AuthTokenData { User = data.UserID };
                        // Only allow masquerade if: The application allows it, and the masquerade is requested. Later we check that the target user doesn't have the 'Block Masquerade' permission
                        if (Constants.AppConfig.Application.AllowMasquerade && masqueradeRequested)
                        {
                            string masqueradeID = Guid.NewGuid().ToString("d");
                            string masqueradeType = "Unknown";

                            // Ldap User Masquerade
                            if (data.Permissions.Contains(ConstantsPermissions.Atlas.CanMasqueradeLdap))
                            {
                                Match masqueradeLdapMatch = BearerRegex.Match(masqueradeHeader);
                                if (masqueradeLdapMatch.Success)
                                {
                                    masqueradeType = "Bearer";
                                    var masqueradeLdapData = ldapHelper.GetUserDataByName(masqueradeLdapMatch.Groups[1].Value, context);
                                    // Make sure the user coming out matches the user going in
                                    if (masqueradeLdapData != null && masqueradeLdapData.User != null && !string.IsNullOrWhiteSpace(masqueradeLdapData.User.UserName) && masqueradeLdapData.User.UserName == masqueradeLdapMatch.Groups[1].Value)
                                    {
                                        // Looks like we match, perform group permission analysis
                                        bool allowMasquerade = true;
                                        var blockMasqueradeRoles = Constants.Permissions.RoleToPermissionMap.Where(x => x.Value.Any(y => y == ConstantsPermissions.Atlas.BlockMasquerade)).Select(x => x.Key).Distinct().ToList();

                                        // Handle the user if they are special cased by ldap name
                                        if (allowMasquerade)
                                        {
                                            foreach (var roleGroup in Constants.Permissions.RoleToLdapNameMap.Where(x => x.Value.Any(y => masqueradeLdapData.User.UserName == y)))
                                            {
                                                foreach (var group in roleGroup.Value)
                                                {
                                                    if (blockMasqueradeRoles.Contains(group))
                                                    {
                                                        allowMasquerade = false;
                                                    }
                                                }
                                            }
                                        }

                                        if (allowMasquerade)
                                        {
                                            // Handle user groups
                                            foreach (var roleGroup in Constants.Permissions.RoleToLdapGroupMap.Where(x => x.Value.Any(y => masqueradeLdapData.Groups.Contains(y))))
                                            {
                                                foreach (var group in roleGroup.Value)
                                                {
                                                    if (blockMasqueradeRoles.Contains(group))
                                                    {
                                                        allowMasquerade = false;
                                                    }
                                                }
                                            }
                                        }

                                        // If we can still masquerade, go ahead and set up the masquerade user
                                        if (allowMasquerade)
                                        {
                                            var masqueradeAuth = CreateUserAuthDataReader(masqueradeHeader, context);
                                            string remoteAddress = HeaderHelper.GetRequestIP(context);
                                            var authTokenData = new AuthTokenData()
                                            {
                                                User = masqueradeLdapData.User.UserName,
                                                Groups = masqueradeLdapData.Groups.ToArray(),
                                                Email = masqueradeLdapData.User.Mail,
                                                Expires = DateTime.UtcNow.Date.AddDays(30)
                                            };
                                            masqueradeconfig = UserAuthDataReaderConfig.New().WithToken(authTokenData);
                                            masqueradeData = masqueradeAuth.ReadData(masqueradeconfig);
                                            masqueradeTokenData = authTokenData;
                                            RecordMasqueradeAttempt(context, masqueradeID, masqueradeType, data, masqueradeLdapMatch.Groups[1].Value, true);
                                        }
                                        else
                                        {
                                            RecordMasqueradeAttempt(context, masqueradeID, masqueradeType, data, masqueradeLdapMatch.Groups[1].Value, false);
                                            context.Response.StatusCode = 400;
                                            throw new WebException("Unable to masquerade user.");
                                        }
                                    }
                                }
                            }

                            // Api Key Masquerade
                            if (data.Permissions.Contains(ConstantsPermissions.Atlas.CanMasqueradeApiKey))
                            {
                                Match masqueradeApiKeyMatch = APIKeyRegex.Match(masqueradeHeader);
                                if (masqueradeApiKeyMatch.Success)
                                {
                                    masqueradeType = "ApiKey";
                                    var masqueradeApiKeyData = Constants.Permissions.Tokens.FirstOrDefault(x => x.TokenID == masqueradeApiKeyMatch.Groups[1].Value);
                                    if (masqueradeApiKeyData != null)
                                    {
                                        var masqueradeAuth = CreateUserAuthDataReader(masqueradeHeader, context);
                                        string remoteAddress = HeaderHelper.GetRequestIP(context);
                                        masqueradeconfig = UserAuthDataReaderConfig.New().WithApiKey(masqueradeApiKeyData.TokenName, remoteAddress);
                                        masqueradeData = masqueradeAuth.ReadData(masqueradeconfig);
                                        masqueradeTokenData = masqueradeconfig.TokenData;
                                        RecordMasqueradeAttempt(context, masqueradeID, masqueradeType, data, masqueradeApiKeyMatch.Groups[1].Value, true);
                                    }
                                    else
                                    {
                                        RecordMasqueradeAttempt(context, masqueradeID, masqueradeType, data, masqueradeApiKeyMatch.Groups[1].Value, false);
                                        context.Response.StatusCode = 400;
                                        throw new WebException("Unable to masquerade api key.");
                                    }
                                }
                            }

                            // Special blocks: 
                            // Application has 'Allow masquerade enabled'
                            // Data was found for the masquerade user as well as the authorized user
                            // Authorized Api User cannot match Masquerade User, 
                            // Masquerade user cannot have the 'Block Masquerade' permission.
                            if
                            (
                                Constants.AppConfig.Application.AllowMasquerade
                                && masqueradeRequested
                                && masqueradeconfig != null
                                && masqueradeTokenData != null
                                && !string.IsNullOrWhiteSpace(masqueradeTokenData.User)
                                && data.UserID != masqueradeData.UserID
                                && !masqueradeData.Permissions.Contains(ConstantsPermissions.Atlas.BlockMasquerade)
                            )
                            {
                                data.MasqueradeUser = masqueradeData;
                                masqueradeData.MasqueradeUser = data;
                                masqueradeData.MasqueradeUser.MasqueradeUser = null; // Heh.

                                // TODO: Audit Log entry
                                RecordMasqueradeSuccess(context, masqueradeID, masqueradeType, data, masqueradeData);

                                // Save masquerade and user data to context items and update headers
                                context.Items.Add(AuthTokenDataKey, masqueradeTokenData);
                                context.Items.Add(MasqueradeDataKey, tokenData);
                                context.Items.Add(UserAuthDataKey, masqueradeData);
                                context.Items.Add(MasqueradeUserAuthDataKey, data);
                                context.Response.Headers["MasqueradeSourceUser"] = data.UserID;
                                context.Response.Headers["MasqueradeTargetUser"] = masqueradeData.UserID;
                                return masqueradeData;
                            }
                            else if (masqueradeRequested)
                            {
                                RecordMasqueradeAttempt(context, masqueradeID, masqueradeType, data, masqueradeData.UserID, false);
                                context.Response.StatusCode = 400;
                                throw new WebException("Invalid scenario for masquerade.");
                            }
                        }
                    }
                }

                if (!context.Items.ContainsKey(AuthTokenDataKey) && tokenData != null)
                {
                    context.Items.Add(AuthTokenDataKey, tokenData);
                }
                if (!context.Items.ContainsKey(UserAuthDataKey) && data != null)
                {
                    context.Items.Add(UserAuthDataKey, data);
                }

                return data;
            }
            catch(Exception ex)
            {
                Log.Error(ex, "userauthdatacontext_populateuserauthdata_error", context);
                CloudwatchHelper.EnqueueMetricRequest("userauthdatacontext_populateuserauthdata_error", 1, context, StandardUnit.Count);

                throw;
            }
        }

        /// <summary>
        /// Records whether or not a masquerade attempt was successful
        /// </summary>
        /// <param name="user">Source user who requested the masquerade</param>
        /// <param name="target">May be null</param>
        /// <param name="success">Was the masquerade attempt successful or a failure</param>
        private void RecordMasqueradeAttempt(HttpContext context, string masqueradeID, string masqueradeType, UserAuthData user, string target, bool success)
        {
            // TODO: Write to db table 'true/false' of masquerade attempt
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText =
                        $@"
                            insert into {Constants.DatabaseSchema}microservice_twitch_atlas_masquerade_audit 
                            (
                                masquerade_id, masquerade_timestamp, masquerade_type, source_user, target_user, is_success, path
                            )
                            select @masquerade_id, @masquerade_timestamp, @masquerade_type, @source_user, @target_user, @is_success, @path
                        ";
                        command.Parameters.AddWithValue("@masquerade_id", masqueradeID);
                        command.Parameters.AddWithValue("@masquerade_timestamp", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@masquerade_type", masqueradeType);
                        command.Parameters.AddWithValue("@source_user", user.UserID);
                        command.Parameters.AddWithValue("@target_user", target);
                        command.Parameters.AddWithValue("@is_success", success);
                        command.Parameters.AddWithValue("@path", context.Request.Path.Value.Truncate(255));
                        command.ExecuteNonQueryWithMeasurements("masquerade_attempt");
                    }
                }
            }
            catch(Exception ex)
            {
                Log.Error(ex, "userauthdatacontext_recordmasqueradeattempt_error", context);
                CloudwatchHelper.EnqueueMetricRequest("userauthdatacontext_recordmasqueradeattempt_error", 1, context, StandardUnit.Count);
            }
        }

        private void RecordMasqueradeSuccess(HttpContext context, string masqueradeID, string masqueradeType, UserAuthData user, UserAuthData masqueradeData)
        {
            // Write to the audit log for the path
            try
            {
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText =
                        $@"
                            insert into {Constants.DatabaseSchema}microservice_twitch_atlas_activity_log
                            (
                                message_timestamp, message, controller, action, user, 
                                tracking_id, tracking_tag, tracking_date, hash
                            )
                            select @message_timestamp, @message, @controller, @action, @user, 
                            @tracking_id, @tracking_tag, @tracking_date, @hash
                        ;";
                        command.Parameters.AddWithValue("@message_timestamp", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@message", $"Masquerade ID: {masqueradeID}, Source: {user.UserID}, Target: {masqueradeData.UserID}, Path: {context.Request.Path.Value.Truncate(255)}".Truncate(8000));
                        command.Parameters.AddWithValue("@controller", "Masquerade");
                        command.Parameters.AddWithValue("@action", "Masquerade");
                        command.Parameters.AddWithValue("@user", user.UserID);
                        command.Parameters.AddWithValue("@tracking_id", masqueradeID);
                        command.Parameters.AddWithValue("@tracking_tag", Guid.NewGuid().ToString("d"));
                        command.Parameters.AddWithValue("@tracking_date", DateTime.UtcNow);
                        command.Parameters.AddWithValue("@hash", HashHelper.SimpleRandomLetterSequence.RandomHash());
                        command.ExecuteNonQueryWithMeasurements("masquerade_activity_log");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex, "userauthdatacontext_recordmasqueradesuccess_error", context);
                CloudwatchHelper.EnqueueMetricRequest("userauthdatacontext_recordmasqueradesuccess_error", 1, context, StandardUnit.Count);
            }
        }

        private UserAuthDataReader CreateUserAuthDataReader(string authHeader, HttpContext context)
        {
            try
            {
                UserAuthDataReader manager = null;

                if (!string.IsNullOrWhiteSpace(authHeader))
                {
                    Match authorizationMatch = BearerRegex.Match(authHeader);
                    Match apiKeyMatch = APIKeyRegex.Match(authHeader);

                    if (authorizationMatch.Success)
                    {
                        manager = new LDAPUserAuthDataReader(authorizationMatch.Groups[1].Value);
                    }
                    else if (apiKeyMatch.Success)
                    {
                        manager = new ApiUserAuthDataReader(apiKeyMatch.Groups[1].Value);
                    }
                }
                return manager;
            }
            catch(Exception ex)
            {
                Log.Error(ex, "userauthdatacontext_createuserauthdr_error", context);
                CloudwatchHelper.EnqueueMetricRequest("userauthdatacontext_createuserauthdr_error", 1, context, StandardUnit.Count);

                throw;
            }
        }
    }
}
