﻿using Amazon;
using Amazon.KeyManagementService;
using Amazon.KeyManagementService.Model;
using Amazon.S3;
using Amazon.S3.Model;
using Resonance.Core.Helpers.LoggingHelpers;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Text;

namespace Resonance.Core.Helpers.AwsHelpers
{
    public static class AwsEncryption
    {
        private static AmazonKeyManagementServiceClient encryptionClientEast = new AmazonKeyManagementServiceClient(RegionEndpoint.USEast1);
        private static AmazonKeyManagementServiceClient encryptionClientWest = new AmazonKeyManagementServiceClient(RegionEndpoint.USWest2);
        private static AmazonS3Client s3Client = new AmazonS3Client(RegionEndpoint.USWest2);

        public static byte[] DecryptKey(byte[] encryptedKey, string keyID)
        {
            var decryptedData = encryptionClientWest.DecryptAsync(new DecryptRequest
            {
                CiphertextBlob = new MemoryStream(encryptedKey)
            }).Result;
            return decryptedData.Plaintext.ToArray().Take((int)decryptedData.ContentLength).ToArray();
        }

        public static string GenerateKey(string kmsArn)
        {
            AesCryptoServiceProvider crypto = new AesCryptoServiceProvider();
            crypto.KeySize = 256;
            crypto.BlockSize = 128;
            crypto.GenerateKey();
            var key = crypto.Key;

            string kmsKeyID = kmsArn.Substring(kmsArn.LastIndexOf("/") + 1);

            var kmsResponse = EncryptKey(key, kmsKeyID);
            byte[] encryptedKey = kmsResponse.CiphertextBlob.ToArray();
            string base64EncodedKey = Convert.ToBase64String(encryptedKey);
            return base64EncodedKey;
        }

        public static GenerateDataKeyResponse EncryptKey(byte[] decryptedKey, string keyID)
        {
            var dataKey = encryptionClientWest.GenerateDataKeyAsync(new GenerateDataKeyRequest
            {
                KeyId = keyID,
                KeySpec = DataKeySpec.AES_256
            }).Result;
            return dataKey;
        }

        public static string DecryptData(string encryptedData, byte[] decryptedKey)
        {
            byte[] encryptedDataBytes = Convert.FromBase64String(encryptedData);

            using (MemoryStream encryptedStream = new MemoryStream(encryptedDataBytes))
            {
                using (MemoryStream decryptedStream = new MemoryStream())
                {
                    Decrypt(encryptedStream, decryptedStream, decryptedKey);
                    return Encoding.UTF8.GetString(decryptedStream.ToArray());
                }
            }

        }

        public static string EncryptString(string input, byte[] key)
        {
            using (MemoryStream inputStream = new MemoryStream(Encoding.UTF8.GetBytes(input)))
            {
                using (MemoryStream outputStream = new MemoryStream())
                {
                    Encrypt(inputStream, outputStream, key);
                    byte[] encryptedBytes = outputStream.ToArray();
                    return Convert.ToBase64String(encryptedBytes);
                }
            }
        }

        public static void Encrypt(Stream input, Stream output, byte[] key)
        {
            using (var algorithm = Aes.Create())
            {
                algorithm.Key = key;
                output.Write(algorithm.IV, 0, algorithm.IV.Length);
                using (var cryptoStream = new CryptoStream(output,
                    algorithm.CreateEncryptor(), CryptoStreamMode.Write))
                {
                    input.CopyTo(cryptoStream);
                    cryptoStream.FlushFinalBlock();
                }
            }
        }

        static void Decrypt(Stream input, Stream output, byte[] key)
        {
            using (var algorithm = Aes.Create())
            {
                algorithm.Key = key;
                var iv = algorithm.IV;
                input.Read(iv, 0, iv.Length);
                algorithm.IV = iv;
                using (var cryptoStream = new CryptoStream(input,
                    algorithm.CreateDecryptor(), CryptoStreamMode.Read))
                {
                    cryptoStream.CopyTo(output);
                }
            }
        }


        public static void Encrypt(string keyID, string sourceFile, string bucket, string s3Path)
        {
            var testFile = new FileInfo(sourceFile);
            var writeFile = $@"{sourceFile}.encrypted";
            using (var memorystream = new MemoryStream())
            {
                using (var filestream = testFile.OpenRead())
                {
                    filestream.CopyTo(memorystream);
                    var encryptAsync = encryptionClientEast.EncryptAsync(new EncryptRequest
                    {
                        KeyId = keyID,//"arn:aws:kms:us-east-1:029773783190:key/0611af9f-78f3-432d-ae85-af45507ce63a",
                        Plaintext = memorystream
                    });
                    encryptAsync.Wait();
                    var response = encryptAsync.Result;
                    using (var ciphertextBlob = response.CiphertextBlob)
                    {
                        string keyId = response.KeyId;
                        File.WriteAllBytes(writeFile, ReadFully(ciphertextBlob));
                    }
                }
            }
            UploadToS3(writeFile, bucket, s3Path);
        }

        public static string Decrypt(string bucket, string path)
        {
            try
            {
                var data = DownloadFromS3(bucket, path);
                using (var memorystream = new MemoryStream(data))
                {
                    var decryptAsync = encryptionClientEast.DecryptAsync(new DecryptRequest
                    {
                        CiphertextBlob = memorystream
                    });
                    decryptAsync.Wait();
                    var response = decryptAsync.Result;
                    string keyId = response.KeyId; // The Amazon Resource Name (ARN) of the CMK that was used to decrypt the data.
                    using (var reader = new StreamReader(response.Plaintext))
                    {
                        var text = reader.ReadToEnd();
                        return text;
                    }
                }
            }
            catch (Exception)
            {
                return null;
            }
        }

        private static byte[] ReadFully(Stream input)
        {
            byte[] buffer = new byte[16 * 1024];
            using (MemoryStream ms = new MemoryStream())
            {
                int read;
                while ((read = input.Read(buffer, 0, buffer.Length)) > 0)
                {
                    ms.Write(buffer, 0, read);
                }
                return ms.ToArray();
            }
        }

        public static void UploadToS3(string inputFile, string bucket, string path)
        {
            FileInfo finfo = new FileInfo(inputFile);

            if (finfo.Length > 4242880)
            {
                List<UploadPartResponse> uploadResponses = new List<UploadPartResponse>();
                List<PartETag> partETags = new List<PartETag>();

                InitiateMultipartUploadRequest initiateRequest = new InitiateMultipartUploadRequest
                {
                    BucketName = bucket,
                    Key = path
                };

                var s3UploadAsync = s3Client.InitiateMultipartUploadAsync(initiateRequest);
                s3UploadAsync.Wait();
                InitiateMultipartUploadResponse initResponse = s3UploadAsync.Result;

                long contentLength = finfo.Length;
                long partSize = 52428800; // 50 MB

                try
                {
                    long filePosition = 0;
                    for (int i = 1; filePosition < contentLength; i++)
                    {

                        UploadPartRequest uploadRequest = new UploadPartRequest
                        {
                            BucketName = bucket,
                            Key = path,
                            UploadId = initResponse.UploadId,
                            PartNumber = i,
                            PartSize = partSize,
                            FilePosition = filePosition,
                            FilePath = inputFile
                        };

                        var uploadAsync = s3Client.UploadPartAsync(uploadRequest);
                        uploadAsync.Wait();
                        var partResponse = uploadAsync.Result;
                        uploadResponses.Add(partResponse);

                        PartETag petag = new PartETag(partResponse.PartNumber, partResponse.ETag);
                        partETags.Add(petag);

                        filePosition += partSize;
                    }

                    CompleteMultipartUploadRequest completeRequest = new CompleteMultipartUploadRequest
                    {
                        BucketName = bucket,
                        Key = path,
                        UploadId = initResponse.UploadId,
                        PartETags = partETags
                    };

                    var multipartUploadComplete = s3Client.CompleteMultipartUploadAsync(completeRequest);
                    multipartUploadComplete.Wait();
                    CompleteMultipartUploadResponse completeUploadResponse = multipartUploadComplete.Result;
                }
                catch (Exception)
                {
                    AbortMultipartUploadRequest abortMPURequest = new AbortMultipartUploadRequest
                    {
                        BucketName = bucket,
                        Key = path,
                        UploadId = initResponse.UploadId
                    };
                    var abort = s3Client.AbortMultipartUploadAsync(abortMPURequest);
                    abort.Wait();
                }
            }
            else
            {
                PutObjectRequest request = new PutObjectRequest()
                {
                    BucketName = bucket,
                    FilePath = inputFile,
                    Key = path
                };
                var putAsync = s3Client.PutObjectAsync(request);
                putAsync.Wait();
                var response = putAsync.Result;
            }
        }

        private static byte[] DownloadFromS3(string bucket, string path)
        {
            try
            {
                var request = new GetObjectRequest()
                {
                    BucketName = bucket,
                    Key = path,
                };

                var getObject = s3Client.GetObjectAsync(request);
                getObject.Wait();
                using (var response = getObject.Result)
                {
                    using (var stream = response.ResponseStream)
                    {
                        var data = ReadFully(stream);
                        return data;
                    }
                }
            }
            catch (Exception)
            {
                return null;
            }
        }
    }
}
