﻿using Amazon;
using Amazon.CloudWatch;
using Amazon.CloudWatch.Model;
using Microsoft.AspNetCore.Http;
using Newtonsoft.Json;
using Resonance.Core.Extensions;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.AuthHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.AuthModels;
using Resonance.Core.Models.MetaDataModels;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;

namespace Resonance.Core.Helpers.AwsHelpers
{
    public static class CloudwatchHelper
    {
        private static string logIdentifier { get; set; } = null;
        private static string projectNamespace { get; set; } = null;
        private static string name { get; set; } = null;
        private static BlockingCollection<PutMetricDataRequest> metricRequests = new BlockingCollection<PutMetricDataRequest>();
        /// <summary>
        /// Aws max batch size
        /// </summary>
        private const int batchsize = 5;

        public static void Initialize(string projectName)
        {
            Log.Info($@"Starting Cloudwatch Metrics Helper");
            projectNamespace = $@"Resonance.{projectName}.{Constants.AppConfig.Application.Environment}";
            name = projectName;
            var worker = new BackgroundWorker();
            worker.DoWork += PublishMetric;
            worker.RunWorkerAsync();

            Thread heartbeat = new Thread(Heartbeat);
            heartbeat.Start();
        }

        private static void Heartbeat()
        {
            do
            {
                try
                {
                    EnqueueMetricRequest("heartbeat", 1, null, StandardUnit.Count);
                }
                catch (Exception ex)
                {
                    Log.Error(ex);
                }
                finally
                {
                    Task.Delay(60000).Wait();
                }
            } while (true);
        }

        public static string GetNamespace()
        {
            return projectNamespace;
        }

        public static bool EnqueueMetricRequest(string key, double value, HttpContext context = null, StandardUnit unit = null)
        {
            try
            {
                var now = DateTime.UtcNow;
                AuthTokenData token = null;
                var logMetric = new CloudwatchMetricModel();
                var trackingID = Guid.NewGuid().ToString("d");
                if (unit == null)
                {
                    unit = StandardUnit.None;
                }
                string userAgent = string.Empty;
                string userIP = string.Empty;

                var dimensions = new Dictionary<string, string>
                {
                    { "host", Environment.MachineName }
                };

                // If we have an http context we can add some extra default metrics
                if (context != null)
                {                    
                    if (context.Items.ContainsKey(UserAuthDataContext.AuthTokenDataKey))
                    {
                        token = ((AuthTokenData)context.Items[UserAuthDataContext.AuthTokenDataKey]) ?? null;
                    }
                    try
                    {
                        userIP = HeaderHelper.GetRequestIP(context, true);
                    }
                    catch (Exception ex)
                    {
                        userIP = "Unknown";
                        Log.Verbose($@"Unable to record user ip. {ex}");
                    }
                    try
                    {
                        userAgent = HeaderHelper.GetUserAgent(context);
                    }
                    catch (Exception ex)
                    {
                        userAgent = "Unknown";
                        Log.Verbose($@"Unable to record user agent. {ex}");
                    }
                }

                var request = new PutMetricDataRequest()
                {
                    Namespace = projectNamespace,
                    MetricData = new List<MetricDatum>()
                    {
                        new MetricDatum()
                        {
                            MetricName = key,
                            Value = value,
                            TimestampUtc = now,
                            Unit = unit,
                            Dimensions = dimensions?.Select(x => new Dimension() { Name = x.Key, Value = x.Value }).ToList()
                        }
                    }
                };
                metricRequests.TryAdd(request);
                
                Log.Metric($@"{JsonConvert.SerializeObject(new CloudwatchMetricModel()
                {
                    Namespace = projectNamespace,
                    Key = key,
                    Value = value,
                    HttpPath = context?.Request?.Path,
                    HttpMethod = context?.Request?.Method,
                    HttpStatus = context?.Response?.StatusCode == null ? null : context.Response.StatusCode.ToString(),
                    UserID = token?.User,
                    Datestamp = now.ToString("yyyy-MM-dd"),
                    Timestamp = now,
                    AppVersion = Constants.ApplicationVersion,
                    Host = Environment.MachineName,
                    TrackingID = trackingID,
                    UserIP = userIP,
                    UserAgent = userAgent
                }, Formatting.Indented)}", context);

                return true;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                return false;
            }
        }

        private static void PublishMetric(object sender, DoWorkEventArgs e)
        {
            if (string.IsNullOrWhiteSpace(projectNamespace))
            {
                Log.Error($@"CloudwatchHelper has not been initialized.");
                return;
            }
            var maxPullWatch = new Stopwatch();
            var maxPullLength = 1000;
            var enqueuedItems = new List<MetricDatum>();
            try
            {
                do
                {
                    try
                    {
                        // Only loop for a maximum amount of time - this prevents cpu pegging under heavy load. Normally this shouldn't even be a factor
                        do
                        {
                            // Try to take for a maximum of 60 seconds before timing out and trying again - This prevents locking under exceptionally light load
                            metricRequests.TryTake(out PutMetricDataRequest data, 60000);
                            if (data != null && !string.IsNullOrWhiteSpace(data.Namespace) && data.MetricData != null && data.MetricData.Count > 0)
                            {
                                if (!maxPullWatch.IsRunning)
                                {
                                    maxPullWatch.Start();
                                }
                                // Add data to the local list for batch metric writing
                                enqueuedItems.AddRange(data.MetricData);
                            }
                        } while (metricRequests.Count > 0 && maxPullWatch.ElapsedMilliseconds <= maxPullLength);
                        maxPullWatch.Stop();
                        maxPullWatch.Reset();

                        try
                        {
                            if (enqueuedItems.Count > 0)
                            {
                                foreach (var batch in enqueuedItems.Partition(batchsize).ToArray())
                                {
                                    try
                                    {
                                        if(batch == null || batch.Count() == 0)
                                        {
                                            continue;
                                        }
                                        using (AmazonCloudWatchClient cloudwatch = new AmazonCloudWatchClient(RegionEndpoint.USWest2))
                                        {
                                            PutMetricDataRequest mdr = new PutMetricDataRequest
                                            {
                                                Namespace = projectNamespace,
                                                MetricData = batch.ToList()
                                            };
                                            Log.Verbose($@"Sending Cloudwatch Metrics");
                                            if(mdr.MetricData.Any(x => x.Dimensions == null || !x.Dimensions.Any()))
                                            {
                                                continue;
                                            }
                                            var respTask = cloudwatch.PutMetricDataAsync(mdr);
                                            if (respTask.Wait(60000))
                                            {
                                                PutMetricDataResponse resp = respTask.Result;
                                                if (resp.HttpStatusCode != System.Net.HttpStatusCode.OK)
                                                {
                                                    Log.Error($@"Metrics Failure. Status code: {resp.HttpStatusCode}({(int)resp.HttpStatusCode})");
                                                }
                                            }
                                            else
                                            {
                                                Log.Error($@"Cloudwatch response failure after 60s");
                                            }
                                        }
                                    }
                                    catch (Exception ex)
                                    {
                                        Log.Error($@"An error occurred while metrics were being processed. Data Lost? {enqueuedItems.Count > 0} :: {ex}, Batch.");
                                    }
                                }
                            }
                        }
                        catch (Exception ex)
                        {
                            Log.Error($@"An error occurred while metrics were being processed. Data Lost? {enqueuedItems.Count > 0} :: {ex}");
                        }

                        enqueuedItems.Clear();
                    }
                    catch (OperationCanceledException) { }
                    catch (TimeoutException) { }
                    catch (Exception ex)
                    {
                        Log.Error(ex);
                    }
                } while (true);
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            finally
            {
                Log.Fatal($@"Cloudwatch Metrics unexpectedly shut down.");
            }
        }
    }
}