﻿using Amazon.CloudWatch;
using Microsoft.AspNetCore.Http;
using Newtonsoft.Json;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.DatabaseHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Models.FilterModels;
using Resonance.Core.Services.CustomConfigService;
using Resonance.Core.Services.LdapService;
using System;
using System.Collections.Generic;
using System.Data.Common;
using System.IO;
using System.Linq;
using System.Text;

namespace Resonance.Core.Helpers.ConfigurationHelpers
{
    public class SelectOption
    {
        [JsonProperty(PropertyName = "value")]
        public dynamic Value { get; set; }

        [JsonProperty(PropertyName = "display")]
        public string Display { get; set; }

        public SelectOption(dynamic value, string display)
        {
            Value = value;
            Display = display;
        }

        public SelectOption(string value)
        {
            Value = value;
            Display = value;
        }
    }

    public class FilterOptionsHelper : ICustomConfigCreator
    {        
        private const string Bucket = "resonance-configuration";

        public void CreateFilterListFile(string outputFilepath)
        {
            //category name
            string channelCategory = "Channel";
            string engagementCategory = "Engagement";
            string revenueCategory = "Revenue";
            string viewershipCategory = "Viewership";

            //type name
            string freeTextTypeName = "Free-Text Entry";
            string multiSelectDropDownName = "Multi-Select Dropdown";
            string rangeName = "Range (Min/Max)";
            string singleSelectDropDownName = "Single-Select Dropdown";

            //default
            string freeTextDefaultText = "Ex: LoremIpsum";
            string freeTextDefaultLocalizationKey = "FilterDefaultText"; 
            string multiSelectDefaultText = "Select all that apply";
            string multiSelectDefaultLocalizationKey = "FilterDefaultMultiSelect";
            string[] rangeDefaultText = new string[] { "Minimum", "Maximum" };
            string[] rangeDefaultLocalizationKey = new string[] { "FilterDefaultMinimum", "FilterDefaultMaximium" };
            string singleSelectDefaultText = "Select one";
            string singleSelectDefaultLocalizationKey = "FilterDefaultSingleSelect";

            #region list
            var values = new List<Object>
            { //Note: KeyName corresponds to the name of the options files on S3
                //Channel
                new { name = "login", displayName="Channel Name Contains", localizationKey="ChannelNameContainsFilter", category =channelCategory, type=freeTextTypeName, defaultText=freeTextDefaultText, defaultLocalizationKey=freeTextDefaultLocalizationKey, keyName = "" },
                new { name="accountmanager", displayName="Account Manager", localizationKey="AccountManagerFilter", category=channelCategory, type=multiSelectDropDownName, defaultText=multiSelectDefaultText, defaultLocalizationKey=freeTextDefaultLocalizationKey, keyName = "account-manager" },
                new { name="usertype", displayName="Partner Status", localizationKey="PartnerStatusFilter", category=channelCategory, type=multiSelectDropDownName, defaultText=multiSelectDefaultText, defaultLocalizationKey=multiSelectDefaultLocalizationKey, keyName="partner-status"},
                new { name="primarybroadcastlanguage", displayName="Language", localizationKey="LanguageFilter", category=channelCategory, type=multiSelectDropDownName, defaultText=multiSelectDefaultText, defaultLocalizationKey=multiSelectDefaultLocalizationKey, keyName="languages" },
                new { name="topgamebroadcast", displayName="Top Game Broadcast", localizationKey="TopGameBroadcastFilter", category=channelCategory, type=multiSelectDropDownName, defaultText=multiSelectDefaultText, defaultLocalizationKey=multiSelectDefaultLocalizationKey, keyName="games" },
                new { name="gamecount", displayName="Games Played", localizationKey="GamesCount", category=channelCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName="" },
                new { name="primarybroadcastcountry", displayName="Country", localizationKey="CountryFilter", category=channelCategory, type=multiSelectDropDownName, defaultText=multiSelectDefaultText, defaultLocalizationKey=multiSelectDefaultLocalizationKey, keyName="countries" },
                new { name="ccutier", displayName="Tier", localizationKey="TierFilter", category=channelCategory, type=multiSelectDropDownName, defaultText=multiSelectDefaultText, defaultLocalizationKey=multiSelectDefaultLocalizationKey, keyName="tiers" },
                new { name="ispremium", displayName="Premium", localizationKey="Premium", category=channelCategory, type=singleSelectDropDownName, defaultText=singleSelectDefaultText, defaultLocalizationKey=singleSelectDefaultLocalizationKey, keyName="ispremium" },
                new { name="timsuseractionrequired", displayName="Tims User Action Required", localizationKey="TimsUserActionRequired", category=channelCategory, type=singleSelectDropDownName, defaultText=singleSelectDefaultText, defaultLocalizationKey=singleSelectDefaultLocalizationKey, keyName="timsuseractionrequired" },
                new { name="valuescoreoverallweighted", displayName="Value Score", localizationKey="ValueScoreFilter", category=channelCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName="" },

                //Engagement
                new { name="channelconcurrentsminuteswatchedtotal", displayName="Minutes Watched", localizationKey="MinutesWatchFilter", category=engagementCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="channelconcurrentsminutesbroadcasttotal", displayName="Minutes Broadcast", localizationKey="MinutesBroadcastFilter", category=engagementCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="subssoldprimetotal", displayName="Prime Subs Sold", localizationKey="PrimeSubsSoldFilter", category=engagementCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="subssoldtotal", displayName="Subs Sold", localizationKey="SubsSoldFilter", category=engagementCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="chatmessagestotal", displayName="Chat Messages", localizationKey="ChatMessagesFilter", category=engagementCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="uniquedaysbroadcast", displayName="Days Broadcast", localizationKey="DaysBroadcastFilter", category=engagementCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="averageccu", displayName="Average CCU", localizationKey="AverageCCUFilter", category=engagementCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="maxccu", displayName="Max CCU", localizationKey="MaxCCUFilter", category=engagementCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },

                //Revenue
                new { name="creatorrevenuetotal", displayName="Revenue", localizationKey="RevenueFilter", category=revenueCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="creatorprimesubrevenuetotal", displayName="Prime Revenue", localizationKey="PrimeRevenueFilter", category=revenueCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="creatorbitsrevenuetotal", displayName="Bits Revenue", localizationKey="BitsRevenueFilter", category=revenueCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="creatoradrevenuetotal", displayName="Ad Revenue", localizationKey="AdRevenueFilter", category=revenueCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="creatorbountyboardrevenuetotal", displayName="Bounty Board Revenue", localizationKey="BountyBoardRevenueFilter", category=revenueCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="creatorsubrevenuetotal", displayName="Subs Revenue", localizationKey="SubsRevenueFilter", category=revenueCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },

                //Viewership
                new { name="mostviewedplatform", displayName="Most Viewed Platform", localizationKey="MostViewedPlatform", category=viewershipCategory, type=multiSelectDropDownName, defaultText=multiSelectDefaultText, defaultLocalizationKey=multiSelectDefaultLocalizationKey, keyName="platforms" },
                new { name="platformpercentwatchedweb", displayName="MW% from Web", localizationKey="PlatformPercentWatchedWeb", category=viewershipCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="platformpercentwatchedconsole", displayName="MW% from Console", localizationKey="PlatformPercentWatchedConsole", category=viewershipCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="platformpercentwatcheddesktop", displayName="MW% from Desktop", localizationKey="PlatformPercentWatchedDesktop", category=viewershipCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="platformpercentwatchedmobile", displayName="MW% from Mobile", localizationKey="PlatformPercentWatchedMobile", category=viewershipCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="platformpercentwatchedchromecast", displayName="MW% from Chromecast", localizationKey="PlatformPercentWatchedChromecast", category=viewershipCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
                new { name="platformpercentwatchedother", displayName="MW% from Other", localizationKey="PlatformPercentWatchedOther", category=viewershipCategory, type=rangeName, defaultText=rangeDefaultText, defaultLocalizationKey=rangeDefaultLocalizationKey, keyName = "" },
            };
            #endregion            

            string serializedJson = JsonConvert.SerializeObject(values);
            var directory = Path.GetDirectoryName(outputFilepath);
            if (!Directory.Exists(directory))
            {
                Directory.CreateDirectory(directory);
            }
            File.WriteAllText(outputFilepath, serializedJson);
        }
        
        private IEnumerable<SelectOption> GetRecords(string sql, bool isTahoe = false, Dictionary<string, string> map = null)
        {            
            var fields = new List<SelectOption>();

            using (DbConnection conn = isTahoe ? (DbConnection)DBManagerRedshift.TahoeConnection(true) : (DbConnection)DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandTimeout = 600;
                    command.CommandText = sql;

                    DataReaderWithMeasurements wrappedReader;
                    if(isTahoe)
                    {
                        wrappedReader = new DataReaderWithMeasurements((Npgsql.NpgsqlCommand)command, null, "filter_options_get_records");
                    }
                    else
                    {
                        wrappedReader = new DataReaderWithMeasurements((MySql.Data.MySqlClient.MySqlCommand)command, null, "filter_options_get_records");
                    }

                    using (wrappedReader)
                    {
                        DbDataReader reader = isTahoe ? (DbDataReader)wrappedReader.NpgReader : wrappedReader.MysqlReader;
                        if (reader.HasRows)
                        {
                            while (reader.Read())
                            {                                
                                string key = reader.GetString(0);
                                string value = null;

                                if(reader.FieldCount > 1)
                                {
                                    value = reader.GetString(1);
                                }
                                else
                                {
                                    if(map != null)
                                    {
                                        if(!map.TryGetValue(key, out value))
                                        {
                                            value = key;
                                        }
                                    }
                                }
                                

                                fields.Add(new SelectOption(key, value));
                            }
                        }
                    }
                }
            }

            return fields;
        }

        //tuple: data, istahoe, issql
        //in SQL first field is key, second is value
        private Dictionary<string, Tuple<string, bool, bool>> OptionsData = new Dictionary<string, Tuple<string, bool, bool>>
        {
            {"games", Tuple.Create($"SELECT name FROM {Constants.DatabaseSchema}microservice_twitch_games where name is not null ORDER BY name", false, true) },
            {"countries", Tuple.Create("SELECT code, country from metadata.countries_2018 order by country", true, true) },
            {"regions", Tuple.Create("SELECT DISTINCT region FROM metadata.country_to_region", true, true) },
            {"tiers", Tuple.Create(JsonConvert.SerializeObject(new List<SelectOption> { new SelectOption("T0: 0-2 CCU", "T0: 0-2 CCU"), new SelectOption("T1: 3-10 CCU", "T1: 3-10 CCU"), new SelectOption("T2: 11-100 CCU", "T2: 11-100 CCU"), new SelectOption("T3: 101-4000 CCU", "T3: 101-4000 CCU"), new SelectOption("T4: 4001+ CCU", "T4: 4001+ CCU") }), false, false) },
            //{"languages", Tuple.Create($"SELECT distinct primarybroadcastlanguage from {Constants.DatabaseSchema}microservice_twitch_user_listing where primarybroadcastlanguage != '' ORDER BY primarybroadcastlanguage", false, true) },
            {"partner-status", Tuple.Create(JsonConvert.SerializeObject(new List<SelectOption> { new SelectOption("Partner"), new SelectOption("Affiliate"), new SelectOption("None") }), false, false) },
            //{"account-manager", Tuple.Create(JsonConvert.SerializeObject(GenerateAccountManagers(null)), false, false) },
            {"platforms", Tuple.Create(JsonConvert.SerializeObject(new List<SelectOption> { new SelectOption("web"), new SelectOption("console"), new SelectOption("desktop"), new SelectOption("other"), new SelectOption("mobile"), new SelectOption("chromecast") }), false, false) },
            {"ispremium", Tuple.Create(JsonConvert.SerializeObject(new List<SelectOption> { new SelectOption(true, "Is Premium"), new SelectOption(false, "Not Premium") }), false, false) },
            {"timsuseractionrequired", Tuple.Create(JsonConvert.SerializeObject(new List<SelectOption> { new SelectOption(true, "Action Required"), new SelectOption(false, "Not Required") }), false, false) }
        };

        private static List<SelectOption> GenerateAccountManagers(HttpContext context)
        {
            try
            {
                LdapService ldapService = new LdapService();
                List<SelectOption> options = new List<SelectOption>();
                using (var conn = DBManagerMysql.GetConnection(true))
                {
                    using (var command = conn.CreateCommand())
                    {
                        command.CommandText = $"select distinct(AccountManager) from {Constants.DatabaseSchema}microservice_twitch_user_listing_past_360_days where AccountManager is not null and AccountManager != ''";
                        using (var reader = new DataReaderWithMeasurements(command, null, "get_distinct_account_managers").MysqlReader)
                        {
                            while (reader.Read())
                            {
                                string accountManager = (string)reader["AccountManager"];
                                string accountManagerName = ldapService.GetUserName(accountManager, context);
                                string label = (accountManagerName == "" || accountManager.Contains(")")) ? accountManager : $"{accountManagerName} ({accountManager})";
                                options.Add(new SelectOption(accountManager, label));
                            }
                        }
                    }
                }
                return options;
            }
            catch(Exception ex)
            {
                Log.Error(ex, "filteroptionshelper_generateams_error", context);
                CloudwatchHelper.EnqueueMetricRequest("filteroptionshelper_generateams_error", 1, context, StandardUnit.Count);

                throw;
            }
        }

        private Dictionary<string, string> LanguageMap = new Dictionary<string, string>()
        {
            { "ar", "Arabic" },
            //{ "asl", "American Sign Language" },
            { "bg", "Bulgarian" },
            { "ca", "Catalan; Valencian" },
            { "cs", "Czech" },
            { "da", "Danish" },
            { "de", "German (Deutsch)" },
            { "el", "Greek (Ελληνικά)" },
            { "en", "English" },
            { "es", "Español (Spanish)" },
            { "fi", "Finnish" },
            { "fr", "French (Français)" },
            { "hu", "Hungarian" },
            { "id", "Indonesian (Bahasa Indonesia )" },
            { "it", "Italiano (Italian)" },
            { "ja", "Japanese (日本語)" },
            { "ko", "Korean (한국어)" },
            { "nl", "Dutch; Flemish" },
            { "no", "Norwegian" },
            { "other", "Other" },
            { "pl", "Polish (Polski)" },
            { "pt", "Português" },
            { "ro", "Romanian" },
            { "ru", "Russian (Pусский)" },
            { "sk", "Slovak" },
            { "sv", "Swedish (Svenska)" },
            { "th", "Thai" },
            { "tl", "Tagalog" },
            { "tr", "Turkish" },
            { "uk", "Ukrainian" },
            { "vi", "Vietnamese" },
            { "zh", "Chinese (Simplified)" },            
            { "zh-hk", "Chinese (Hong Kong)" },
            { "tw", "Chinese (Traditional)" },
            { "mx", "Latin American Spanish" },
            { "gb", "British English" }
        };

        public bool CreateOptionsFile(string name, string outputFilePath)
        {
            bool result = false;
            if (OptionsData.ContainsKey(name))
            {
                result = true;
                var item = OptionsData[name];
                string output;

                if (item.Item3)
                {
                    IEnumerable<SelectOption> records = null;
                    if (name.ToLower() == "languages")
                    {   //this method selects 1 field from the database for the value then gets display from the map
                        records = GetRecords(item.Item1, item.Item2, LanguageMap);
                    }
                    else
                    {   //this method selects 2 fields from the database for value/display
                        records = GetRecords(item.Item1, item.Item2);
                    }
                    output = JsonConvert.SerializeObject(records, Formatting.None, new JsonSerializerSettings
                    {
                        NullValueHandling = NullValueHandling.Ignore
                    });
                }
                else
                {
                    output = item.Item1;
                }
                var directory = Path.GetDirectoryName(outputFilePath);
                if (!Directory.Exists(directory))
                {
                    Directory.CreateDirectory(directory);
                }
                File.WriteAllText(outputFilePath, output);
            }

            return result;
        }

        private void WriteLanguageFile(string outputPath)
        {
            var fields = new List<SelectOption>();

            foreach (var lang in LanguageMap.OrderBy(x => x.Value))
            {
                fields.Add(new SelectOption(lang.Key, lang.Value));
            }

            string serialized = JsonConvert.SerializeObject(fields, Formatting.None);
            var directory = Path.GetDirectoryName(outputPath);
            if (!Directory.Exists(directory))
            {
                Directory.CreateDirectory(directory);
            }
            File.WriteAllText(outputPath, serialized);
        }

        public void CreateFiles(ICustomConfigService service)
        {
            string system = "filter";
            string baseDir = Path.GetTempPath();      
            string path;
            foreach (var item in OptionsData)
            {
                string key = item.Key;
                path = $"{baseDir}{key}.txt";
                CreateOptionsFile(key, path);

                //uplodate to S3 here
                service.UploadConfigFile("filter", key, path);
                try { File.Delete(path); }
                catch { }
            }

            path = Path.Combine(baseDir, "languages.txt");
            WriteLanguageFile(path);
            service.UploadConfigFile(system, "languages", path);

            path = Path.Combine(Path.GetTempPath(), "list.txt");
            CreateFilterListFile(path);
            service.UploadConfigFile(system, "list", path);
            try { File.Delete(path); }
            catch { }
        }

        public static void GenerateCountryList(string outputFilePath)
        {
            FilterOptionsHelper helper = new FilterOptionsHelper();
            var countries = helper.GetRecords("SELECT code, country from metadata.countries_2018 order by country", true, null);
            StringBuilder sb = new StringBuilder();

            sb.Append($@"{{ value: '', label: 'Any' }},");
            sb.AppendLine();

            foreach (var item in countries)
            {
                sb.Append($@"{{ value: '{item.Value}', label: '{item.Display}' }},");
                sb.AppendLine();
            }

            File.WriteAllText(outputFilePath, sb.ToString());
        }

        public static Dictionary<string, List<dynamic>> GenerateRegionToCountryMapping()
        {
            FilterOptionsHelper helper = new FilterOptionsHelper();

            var countryNameRecords = helper.GetRecords("SELECT code, country from metadata.countries_2018 order by country;", true);
            Dictionary<string, string> countryNameMapping = new Dictionary<string, string>();
            foreach(var item in countryNameRecords)
            {
                countryNameMapping.Add(item.Value, item.Display);
            }

            var records = helper.GetRecords("select country,metaregion from metadata.country_to_region order by country;", true, null);
            StringBuilder sb = new StringBuilder();
            Dictionary<string, List<dynamic>> regionMapping = new Dictionary<string, List<dynamic>>();

            foreach (var item in records)
            {
                if (!regionMapping.ContainsKey(item.Display))
                {
                    regionMapping.Add(item.Display, new List<dynamic>());
                }

                string countryName = item.Value;
                if (countryNameMapping.ContainsKey(item.Value))
                {
                    countryName = countryNameMapping[item.Value];
                }

                regionMapping[item.Display].Add(new
                {
                    country_code = item.Value,
                    country_name = countryName
                });
            }

            return regionMapping;
        }
    }
}
