﻿using CsvHelper;
using Newtonsoft.Json;
using Resonance.Core.Services.CustomConfigService;
using Resonance.Core.Services.LocalizationService;
using System;
using System.Collections.Generic;
using System.IO;
using System.Text;

namespace Resonance.Core.Helpers.ConfigurationHelpers
{
    public class LocalizationsHelper: ICustomConfigCreator
    {
        public static void ExportLocalizationCSV(string locale, string filePath)
        {
            // Get full localization dictionary
            var ls = new LocalizationService(locale, Constants.AppConfig.Application.Environment);
            Dictionary<string, string> localizationDataCopy = ls.GetAllStrings();

            // Write dictionary to proper CSV
            using (FileStream fs = File.Open(filePath, FileMode.Create, FileAccess.Write))
            using (TextWriter sw = new StreamWriter(fs, Encoding.UTF8))
            using (CsvWriter cw = new CsvWriter(sw, new CsvHelper.Configuration.Configuration { Delimiter = ",", Quote = '"' }))
            {
                // Write header
                List<string> header = new List<string>()
                {
                    "Key",
                    "Value"
                };
                foreach (var field in header)
                {
                    cw.WriteField(field);
                }
                cw.NextRecord();

                foreach (var pair in localizationDataCopy)
                {
                    cw.WriteField(pair.Key);
                    cw.WriteField(pair.Value);
                    cw.NextRecord();
                }
            }
        }

        public static void ImportLocalizationCSV(string locale, string filePath)
        {
            // Initialize localization service
            var ls = new LocalizationService(locale, Constants.AppConfig.Application.Environment);

            // Read proper CSV, assign strings to service, and then upload results
            using (MemoryStream MS = new MemoryStream(File.ReadAllBytes(filePath)))
            using (TextReader SR = new StreamReader(MS))
            using (CsvReader CR = new CsvReader(SR, new CsvHelper.Configuration.Configuration() { HasHeaderRecord = true, Delimiter = "," }))
            {
                CR.Read();
                CR.ReadHeader();

                while (CR.Read())
                {
                    ls.AssignString(CR.GetField("Key"), CR.GetField("Value"));
                }
            }

            // Upload modified dictionary data
            ls.UploadLocalization();
        }

        public void CreateLocalizationFileUS_EN(string outputPath)
        {            
            Dictionary<string, string> localizations = new Dictionary<string, string>()
            {
                //heading
                { "ChannelListings", "Channel Listings" },
                { "RequestCenter", "Request Center" },
                { "Help", "Help" },
                { "SignOut", "Sign Out" },

                //filters
                { "FiltersTitle", "Filters" },
                { "FiltersTimeframe", "TIME FRAME (DAYS)" },
                { "FiltersSubmitButton", "Submit" },
                { "FiltersClearFiltersButton", "Clear Filters" },
                { "FiltersChannel", "Channel" },

                //channel filter
                {"ChannelFilterTitle", "Channel" },
                {"ChannelNameContainsFilter", "Channel Name Contains" },
                {"AccountManagerFilter", "Account Manager" },
                {"PartnerStatusFilter", "Partner Status" },
                {"ChannelFilterCountry", "Country" },
                {"ChannelFilterTier", "Tier" },

                //engagement filter
                {"EngagementFilterTitle", "Engagement" },
                {"MinutesWatchFilter", "Minutes Watched" },
                {"MinutesBroadcastFilter", "Minutes Broadcast" },
                {"PrimeSubsSoldFilter", "Prime Subs Sold" },
                {"SubsSoldFilter", "Subs Sold" },
                {"ChatMessagesFilter", "Chat Messages" },
                {"DaysBroadcastFilter", "Days Broadcast" },
                {"AverageCCUFilter", "Average CCU" },
                {"MaxCCUFilter", "Max CCU" },

                //revenue filter
                {"RevenueFilterTitle", "Revenue" },
                {"RevenueFilter", "Revenue" },
                {"PrimeRevenueFilter", "Prime Revenue" },
                {"BitsRevenueFilter", "Bits Revenue" },
                {"AdRevenueFilter", "Ad Revenue" },
                {"BountyBoardRevenueFilter", "Bounty Board Revenue" },
                {"SubsRevenueFilter", "Subs Revenue" },

                //filter other
                {"LanguageFilter", "Language" },
                {"TopGameBroadcastFilter", "Top Game Broadcast" },
                {"CountryFilter", "Country" },
                {"TierFilter", "Tier" },

                //filter defaults
                {"FilterDefaultMinimum", "Minimum" },
                {"FilterDefaultMaximium", "Maximium" },
                {"FilterDefaultMultiSelect", "Select All That Apply" },
                {"FilterDefaultText", "Ex. LoremIpsum" },                

                //request targeting at a glance section
                { "RequestTargetingAtAGlance", "Request Targeting At-A Glance" },
                { "RegionSplit", "Region Split" },
                { "TierSplit", "Tier Split" },
                { "TotalChannelsTargeted", "Total Channels Targeted" },
                { "AverageHoursWatched", "Avg Hours Watched" },
                { "AverageHoursBroadcasted", "Avg Hours Broadcasted" },
                { "AverageCCU", "Avg CCU" },
                { "NextSteps", "Next Steps" },
                { "NextStepsMessage", "Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut commodo turpis orci, ac sagittis neque vestibulum sed. Praesent vitae enim id augue placerat feugiat vel dignissim mauris.Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut commodo turpis orci, ac sagittis neque vestibulum sed. Praesent vitae enim id augue placerat feugiat vel dignissim mauris.Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut commodo turpis orci, ac sagittis neque vestibulum sed. Praesent vitae enim id augue placerat feugiat vel dignissim mauris.Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut commodo turpis orci, ac sagittis neque vestibulum sed. Praesent vitae enim id augue placerat feugiat vel dignissim mauris.Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut commodo turpis orci, ac sagittis neque vestibulum sed." },
                { "CreateInternalRequest", "Create Internal Request" },                

                //other
                { "NoResultsFound", "No Results Found" },

                //Edit Request page
                { "EditDetails", "Edit Details" },
                { "RequestProgress", "Request Progress" },                
                { "RequestName", "Request Name" },
                { "TargetDate", "Target Date" },                
                { "DayRemaining", "Days Remaining" },
                { "Request Privacy", "Request Privacy" },
                { "AuthorizedUsers", "Authorized Users" },
                { "TypeUsersName", "Type users's name" },
                { "TypeOfRequest", "Type of Request" },
                { "ExpectedParticipation", "Expected Participation" },
                { "Creators", "Creators" },
                { "TimeCommitment", "Time Commitment" },
                { "SelectAnOption", "Select an option" },
                { "PreferredContactMethod", "Preferred Contact Method" },
                {"Summary", "Summary" },
                {"SummaryPlaceholder", "Please describe your request here" },
                {"OfficialStatement", "Official Statement" },
                {"OfficialStatementPlaceholder", "Please write the official statement that will be used in communications with partners" },
                { "TargetedCreators", "Targeted Creators" },
                { "TargetedCreatorsExplanation", "Your expected participation is: {0} Creators. We recommend targeting approximately {y} Creators to reach that goal." },
                { "AddAdditionalCreators", "Add Additional Creators" },
                { "SubmitRequest", "Submit Request" },

                //requests                                
                { "DueDate", "Due Date" },
                { "Participants", "Participants" },                
                { "Progress", "Progress" },
                { "RequestDetails", "Request Details" },

                //options lists
                {"StatusPartner", "Partner" },
                {"StatusAffiliate", "Affiliate" },
                {"StatusNone", "None" },
                { "Hours", "Hours" },
                { "Days", "Days" },
                { "Weeks", "Weeks" },

                {"IamHereTo", "I'm here to..." },
                {"BuildRequestFromData", "Build a Request From Data" },
                {"BuildRequestFromDataButton", "Build Request from Data" },
                {"ReviewRequest", "Review/Edit Existing Requests" },
                {"RequestListingsButton", "Request Listings" },
                {"BuildRequestManually", "Build a Request Manually" },
                {"BuildRequestManuallyButton", "Building Request Manually" },                
            };


            string serialized = JsonConvert.SerializeObject(localizations, Formatting.None);
            File.WriteAllText(outputPath, serialized);            
        }

        private void WriteListFile(string outputPath)
        {
            var list = new List<Object>
            {
                new { language = "English", cultureCode="en", fullCultureCode = "en-us", supported = true, keyName= "en-us" },
                new { language= "Français (French)", cultureCode= "fr", fullCultureCode = "fr-fr", supported = false },
                new { language= "Deutsch (German)", cultureCode= "de", fullCultureCode = "de-de", supported = false },
                new { language= "Español (Spanish)", cultureCode= "es", fullCultureCode = "es-es", supported = false },
                new { language= "Pусский (Russian)", cultureCode= "ru", fullCultureCode = "ru-ru", supported = false },
                new { language= "汉语 (Simplified Chinese)", cultureCode= "zh", fullCultureCode = "zh-cn", supported = false },
                new { language= "日本語 (Japanese)", cultureCode= "ja", fullCultureCode = "ja-jp", supported = false },
                new { language= "한국어 (Korean)", cultureCode= "ko", fullCultureCode = "ko-kr", supported = false },
                new { language= "Svenska (Swedish)", cultureCode= "sv", fullCultureCode = "sv-se", supported = false },
                new { language= "Bahasa Indonesia (Indonesian)", cultureCode= "id", fullCultureCode = "id-id", supported = false },
                new { language= "Ελληνικά (Greek)", cultureCode= "el", fullCultureCode = "el-gr", supported = false },
                new { language= "Polski (Polish)", cultureCode= "pl", fullCultureCode = "pl-pl", supported = false },
                new { language= "Italiano (Italian)", cultureCode= "it", fullCultureCode = "it-it", supported = false },
                new { language= "繁體中文 (Traditional Chinese)", cultureCode= "tw", fullCultureCode = "zh-tw", supported = false },
                new { language= "Latin American Spanish", cultureCode= "mx", fullCultureCode = "es-ms", supported = false },
                new { language= "British English", cultureCode= "gb", fullCultureCode = "en-gb", supported = false },
                new { language= "Português", cultureCode= "pt", fullCultureCode = "pt-br", supported = false },
            };

            string serialized = JsonConvert.SerializeObject(list, Formatting.None);
            File.WriteAllText(outputPath, serialized);
        }

        public void CreateFiles(ICustomConfigService service)
        {
            string baseDir = Path.GetTempPath();

            string path = Path.Combine(baseDir, "list.txt");
            WriteListFile(path);
            service.UploadConfigFile("localization", "list", path);

            path = Path.Combine(baseDir, "en-us.txt");
            CreateLocalizationFileUS_EN(path);
            service.UploadConfigFile("localization", "en-us", path);
        }
    }
}
