﻿using Npgsql;
using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.IO;
using System.Text;

namespace Resonance.Core.Helpers.DatabaseHelpers
{
    public class BulkCopyBatch
    {
        /// <summary>
        /// Bulk inserts data into a sql server table. 
        /// </summary>
        /// <param name="conn"></param>
        /// <param name="tableName">
        /// - dbo.TableName = Normal database tables for the current connection
        /// - tempdb..#TempTableName = Temporary table for the current connection session. (Table must exist from previous command)
        /// </param>
        /// <param name="payload">Datatable with information to be uploaded</param>
        /// <param name="commandTimeout">Time before bulk copy operation times out</param>
        /// <param name="truncateOnLoad">If true, removes all existing </param>
        public static void BulkInsertBatchData(SqlConnection conn, string tableName, DataTable payload, bool truncateOnLoad = false, int commandTimeout = 4000, int batchsize = 5000, bool stream = false, bool allowTableLock = false)
        {
            try
            {
                if (payload.Rows.Count == 0)
                {
                    return;
                }
                Console.WriteLine($"Beginning Bulk Copy Batch - {payload.Rows.Count} rows to be inserted");
                if (payload.Rows.Count == 0)
                {
                    return;
                }
                if (truncateOnLoad)
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandText = string.Format("truncate table {0}", tableName);
                        command.ExecuteNonQueryWithMeasurements(logname: "SqlServerBulkInsertBatchData");
                    }
                }
                if (allowTableLock)
                {
                    using (var bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.TableLock | SqlBulkCopyOptions.UseInternalTransaction, null))
                    {
                        bulk.BulkCopyTimeout = commandTimeout;
                        bulk.DestinationTableName = tableName;
                        bulk.BatchSize = batchsize;
                        bulk.SqlRowsCopied += new SqlRowsCopiedEventHandler(OnSqlRowsCopied);
                        bulk.NotifyAfter = batchsize;
                        if (stream)
                        {
                            bulk.EnableStreaming = true;
                        }
                        bulk.WriteToServer(payload);
                    }
                }
                else
                {
                    using (var bulk = new SqlBulkCopy(conn))
                    {
                        bulk.BulkCopyTimeout = commandTimeout;
                        bulk.DestinationTableName = tableName;
                        bulk.BatchSize = batchsize;
                        bulk.SqlRowsCopied += new SqlRowsCopiedEventHandler(OnSqlRowsCopied);
                        bulk.NotifyAfter = batchsize;
                        if (stream)
                        {
                            bulk.EnableStreaming = true;
                        }
                        bulk.WriteToServer(payload);
                    }
                }


                Console.WriteLine($"Completed Bulk Copy Batch - {payload.Rows.Count} rows inserted");
            }
            catch (Exception)
            {
                throw;
            }
        }

        public static void ExecuteStoredProcedure(SqlConnection conn, string procedureName, int timeout = 0)
        {
            using (var command = conn.GetCommand())
            {
                command.CommandTimeout = timeout;
                command.CommandType = CommandType.StoredProcedure;
                command.CommandText = procedureName;
                command.ExecuteNonQueryWithMeasurements(logname: "SqlServerExecuteStoredProcedure");
            }
        }

        private static void OnSqlRowsCopied(object sender, SqlRowsCopiedEventArgs e)
        {
            Console.WriteLine($"Copied {e.RowsCopied} so far...");
        }
    }

}
