﻿using Amazon.CloudWatch;
using Microsoft.AspNetCore.Http;
using Microsoft.Win32.SafeHandles;
using MySql.Data.MySqlClient;
using Npgsql;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StatsDHelpers;
using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Diagnostics;
using System.Runtime.InteropServices;
using System.Text;

namespace Resonance.Core.Helpers.DatabaseHelpers
{
    public class DataReaderWithMeasurements : IDisposable
    {
        bool disposed = false;
        bool? success = null;
        SafeHandle handle = new SafeFileHandle(IntPtr.Zero, true);
        private Stopwatch stopwatch = new Stopwatch();
        private string logname { get; set; } = null;
        public MySqlDataReader MysqlReader { get; set; } = null;
        public NpgsqlDataReader NpgReader { get; set; } = null;
        public SqlDataReader SqlReader { get; set; } = null;
        private HttpContext context { get; set; } = null;

        public DataReaderWithMeasurements(MySqlCommand command, HttpContext context, string statsdname, bool prepare = false, bool echoQuery = true, HttpContext _context = null)
        {
            try
            {
                if (echoQuery)
                {
                    context = _context;
                    string echo = command.CommandText;
                    if (Constants.AppConfig.Application.Environment == "Production")
                    {
                        Log.Verbose(echo);
                    }
                    if (command.Parameters != null && command.Parameters.Count > 0)
                    {
                        foreach (MySqlParameter param in command.Parameters)
                        {
                            if (Constants.AppConfig.Application.Environment != "Production")
                            {
                                echo = echo.Replace(param.ParameterName, $"'{param.Value}'");
                            }
                            else
                            {
                                Log.Verbose($@"{param.ParameterName}:{param.Value}:{param.MySqlDbType}");
                            }
                        }
                    }
                    if (Constants.AppConfig.Application.Environment != "Production")
                    {
                        Log.Verbose(echo);
                    }
                }

                logname = statsdname;
                if (!command.IsPrepared && prepare)
                {
                    command.Prepare();
                }
                stopwatch.Start();

                MysqlReader = command.ExecuteReader();

                success = true;
            }
            catch (Exception)
            {
                success = false;
                throw;
            }
            finally
            {
                CloudwatchHelper.EnqueueMetricRequest($"query_reader", stopwatch.ElapsedMilliseconds, context: context, unit: StandardUnit.Milliseconds);
            }
        }

        public DataReaderWithMeasurements(NpgsqlCommand command, HttpContext context, string statsdname, bool prepare = false, bool echoQuery = true)
        {
            try
            {
                if (echoQuery)
                {
                    string echo = command.CommandText;
                    if (Constants.AppConfig.Application.Environment == "Production")
                    {
                        Log.Verbose(echo);
                    }
                    if (command.Parameters != null && command.Parameters.Count > 0)
                    {
                        foreach (NpgsqlParameter param in command.Parameters)
                        {
                            if (Constants.AppConfig.Application.Environment != "Production")
                            {
                                echo = echo.Replace(param.ParameterName, $"'{param.Value}'");
                            }
                            else
                            {
                                Log.Verbose($@"{param.ParameterName}:{param.Value}:{param.NpgsqlDbType}");
                            }
                        }
                    }
                    if (Constants.AppConfig.Application.Environment != "Production")
                    {
                        Log.Verbose(echo);
                    }
                }

                logname = statsdname;
                if (!command.IsPrepared && prepare)
                {
                    command.Prepare();
                }

                stopwatch.Start();
                NpgReader = command.ExecuteReader();

                success = true;
            }
            catch (Exception)
            {
                success = false;
                throw;
            }
            finally
            {
                CloudwatchHelper.EnqueueMetricRequest($"query_reader", stopwatch.ElapsedMilliseconds, context: context, unit: StandardUnit.Milliseconds);
            }
        }

        /*
        public DataReaderWithMeasurements(SqlCommand command, string statsdname)
        {
            try
            {
                logname = statsdname;
                stopwatch.Start();
                SqlReader = command.ExecuteReader();

                success = true;
            }
            catch (Exception)
            {
                success = false;
                throw;
            }
        }
        */

        public void Dispose()
        {
            stopwatch.Stop();
            var print =
                success == null
                ? "Query Not Attempted: "
                : success == true
                    ? "Query Success: "
                    : "Query Failure: ";
            var querysource = MysqlReader != null
                ? "mysql"
                : NpgReader != null
                    ? "redshift"
                    : SqlReader != null
                        ? "sqlserver"
                        : "unknown";
            CloudwatchHelper.EnqueueMetricRequest($"query", stopwatch.ElapsedMilliseconds, context: context, unit: StandardUnit.Milliseconds);
            NLog.LogManager.GetCurrentClassLogger().Debug($"{logname ?? "Unnamed"}:{print}MeasureExecuteReader took {stopwatch.ElapsedMilliseconds}ms");
            if (!string.IsNullOrWhiteSpace(logname))
            {
                StatsDHelper.CounterLong(measurement: $"query", measurementType: logname, val: stopwatch.ElapsedMilliseconds, location: "DataReaderWithMeasurements", additionalTags: $",db_type={querysource}");
            }
            stopwatch.Reset();
            Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (MysqlReader != null)
            {
                MysqlReader.Dispose();
            }

            if (NpgReader != null)
            {
                NpgReader.Dispose();
            }

            if (SqlReader != null)
            {
                SqlReader.Dispose();
            }

            if (disposed)
            {
                return;
            }

            if (disposing)
            {
                handle.Dispose();
            }

            disposed = true;
        }
    }
}