﻿using Amazon.CloudWatch;
using Microsoft.AspNetCore.Http;
using MySql.Data.MySqlClient;
using Newtonsoft.Json;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.StatsDHelpers;
using Resonance.Core.Models.DatabaseModels;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Threading.Tasks;

namespace Resonance.Core.Helpers.DatabaseHelpers
{
    public static class DBManagerMysql
    {
        private static MySqlModel resonanceMySql = null;

        static DBManagerMysql()
        {

        }

        public static void Initialize(string db)
        {
            InitializeMysqlCredentials(db);
        }

        private static void InitializeMysqlCredentials(string db)
        {
            try
            {
                Log.Verbose($@"Fetching Mysql Credentials");
                resonanceMySql = JsonConvert.DeserializeObject<MySqlModel>(S3Helper.ReadTextFromS3("resonance-configuration", $"{Constants.AppConfig.Application.Environment}/credentials/mysql/{db}.json.gz", true, null));

                if (Constants.AppConfig.Application.Environment == "Development" || Constants.AppConfig.Application.Environment == "Staging")
                {
                    Log.Verbose($@"({Constants.AppConfig.Application.Environment}) Testing Mysql Connection To CrsData");
                    using (var conn = GetConnection(true))
                    {
                        Log.Verbose($@"({Constants.AppConfig.Application.Environment}) Testing Get Command To Mysql");
                        using (var command = conn.GetCommand())
                        {
                            Log.Verbose($@"({Constants.AppConfig.Application.Environment}) Executing Test Command To Mysql");
                            command.CommandText = @"select 1 as test";
                            command.ExecuteNonQueryWithMeasurements(logname: "MysqlInit");
                            Log.Verbose($@"Successfully connected to Mysql.");
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                throw;
            }
        }

        private static MySqlConnectionStringBuilder GetDefaultConnection()
        {
            return new MySqlConnectionStringBuilder()
            {
                Server = resonanceMySql.Server,
                Port = resonanceMySql.Port,
                UserID = resonanceMySql.UserName,
                Password = resonanceMySql.Password,
                Database = Constants.AppConfig.Application.Environment.ToLower(),
                Pooling = resonanceMySql.Pooling,
                SslMode = MySqlSslMode.Required,
                SqlServerMode = true,
                UseCompression = true,
                ConvertZeroDateTime = true,
                IgnorePrepare = false
            };
        }

        public static MySqlConnection GetConnection(bool open = true, MySqlConnectionStringBuilder alternate = null)
        {
            try
            {
                MySqlConnectionStringBuilder builder = alternate ?? GetDefaultConnection();
                var conn = new MySqlConnection(builder.ConnectionString);
                if (open)
                {
                    conn.Open();
                }
                return conn;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                throw;
            }
        }

        public static async Task<MySqlConnection> GetConnectionAsync(bool open = true, MySqlConnectionStringBuilder alternate = null)
        {
            try
            {
                MySqlConnectionStringBuilder builder = alternate ?? GetDefaultConnection();
                var conn = new MySqlConnection(builder.ConnectionString);
                if (open)
                {
                    await conn.OpenAsync();
                }
                return conn;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                throw;
            }
        }

        public static MySqlCommand GetCommand(this MySqlConnection conn, string commandText = "")
        {
            try
            {
                var command = new MySqlCommand()
                {
                    Connection = conn,
                    CommandText = commandText
                };
                return command;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                throw;
            }
        }

        public static IEnumerable<dynamic> GetDynamicSqlData(HttpContext context, string sql, string statsdname, int timeout = 60, Dictionary<string, dynamic> parameters = null, bool echoQuery = true)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();
            using (var conn = DBManagerMysql.GetConnection(true))
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandTimeout = timeout;
                    command.CommandText = sql;
                    if (parameters != null && parameters.Count > 0)
                    {
                        foreach (var param in parameters)
                        {
                            command.Parameters.AddWithValue(param.Key, param.Value);
                        }
                    }
                    if (!command.IsPrepared)
                    {
                        command.Prepare();
                    }
                    CloudwatchHelper.EnqueueMetricRequest($"query", stopwatch.ElapsedMilliseconds, context: context, unit: StandardUnit.Milliseconds);
                    using (var wrappedreader = new DataReaderWithMeasurements(command, context: context, statsdname: statsdname, echoQuery: echoQuery, _context: context))
                    {
                        var reader = wrappedreader.MysqlReader;
                        while (reader.Read())
                        {
                            yield return DataReaderHelper.SqlDataReaderToExpando(reader);
                        }
                    }
                }
            }
            stopwatch.Stop();
            Log.Debug($"GetDynamicSqlData took {stopwatch.ElapsedMilliseconds}ms");
        }

        public static IEnumerable<T> GetSqlData<T>(MySqlCommand command, HttpContext context, bool prepare = false, bool echoQuery = true, string logname = null)
        {
            var properties = typeof(T).GetProperties();

            var stopwatch = new Stopwatch();
            stopwatch.Start();

            using (var conn = DBManagerMysql.GetConnection(true))
            {
                command.Connection = conn;

                try
                {
                    if (!command.IsPrepared && prepare)
                    {
                        command.Prepare();
                    }

                    using (var wrappedreader = new DataReaderWithMeasurements(command, context: context, statsdname: logname ?? "wrapped_reader", echoQuery: echoQuery))
                    {
                        var reader = wrappedreader.MysqlReader;
                        while (reader.Read())
                        {
                            yield return DataReaderHelper.SqlDataReaderToExpando(reader);
                        }
                    }

                    conn.Close();
                }
                finally
                {
                    if (command != null)
                    {
                        command.Dispose();
                    }
                }

            }
            stopwatch.Stop();
            Log.Debug($"GetDynamicSqlData took {stopwatch.ElapsedMilliseconds}ms");
        }

        public static long? ImportToMysqlFromS3(MySqlCommand primaryQuery, MySqlCommand postfixQuery = null, int timeout = 86400, MySqlCommand prefixQuery = null, MySqlCommand resultQuery = null, bool echoQuery = true)
        {
            var stopwatch = new Stopwatch();
            stopwatch.Start();
            long? records = null;
            try
            {
                using (var conn = DBManagerMysql.GetConnection())
                {
                    if (conn.State != System.Data.ConnectionState.Open)
                    {
                        conn.Open();
                    }
                    if (prefixQuery != null)
                    {
                        try
                        {
                            prefixQuery.CommandTimeout = timeout;
                            prefixQuery.Connection = conn;
                            prefixQuery.ExecuteNonQueryWithMeasurements("prefix");
                        }
                        catch (Exception ex)
                        {
                            Log.Error(ex);
                        }
                        finally
                        {
                            if (prefixQuery != null)
                            {
                                prefixQuery.Dispose();
                            }
                        }
                    }
                    try
                    {
                        primaryQuery.CommandTimeout = timeout;
                        primaryQuery.Connection = conn;
                        primaryQuery.ExecuteNonQueryWithMeasurements("primary");
                    }
                    catch (Exception ex)
                    {
                        Log.Error(ex);
                    }
                    finally
                    {
                        if (primaryQuery != null)
                        {
                            primaryQuery.Dispose();
                        }
                    }
                    if (postfixQuery != null)
                    {
                        try
                        {
                            postfixQuery.CommandTimeout = timeout;
                            postfixQuery.Connection = conn;
                            postfixQuery.ExecuteNonQueryWithMeasurements("postfix", false);
                        }
                        catch (Exception ex)
                        {
                            Log.Error(ex);
                        }
                        finally
                        {
                            if (postfixQuery != null)
                            {
                                postfixQuery.Dispose();
                            }
                        }
                    }
                    if (resultQuery != null)
                    {
                        try
                        {
                            Log.Verbose($"Executing Result Query {resultQuery.CommandText}");
                            resultQuery.CommandTimeout = timeout;
                            resultQuery.Connection = conn;
                            records = (long?)resultQuery.ExecuteScalar();
                            Log.Verbose($"Result Query Count {records ?? 0}");
                        }
                        catch (Exception ex)
                        {
                            Log.Error(ex);
                        }
                        finally
                        {
                            if (resultQuery != null)
                            {
                                resultQuery.Dispose();
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
            }
            stopwatch.Start();
            Log.Debug($@"ImportToMysqlFromS3 took {stopwatch.ElapsedMilliseconds}ms");
            return records;
        }

        public static int ExecuteNonQueryWithMeasurements(this MySqlCommand command, string logname, bool prepare = false, bool echoQuery = true, HttpContext context = null)
        {
            try
            {
                Log.Verbose($"Executing: {command.CommandText}");
                // Mysql does not support preparing commands for load data queries.
                bool shouldPrepare = !command.CommandText.Contains("load data");

                if (shouldPrepare && prepare)
                {
                    if (!command.IsPrepared)
                    {
                        try
                        {
                            Log.Verbose($"Preparing Command");
                            command.Prepare();
                        }
                        catch (Exception ex)
                        {
                            Log.Error($"Sql Exception While Preparing: {command.CommandText}");
                            Log.Error(ex);
                            throw;
                        }
                    }
                    if (command.IsPrepared)
                    {
                        Log.Verbose($"Command successfully prepared");
                    }
                    else
                    {
                        Log.Verbose("Command failed to prepare");
                    }
                }

                if (command.Parameters != null && command.Parameters.Count > 0)
                {
                    var builder = new StringBuilder();
                    foreach (MySqlParameter parameter in command.Parameters)
                    {
                        var key = $@"{parameter.ParameterName}:{parameter.Value ?? DBNull.Value}:{parameter.MySqlDbType}";
                        builder.AppendLine(key);
                    }
                    if (builder.Length > 0)
                    {
                        Log.Verbose(builder.ToString());
                    }
                }
                var stopwatch = new Stopwatch();
                stopwatch.Start();
                int ret = command.ExecuteNonQuery();
                stopwatch.Stop();
                Log.Verbose($"{logname ?? "Unnamed"}:MeasureExecuteNonQuery took {stopwatch.ElapsedMilliseconds}ms");
                CloudwatchHelper.EnqueueMetricRequest($"query", stopwatch.ElapsedMilliseconds, context: context, unit: StandardUnit.Milliseconds);
                if (!string.IsNullOrWhiteSpace(logname))
                {
                    StatsDHelper.CounterLong(measurement: $"query", measurementType: logname, val: stopwatch.ElapsedMilliseconds, location: "ExecuteNonQueryWithMeasurements", additionalTags: $",db_type=mysql");
                }
                return ret;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                throw;
            }
        }

        public static T ExecuteScalarWithMeasurements<T>(this MySqlCommand command, string logname, bool echoQuery = true, HttpContext context = null)
        {
            try
            {
                if (echoQuery)
                {
                    string echo = command.CommandText;
                    if (Constants.AppConfig.Application.Environment == "Production")
                    {
                        Log.Verbose(echo);
                    }
                    if (command.Parameters != null && command.Parameters.Count > 0)
                    {
                        foreach (MySqlParameter p in command.Parameters)
                        {
                            if (Constants.AppConfig.Application.Environment != "Production")
                            {
                                echo = echo.Replace(p.ParameterName, $"'{p.Value}'");
                            }
                            else
                            {
                                Log.Verbose($@"{p.ParameterName}:{p.Value}:{p.MySqlDbType}");
                            }
                        }
                    }
                    if (Constants.AppConfig.Application.Environment != "Production")
                    {
                        Log.Verbose(echo);
                    }
                }

                var stopwatch = new Stopwatch();
                stopwatch.Start();
                var result = (T)command.ExecuteScalar();
                stopwatch.Stop();
                Log.Debug($"{logname ?? "Unnamed"}:MeasureExecuteScalar took {stopwatch.ElapsedMilliseconds}ms");
                CloudwatchHelper.EnqueueMetricRequest($"query", stopwatch.ElapsedMilliseconds, context: context, unit: StandardUnit.Milliseconds);
                if (!string.IsNullOrWhiteSpace(logname))
                {
                    StatsDHelper.CounterLong(measurement: $"query", measurementType: logname, val: stopwatch.ElapsedMilliseconds, location: "ExecuteScalarWithMeasurements", additionalTags: $",db_type=mysql");
                }
                return result;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                throw;
            }
        }

        public static string UnloadToS3
        (
            string bucket, string keyfolderpath, string sql,
            int timeout = 86400, string region = "-us-west-2",
            MySqlConnectionStringBuilder alternateConnection = null,
            bool explicitUnencrypted = false, Dictionary<string, dynamic> tokens = null
        )
        {
            try
            {
                if (!keyfolderpath.EndsWith("/"))
                {
                    keyfolderpath = $@"{keyfolderpath}/";
                }

                using (var conn = DBManagerMysql.GetConnection(true, alternateConnection))
                {
                    using (var command = conn.GetCommand())
                    {
                        var encloseChar = "\"";
                        var wrapper =
                        $@"
                            {sql}
                            into outfile s3 's3{region}://{bucket}/{keyfolderpath}data_'
                            fields terminated by ','
                            enclosed by '{encloseChar}'
                            manifest on
                            overwrite on
                            ;
                        ";
                        command.CommandTimeout = timeout;
                        command.CommandText = wrapper;
                        if (tokens != null && tokens.Count > 0)
                        {
                            foreach (var token in tokens)
                            {
                                command.Parameters.AddWithValue(token.Key, token.Value);
                            }
                        }
                        command.ExecuteNonQueryWithMeasurements(logname: "RedshiftUnload");
                        conn.Close();
                    }
                }
                return $@"{keyfolderpath}data_manifest";
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
                Log.Error($"Exception: {ex}, Sql: {sql}");
                return null;
            }
        }

        public static MySqlCommand LoadFileFromS3(string bucket, string keypath, string table, string quotechar = "\"", string ignore = "", string fields = "", string lineterm = "\\r\\n", bool allowAutomaticLineTerminationChange = true)
        {
            Log.Verbose($@"Loading {bucket}/{keypath} to {table} {fields}");
            string manifest = "";
            if (keypath.EndsWith("manifest"))
            {
                Log.Info($"LoadFileFromS3 detected manifest file for {bucket}/{keypath}. Flagging as a manifest data load.");
                manifest = "manifest";
                if (allowAutomaticLineTerminationChange)
                {
                    Log.Info($"Changing to \\n line terminator");
                    lineterm = "\\n";
                }
            }
            MySqlCommand command = new MySqlCommand()
            {
                CommandText =
                $@"
                    load data from s3 {manifest} 's3://{bucket}/{keypath}'
                    replace
                    into table {table}
                    character set 'utf8'
                    fields terminated by ','
                    enclosed by '{quotechar}'
                    lines terminated by '{lineterm}'
                    {ignore}
                    {fields}
                    ;
                "
            };
            return command;
        }
    }
}