﻿using System;
using Npgsql;
using Newtonsoft.Json;
using Resonance.Core.Models.DatabaseModels;
using Resonance.Core.Helpers.LoggingHelpers;
using Resonance.Core.Helpers.AwsHelpers;
using System.Collections.Generic;
using Resonance.Core.Models.DatabaseModels.RedshiftModels;
using Resonance.Core.Models.ConfigurationModels.Aws;
using System.Diagnostics;
using System.Dynamic;
using Resonance.Core.Helpers.StatsDHelpers;

namespace Resonance.Core.Helpers.DatabaseHelpers
{
    public static class DBManagerRedshift
    {
        private static UnloadOptionsModel unloadOptions = new UnloadOptionsModel();
        private static NpgSqlModel resonanceRedshift = null;
        private static NpgSqlModel tahoeRedshift = null;

        public static void Initialize()
        {
            InitializeRedshiftCredentials();
        }

        private static void InitializeRedshiftCredentials()
        {
            try
            {
                Log.Verbose($@"Fetching Redshift Credentials");
                resonanceRedshift = JsonConvert.DeserializeObject<NpgSqlModel>(S3Helper.ReadTextFromS3("resonance-configuration", $"{Constants.AppConfig.Application.Environment}/credentials/redshift/resonance.json.gz", true, null));
                tahoeRedshift = JsonConvert.DeserializeObject<NpgSqlModel>(S3Helper.ReadTextFromS3("resonance-configuration", $"{Constants.AppConfig.Application.Environment}/credentials/redshift/tahoe.json.gz", true, null));

                if (Constants.AppConfig.Application.Environment == "Development" || Constants.AppConfig.Application.Environment == "Staging")
                {
                    Log.Verbose($@"({Constants.AppConfig.Application.Environment}) Testing Redshift Connection To CrsData");
                    using (var conn = RedshiftConnection(true))
                    {
                        Log.Verbose($@"({Constants.AppConfig.Application.Environment}) Testing Get Command To CrsData");
                        using (var command = conn.GetCommand())
                        {
                            Log.Verbose($@"({Constants.AppConfig.Application.Environment}) Executing Test Command To CrsData");
                            command.CommandText = @"select 1 as test";
                            command.ExecuteNonQueryWithMeasurements(logname: "RedshiftInit");
                            Log.Verbose($@"Successfully connected to Redshift.");
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                throw;
            }
        }

        public static NpgsqlConnection RedshiftConnection(bool open = true, NpgsqlConnectionStringBuilder alternate = null)
        {
            try
            {
                NpgsqlConnectionStringBuilder builder = alternate ?? new NpgsqlConnectionStringBuilder()
                {
                    Host = resonanceRedshift.Host,
                    Port = resonanceRedshift.Port,
                    Username = resonanceRedshift.UserName,
                    Password = resonanceRedshift.Password,
                    Database = resonanceRedshift.Database,
                    Pooling = resonanceRedshift.Pooling,
                    SslMode = SslMode.Require,
                    ServerCompatibilityMode = ServerCompatibilityMode.Redshift,
                    TrustServerCertificate = true
                };
                var conn = new NpgsqlConnection(builder.ConnectionString);
                if (open)
                {
                    conn.Open();
                }
                return conn;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                throw;
            }
        }

        public static NpgsqlConnection TahoeConnection(bool open = true)
        {
            try
            {
                NpgsqlConnectionStringBuilder builder = new NpgsqlConnectionStringBuilder()
                {
                    Host = tahoeRedshift.Host,
                    Port = tahoeRedshift.Port,
                    Username = tahoeRedshift.UserName,
                    Password = tahoeRedshift.Password,
                    Database = tahoeRedshift.Database,
                    Pooling = tahoeRedshift.Pooling,
                    SslMode = SslMode.Require,
                    ServerCompatibilityMode = ServerCompatibilityMode.Redshift,
                    TrustServerCertificate = true
                };
                var conn = new NpgsqlConnection(builder.ConnectionString);
                if (open)
                {
                    conn.Open();
                }
                return conn;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                throw;
            }
        }

        public static NpgsqlConnectionStringBuilder GetDefaultConnectionString()
        {
            return new NpgsqlConnectionStringBuilder()
            {
                Host = resonanceRedshift.Host,
                Port = resonanceRedshift.Port,
                Username = resonanceRedshift.UserName,
                Password = resonanceRedshift.Password,
                Database = resonanceRedshift.Database,
                Pooling = resonanceRedshift.Pooling,
                SslMode = SslMode.Require,
                ServerCompatibilityMode = ServerCompatibilityMode.Redshift,
                TrustServerCertificate = true
            };
        }

        public static NpgsqlConnectionStringBuilder GetTahoeConnectionString()
        {
            return new NpgsqlConnectionStringBuilder()
            {
                Host = tahoeRedshift.Host,
                Port = tahoeRedshift.Port,
                Username = tahoeRedshift.UserName,
                Password = tahoeRedshift.Password,
                Database = tahoeRedshift.Database,
                Pooling = tahoeRedshift.Pooling,
                SslMode = SslMode.Require,
                ServerCompatibilityMode = ServerCompatibilityMode.Redshift,
                TrustServerCertificate = true
            };
        }

        /// <summary>
        /// Creates a sql command for the given sql connection
        /// </summary>
        public static NpgsqlCommand GetCommand(this NpgsqlConnection conn, string commandText = "")
        {
            try
            {
                var command = new NpgsqlCommand()
                {
                    Connection = conn,
                    CommandText = commandText,
                };
                return command;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                throw;
            }
        }

        public static void ImportToRedshift(string bucket, string path, string table, int ignoreHeader = 1, string dateFormat = "YYYYMMDD", string prefix = "", string suffix = "", int timeout = 0, string timeOrDate = "date", string compressionType = "", int maxerrors = 0)
        {
            try
            {
                using (var conn = DBManagerRedshift.RedshiftConnection())
                {
                    if (conn.State != System.Data.ConnectionState.Open)
                    {
                        conn.Open();
                    }
                    using (var command = conn.CreateCommand())
                    {
                        command.CommandTimeout = timeout;
                        command.CommandText = $@"{prefix}copy {table} 
                        from 's3://{bucket}/{path}'
                        credentials 'aws_iam_role=arn:aws:iam::029773783190:role/redshift-s3-access-role' region 'us-west-2'
                        delimiter as ','
                        CSV QUOTE as '""'
                        {timeOrDate.ToUpper()}FORMAT AS '{dateFormat}'
                        NULL as 'NULL'
                        maxerror as {maxerrors}
                        IGNOREHEADER as {ignoreHeader}
                        ACCEPTINVCHARS 
                        TRUNCATECOLUMNS
                        {compressionType};
                        Commit;{suffix}";
                        command.ExecuteNonQueryWithMeasurements("ImportToRedshift");
                    }
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                throw;
            }
        }

        public static IEnumerable<dynamic> GetDynamicSqlData(string sql, int timeout = 60, Dictionary<string, dynamic> parameters = null, bool useTahoe = false)
        {
            using
            (
                var conn = DBManagerRedshift.RedshiftConnection(
                    true,
                    alternate: useTahoe
                    ? DBManagerRedshift.GetTahoeConnectionString()
                    : DBManagerRedshift.GetDefaultConnectionString())
            )
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandTimeout = timeout;
                    command.CommandText = sql;
                    if (parameters != null && parameters.Count > 0)
                    {
                        foreach (var param in parameters)
                        {
                            command.Parameters.AddWithValue(param.Key, param.Value);
                        }
                    }
                    using (var wrappedreader = new DataReaderWithMeasurements(command, null, "redshift_get_dynamic_sql"))
                    {
                        var reader = wrappedreader.NpgReader;
                        while (reader.Read())
                        {
                            yield return DataReaderHelper.SqlDataReaderToExpando(reader);
                        }
                    }
                    conn.Close();
                }
            }
        }

        public static IEnumerable<dynamic> GetImmediateDynamicSqlData(string sql, int timeout = 60, Dictionary<string, dynamic> parameters = null)
        {
            List<ExpandoObject> results = new List<ExpandoObject>();

            try
            {
                using (var conn = DBManagerRedshift.RedshiftConnection(true))
                {
                    using (var command = conn.GetCommand())
                    {
                        command.CommandTimeout = timeout;
                        command.CommandText = sql;
                        if (parameters != null && parameters.Count > 0)
                        {
                            foreach (var param in parameters)
                            {
                                command.Parameters.AddWithValue(param.Key, param.Value);
                            }
                        }
                        using (var wrappedreader = new DataReaderWithMeasurements(command, null, "redshift_get_immediate_dynamic_sql_data"))
                        {
                            var reader = wrappedreader.NpgReader;
                            while (reader.Read())
                            {
                                results.Add(DataReaderHelper.SqlDataReaderToExpando(reader));
                            }
                        }
                        conn.Close();
                    }
                }
                return results;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                throw;
            }
        }

        public static IEnumerable<T> GetSqlData<T>(string sql, string prefixsql = "",  int timeout = 60, Dictionary<string, dynamic> parameters = null, bool useTahoe = false)
        {
            var properties = typeof(T).GetProperties();

            using
            (
                var conn = DBManagerRedshift.RedshiftConnection(
                    true,
                    alternate: useTahoe
                    ? DBManagerRedshift.GetTahoeConnectionString()
                    : DBManagerRedshift.GetDefaultConnectionString())
            )
            {
                using (var command = conn.GetCommand())
                {
                    command.CommandTimeout = timeout;
                    command.CommandText = sql;
                    if (parameters != null && parameters.Count > 0)
                    {
                        foreach (var param in parameters)
                        {
                            command.Parameters.AddWithValue(param.Key, param.Value);
                        }
                    }

                    using (var wrappedreader = new DataReaderWithMeasurements(command, null, "redshift_get_sql_data"))
                    {
                        var reader = wrappedreader.NpgReader;
                        while (reader.Read())
                        {
                            var element = Activator.CreateInstance<T>();

                            foreach (var f in properties)
                            {
                                var o = reader[f.Name];
                                if (o.GetType() != typeof(DBNull)) f.SetValue(element, o, null);
                            }
                            yield return element;
                        }
                    }
                    conn.Close();
                }
            }
        }

        public static string UnloadToS3
        (
            string bucket, string keyfolderpath, string wrappedsql, string kmsarn,
            string prefixsql = "", string postfixsql = "", int timeout = 86400,
            NpgsqlConnectionStringBuilder alternateConnection = null, UnloadOptionsModel options = null,
            bool explicitUnencrypted = false
        )
        {
            try
            {
                if(options == null)
                {
                    options = new UnloadOptionsModel();
                }
                if (!keyfolderpath.EndsWith("/"))
                {
                    keyfolderpath = $@"{keyfolderpath}/";
                }

                using (var conn = DBManagerRedshift.RedshiftConnection(true, alternateConnection))
                {
                    using (var command = conn.GetCommand())
                    {
                        if (options.AutomaticallyReplaceQuotes)
                        {
                            wrappedsql = wrappedsql.Replace("'", "''");
                        }
                        var encrypted = 
                        $@"
                            encrypted
                            kms_key_id '{kmsarn}'
                        ";
                        if (explicitUnencrypted)
                        {
                            encrypted = "";
                        }
                        var wrapper =
                        $@"
                            {prefixsql}
                            unload ('{wrappedsql}')
                            to 's3://{bucket}/{keyfolderpath}{options.FileNamePart}' 
                                iam_role 'arn:aws:iam::029773783190:role/redshift-s3-access-role' region 'us-west-2'
                                {encrypted}
                                {options.Header}
                                {options.Gzip}
                                {options.AddQuotes}
                                delimiter as '{options.Delimiter}'
                                {options.Escape}
                                parallel {options.Parallel}     
                                {options.Manifest} 
                                {options.AllowOverwrite}
                                {options.NullString}
                                maxfilesize {options.MaxFileSizeInMB} mb
                            ;
                            {postfixsql}
                        ";
                        command.CommandTimeout = timeout;
                        command.CommandText = wrapper;
                        command.ExecuteNonQueryWithMeasurements(logname: "RedshiftUnload");
                        conn.Close();
                    }
                }
                return $@"{keyfolderpath}{options.FileNamePart}manifest";
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
                Log.Error($"Exception: {ex}, Prefix Sql: {prefixsql}, Wrapped Sql: {wrappedsql}, Postfix sql: {postfixsql}");
                return null;
            }
        }

        public static void ExecuteNonQueryWithMeasurements(this NpgsqlCommand command, string logname = null, bool echoQuery = true)
        {
            try
            {
                if (echoQuery)
                {
                    string echo = command.CommandText;
                    if (Constants.AppConfig.Application.Environment == "Production")
                    {
                        Log.Verbose(echo);
                    }
                    if (command.Parameters != null && command.Parameters.Count > 0)
                    {
                        foreach (NpgsqlParameter p in command.Parameters)
                        {
                            if (Constants.AppConfig.Application.Environment != "Production")
                            {
                                echo = echo.Replace(p.ParameterName, $"'{p.Value}'");
                            }
                            else
                            {
                                Log.Verbose($@"{p.ParameterName}:{p.Value}:{p.NpgsqlDbType}");
                            }
                        }
                    }
                    if (Constants.AppConfig.Application.Environment != "Production")
                    {
                        Log.Verbose(echo);
                    }
                }

                var stopwatch = new Stopwatch();
                stopwatch.Start();
                command.ExecuteNonQuery();
                stopwatch.Stop();
                Log.Debug($"{logname ?? "Unnamed"}:MeasureExecuteNonQuery took {stopwatch.ElapsedMilliseconds}ms");
                if (!string.IsNullOrWhiteSpace(logname))
                {
                    StatsDHelper.CounterLong(measurement: $"query", measurementType: logname, val: stopwatch.ElapsedMilliseconds, location: "ExecuteNonQueryWithMeasurements", additionalTags: $",db_type=redshift");
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                throw;
            }
        }

        public static T ExecuteScalarWithMeasurements<T>(this NpgsqlCommand command, string logname = null, bool echoQuery = true)
        {
            try
            {
                if (echoQuery)
                {
                    string echo = command.CommandText;
                    if (Constants.AppConfig.Application.Environment == "Production")
                    {
                        Log.Verbose(echo);
                    }
                    if (command.Parameters != null && command.Parameters.Count > 0)
                    {
                        foreach (NpgsqlParameter p in command.Parameters)
                        {
                            if (Constants.AppConfig.Application.Environment != "Production")
                            {
                                echo = echo.Replace(p.ParameterName, $"'{p.Value}'");
                            }
                            else
                            {
                                Log.Verbose($@"{p.ParameterName}:{p.Value}:{p.NpgsqlDbType}");
                            }
                        }
                    }
                    if (Constants.AppConfig.Application.Environment != "Production")
                    {
                        Log.Verbose(echo);
                    }
                }

                var stopwatch = new Stopwatch();
                stopwatch.Start();
                var result = (T)command.ExecuteScalar();
                stopwatch.Stop();
                Log.Debug($"{logname ?? "Unnamed"}:MeasureExecuteScalar took {stopwatch.ElapsedMilliseconds}ms");
                if (!string.IsNullOrWhiteSpace(logname))
                {
                    StatsDHelper.CounterLong(measurement: $"query", measurementType: logname, val: stopwatch.ElapsedMilliseconds, location: "ExecuteScalarWithMeasurements", additionalTags: $",db_type=redshift");
                }
                return result;
            }
            catch (Exception ex)
            {
                Log.Error(ex);
                throw;
            }
        }
    }
}
