﻿using Resonance.Core.Models.FilterModels;
using System;
using System.Collections.Generic;
using System.Reflection;
using System.Text;
using static Resonance.Core.Constants;

namespace Resonance.Core.Helpers.FilterHelpers
{
    public static class FilterHelperMysql
    {
        public static string[] ValidDataTypes = new string[4]
        {
            "boolean",
            "datetime",
            "numeric",
            "string"
        };

        public static FilterType[] DetermineValidFilters(Type type)
        {
            FilterType[] validFilters = null;
            if (type == null)
            {
                return null;
            }

            if(type.BaseType == typeof(Enum))
            {
                type = typeof(Int32);
            }

            if (type == typeof(DateTime) || type == typeof(DateTime?))
            {
                validFilters = DateFilters();
            }
            else if(type == typeof(String))
            {
                validFilters = StringFilters();
            }
            else if (type == typeof(Boolean) || type == typeof(Boolean?))
            {
                validFilters = BoolFilters();
            }
            else if 
            (
                type == typeof(Int32)
                || type == typeof(Int64)
                || type == typeof(float)
                || type == typeof(Double)
                || type == typeof(Decimal)

                || type == typeof(Int32?)
                || type == typeof(Int64?)
                || type == typeof(float?)
                || type == typeof(Double?)
                || type == typeof(Decimal?)
            )
            {
                validFilters = NumericFilters();
            }

            return validFilters;
        }

        private static FilterType[] StringFilters()
        {
            FilterType[] validFilters = new FilterType[7]
            {
                FilterType.Exact,
                FilterType.StartsWith,
                FilterType.Contains,
                FilterType.In,
                FilterType.Regex,
                FilterType.Json,
                FilterType.NotEqual
            };
            return validFilters;
        }

        public static FilterType[] NumericFilters()
        {
            FilterType[] validFilters = new FilterType[5]
            {
                FilterType.Exact,
                FilterType.LessThan,
                FilterType.GreaterThan,
                FilterType.Range,
                FilterType.In
            };
            return validFilters;
        }

        public static FilterType[] BoolFilters()
        {
            FilterType[] validFilters = new FilterType[1]
            {
                FilterType.Exact
            };
            return validFilters;
        }

        public static FilterType[] DateFilters()
        {
            FilterType[] validFilters = new FilterType[]
            {
                FilterType.Exact,
                FilterType.LessThan,
                FilterType.GreaterThan,
                FilterType.In,
                FilterType.Range
            };
            return validFilters;
        }

        public static QueryFilterResult ProcessFilterQuery(QueryFilter filter)
        {
            switch (filter.FilterType)
            {
                case FilterType.Exact:
                    {
                        var paramKey = $"@p{Guid.NewGuid().ToString().Replace("-", "")}";
                        var result = new QueryFilterResult()
                        {
                            SqlSnippet = $@"and ({filter.Key} = {paramKey})",
                            Parameters = new Dictionary<string, dynamic>()
                        {
                            { paramKey, filter.Value ?? DBNull.Value }
                        }
                        };
                        return result;
                    }
                case FilterType.StartsWith:
                    {
                        var paramKey = $"@p{Guid.NewGuid().ToString().Replace("-", "")}";
                        var op = filter.IsCaseSensitive
                            ? "like binary"
                            : "like";
                        var result = new QueryFilterResult()
                        {
                            SqlSnippet = $@"and ({filter.Key} {op} concat({paramKey}, '%'))",
                            Parameters = new Dictionary<string, dynamic>()
                        {
                            { paramKey, filter.Value ?? DBNull.Value }
                        }
                        };
                        return result;
                    }
                case FilterType.Contains:
                    {
                        var paramKey = $"@p{Guid.NewGuid().ToString().Replace("-", "")}";
                        var op = filter.IsCaseSensitive
                            ? "like binary"
                            : "like";
                        var result = new QueryFilterResult()
                        {
                            SqlSnippet = $@"and ({filter.Key} {op} concat('%', {paramKey}, '%'))",
                            Parameters = new Dictionary<string, dynamic>()
                        {
                            { paramKey, filter.Value ?? DBNull.Value }
                        }
                        };
                        return result;
                    }
                case FilterType.In:
                    {
                        var paramKeys = new List<string>();
                        var builder = new StringBuilder();
                        var result = new QueryFilterResult()
                        {
                            Parameters = new Dictionary<string, dynamic>()
                        };

                        builder.AppendLine($"and {filter.Key} in (");
                        foreach (var p in filter.ValueArray)
                        {
                            var key = $"@p{Guid.NewGuid().ToString().Replace("-", "")}";
                            paramKeys.Add(key);
                            result.Parameters.Add(key, p ?? DBNull.Value);
                        }
                        builder.AppendLine($"{string.Join(",", paramKeys)})");
                        result.SqlSnippet = builder.ToString();
                        return result;
                    }
                case FilterType.NotEqual:
                { 
                    var paramKey = $"@p{Guid.NewGuid().ToString().Replace("-", "")}";
                    var op = "!=";
                    var result = new QueryFilterResult()
                    {
                        SqlSnippet = $@"and ({filter.Key} {op} {paramKey})",
                        Parameters = new Dictionary<string, dynamic>()
                        {
                            { paramKey, filter.Value ?? DBNull.Value }
                        }
                    };
                    return result;
                }
                case FilterType.LessThan:
                {
                    var paramKey = $"@p{Guid.NewGuid().ToString().Replace("-", "")}";
                    var op = filter.IsInclusive
                        ? "<="
                        : "<";
                    var result = new QueryFilterResult()
                    {
                        SqlSnippet = $@"and ({filter.Key} {op} {paramKey})",
                        Parameters = new Dictionary<string, dynamic>()
                        {
                            { paramKey, filter.Value ?? DBNull.Value }
                        }
                    };
                    return result;
                }
                case FilterType.GreaterThan:
                {
                    var paramKey = $"@p{Guid.NewGuid().ToString().Replace("-", "")}";
                    var op = filter.IsInclusive
                        ? ">="
                        : ">";
                    var result = new QueryFilterResult()
                    {
                        SqlSnippet = $@"and ({filter.Key} {op} {paramKey})",
                        Parameters = new Dictionary<string, dynamic>()
                        {
                            { paramKey, filter.Value ?? DBNull.Value }
                        }
                    };
                    return result;
                }                
                case FilterType.Range:
                {
                    var paramKey1 = $"@p{Guid.NewGuid().ToString().Replace("-", "")}";
                    var paramKey2 = $"@p{Guid.NewGuid().ToString().Replace("-", "")}";
                    var op = filter.IsInclusive
                        ? "<="
                        : "<";
                    var result = new QueryFilterResult()
                    {
                        SqlSnippet = $@"and ({filter.Key} >= {paramKey1} and {filter.Key} {op} {paramKey2})",
                        Parameters = new Dictionary<string, dynamic>()
                        {
                            { paramKey1, filter.Value ?? DBNull.Value },
                            { paramKey2, filter.Value2 ?? DBNull.Value }
                        }
                    };
                    return result;
                }
                case FilterType.Regex:
                {
                    throw new NotImplementedException();
                }
                case FilterType.Json:
                {
                    throw new NotImplementedException();
                }
                default:
                {
                    return null;
                }
            }

            throw new NotImplementedException();
        }
    }
}
