﻿using Amazon.CloudWatch;
using Microsoft.AspNetCore.Http;
using Newtonsoft.Json;
using NLog;
using NLog.AWS.Logger;
using NLog.Config;
using Resonance.Core.Helpers.ApiHelpers;
using Resonance.Core.Helpers.AuthHelpers;
using Resonance.Core.Helpers.AwsHelpers;
using Resonance.Core.Helpers.FileHelpers;
using Resonance.Core.Helpers.StatsDHelpers;
using Resonance.Core.Models.MetaDataModels;
using System;
using System.Text;

namespace Resonance.Core.Helpers.LoggingHelpers
{
    public static class Log
    {
        private static NLog.Logger logger { get; set; }
        private static NLog.Logger metricLogger { get; set; }
        static Log()
        {
            Initialize();
        }

        public static void Initialize(string environment = null)
        {
            var env = environment ?? Constants.AppConfig?.Application?.Environment;
            logger = NLog.Web.NLogBuilder.ConfigureNLog(FileHelper.GetRelativePath($"nlog.{env}.config")).GetCurrentClassLogger();
            metricLogger = NLog.Web.NLogBuilder.ConfigureNLog(FileHelper.GetRelativePath($"nlog.{Constants.AppConfig.Application.Environment}.config")).GetLogger("MetricLogger");
        }

        /// <summary>
        /// Preserve for cross compatability with legacy logging
        /// </summary>
        public static void Verbose(string message, HttpContext context = null)
        {
            logger.Trace($@"{message}{ProcessHttpContext(context)}");
        }

        public static void Debug(string message, HttpContext context = null)
        {
            logger.Debug($@"{message}{ProcessHttpContext(context)}");
        }

        public static void Info(string message, HttpContext context = null)
        {
            logger.Info($@"{message}{ProcessHttpContext(context)}");
        }
        public static void Warn(string message, string location = null, HttpContext context = null)
        {
            logger.Warn($@"{message}{ProcessHttpContext(context)}");
            CloudwatchHelper.EnqueueMetricRequest("logging", 1, context, StandardUnit.Count);
            StatsDHelper.Counter(measurement: $"warning", measurementType: "LogWarn", val: 1, location: $"{location ?? "LogWarn"}", additionalTags: ProcessHttpContext(context));
        }

        public static void Error(string message, string location = null, HttpContext context = null)
        {
            logger.Error($@"{message}{ProcessHttpContext(context)}");
            CloudwatchHelper.EnqueueMetricRequest("logging", 1, context, StandardUnit.Count);
            StatsDHelper.Counter(measurement: $"error", measurementType: "LogError", val: 1, location: $"{location ?? "LogError"}", additionalTags: ProcessHttpContext(context));
        }

        public static void Error(Exception ex, string location = null, HttpContext context = null)
        {
            logger.Error($@"{ex}{ProcessHttpContext(context)}");
            CloudwatchHelper.EnqueueMetricRequest("logging", 1, context, StandardUnit.Count);
            StatsDHelper.Counter(measurement: $"error", measurementType: "LogError", val: 1, location: $"{location ?? "LogException"}", additionalTags: ProcessHttpContext(context));
        }

        public static void Fatal(string message, string location = null, HttpContext context = null)
        {
            logger.Fatal($@"{message}{ProcessHttpContext(context)}");
            CloudwatchHelper.EnqueueMetricRequest("logging", 1, context, StandardUnit.Count);
            StatsDHelper.Counter(measurement: $"fatal", measurementType: "LogFatal", val: 1, location: $"{location ?? "LogFatal"}", additionalTags: ProcessHttpContext(context));
        }

        public static void Fatal(Exception ex, string location = null, HttpContext context = null)
        {
            logger.Fatal($@"{ex}{ProcessHttpContext(context)}");
            CloudwatchHelper.EnqueueMetricRequest("logging", 1, context, StandardUnit.Count);
            StatsDHelper.Counter(measurement: $"fatal", measurementType: "LogFatal", val: 1, location: $"{location ?? "LogFatalException"}", additionalTags: ProcessHttpContext(context));
        }

        public static void Metric(string data, HttpContext context, bool isSimpleMetric = false)
        {
            if (!isSimpleMetric)
            {
                metricLogger.Trace(data);
            }
            else
            {
                UserAuthData tokenData = (UserAuthData)context?.Items[UserAuthDataContext.UserAuthDataKey];
                string user = tokenData?.UserID;

                string userAgent = string.Empty;
                string userIP = string.Empty;
                var trackingID = Guid.NewGuid().ToString("d");
                if (context != null)
                {
                    try
                    {
                        userIP = HeaderHelper.GetRequestIP(context, true);
                    }
                    catch (Exception ex)
                    {
                        userIP = "Unknown";
                        Log.Verbose($@"Unable to record user ip. {ex}");
                    }
                    try
                    {
                        userAgent = HeaderHelper.GetUserAgent(context);
                    }
                    catch (Exception ex)
                    {
                        userAgent = "Unknown";
                        Log.Verbose($@"Unable to record user agent. {ex}");
                    }
                }
                Log.Metric(JsonConvert.SerializeObject(new CloudwatchMetricModel()
                {
                    Namespace = CloudwatchHelper.GetNamespace(),
                    Key = data,
                    Value = 1,
                    HttpPath = context?.Request?.Path,
                    HttpMethod = context?.Request?.Method,
                    HttpStatus = context?.Response?.StatusCode == null ? null : context.Response.StatusCode.ToString(),
                    UserID = tokenData?.UserID,
                    AppVersion = Constants.ApplicationVersion,
                    Datestamp = DateTime.UtcNow.ToString("yyyy-MM-dd"),
                    Timestamp = DateTime.UtcNow,
                    Host = Environment.MachineName,
                    TrackingID = trackingID,
                    UserIP = userIP,
                    UserAgent = userAgent
                }), context);
            }
        }
        
        private static string ProcessHttpContext(HttpContext context)
        {
            if (context == null)
            {
                return string.Empty;
            }

            var builder = new StringBuilder();
            if (context.Items.ContainsKey("RequestPath"))
            {
                builder.Append($",request_path={context.Items["RequestPath"]}");
            }
            if (context.Items.ContainsKey("ResonanceVersion"))
            {
                builder.Append($",request_version={context.Items["ResonanceVersion"]}");
            }
            return builder.ToString();
        }
    }
}